package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetDetailPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetDetailQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QPmsProjectRoleDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetDetailDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetDetailDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 预算明细
 *
 * @author xxb
 * @date 2023-07-10
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetDetailRepo repo;
    private final QPmsBudgetDetailDO qdo = QPmsBudgetDetailDO.pmsBudgetDetailDO;

    private final QPmsProjectRoleDO roleQdo = QPmsProjectRoleDO.pmsProjectRoleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetDetailVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 预算id
                qdo.budgetId,
                // 资源类型(labour人工、material材料、device设备)
                qdo.sourceType,
                // 项目角色主键
                qdo.projectRoleId,
                // 角色单价
                qdo.unitPrice,
                // 平工期（天
                qdo.days,
                // 费用总预算
                qdo.totalMoney,
                // 使用占比=已使用预算金额/预算总金额
                qdo.proportion,
                // 已使用预算金额=已经报销完成的金额
                qdo.usedMoney,
                // 科目主键
                qdo.subjectId,
                // 科目父主键
                qdo.subjectParentId,
                // 角色名称
                roleQdo.roleName
        )).from(qdo).leftJoin(roleQdo).on(qdo.projectRoleId.eq(roleQdo.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDetailVO> getJpaQueryWhere(PmsBudgetDetailQuery query) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 预算id 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetId())) {
            list.add(qdo.budgetId.eq(query.getBudgetId()));
        }
        /** 资源类型(labour人工、material材料、device设备) 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceType())) {
            list.add(qdo.sourceType.eq(query.getSourceType()));
        }
        /** 项目角色主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectRoleId())) {
            list.add(qdo.projectRoleId.eq(query.getProjectRoleId()));
        }
        /** 角色单价 精确 */
        if (!ObjectUtils.isEmpty(query.getUnitPrice())) {
            list.add(qdo.unitPrice.eq(query.getUnitPrice()));
        }
        /** 平工期（天 精确 */
        if (!ObjectUtils.isEmpty(query.getDays())) {
            list.add(qdo.days.eq(query.getDays()));
        }
        /** 费用总预算 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalMoney())) {
            list.add(qdo.totalMoney.eq(query.getTotalMoney()));
        }
        /** 使用占比=已使用预算金额/预算总金额 精确 */
        if (!ObjectUtils.isEmpty(query.getProportion())) {
            list.add(qdo.proportion.eq(query.getProportion()));
        }
        /** 已使用预算金额=已经报销完成的金额 精确 */
        if (!ObjectUtils.isEmpty(query.getUsedMoney())) {
            list.add(qdo.usedMoney.eq(query.getUsedMoney()));
        }
        /** 科目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectId())) {
            list.add(qdo.subjectId.eq(query.getSubjectId()));
        }
        /** 科目父主键 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectParentId())) {
            list.add(qdo.subjectParentId.eq(query.getSubjectParentId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetDetailVO queryByKey(Long id) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<PmsBudgetDetailVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据预算主键查询
     *
     * @param budgetId 预算主键
     * @return 结果
     */
    public List<PmsBudgetDetailVO> queryByBudgetId(Long budgetId) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.budgetId.eq(budgetId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据预算主键查询
     *
     * @param budgetId 预算主键
     * @return 结果
     */
    public List<PmsBudgetDetailVO> queryByBudgetIds(List<Long> budgetId) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.budgetId.in(budgetId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据预算主键查询
     *
     * @param budgetId 预算主键
     * @return 结果
     */
    public PmsBudgetDetailVO queryByBudgetIdAndRole(Long budgetId,Long projectRoleId) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.budgetId.eq(budgetId));
        jpaQuery.where(qdo.projectRoleId.eq(projectRoleId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchOne();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetDetailVO> queryListDynamic(PmsBudgetDetailQuery query) {
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetDetailVO> queryPaging(PmsBudgetDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetDetailDO save(PmsBudgetDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetDetailDO> saveAll(List<PmsBudgetDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 预算id
        if (payload.getBudgetId() != null) {
            update.set(qdo.budgetId, payload.getBudgetId());
        }
        // 资源类型(labour人工、material材料、device设备)
        if (payload.getSourceType() != null) {
            update.set(qdo.sourceType, payload.getSourceType());
        }
        // 项目角色主键
        if (payload.getProjectRoleId() != null) {
            update.set(qdo.projectRoleId, payload.getProjectRoleId());
        }
        // 角色单价
        if (payload.getUnitPrice() != null) {
            update.set(qdo.unitPrice, payload.getUnitPrice());
        }
        // 平工期（天
        if (payload.getDays() != null) {
            update.set(qdo.days, payload.getDays());
        }
        // 费用总预算
        if (payload.getTotalMoney() != null) {
            update.set(qdo.totalMoney, payload.getTotalMoney());
        }
        // 使用占比=已使用预算金额/预算总金额
        if (payload.getProportion() != null) {
            update.set(qdo.proportion, payload.getProportion());
        }
        // 已使用预算金额=已经报销完成的金额
        if (payload.getUsedMoney() != null) {
            update.set(qdo.usedMoney, payload.getUsedMoney());
        }
        // 科目主键
        if (payload.getSubjectId() != null) {
            update.set(qdo.subjectId, payload.getSubjectId());
        }
        // 科目父主键
        if (payload.getSubjectParentId() != null) {
            update.set(qdo.subjectParentId, payload.getSubjectParentId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 预算id
            if (nullFields.contains("budgetId")) {
                update.setNull(qdo.budgetId);
            }
            // 资源类型(labour人工、material材料、device设备)
            if (nullFields.contains("sourceType")) {
                update.setNull(qdo.sourceType);
            }
            // 项目角色主键
            if (nullFields.contains("projectRoleId")) {
                update.setNull(qdo.projectRoleId);
            }
            // 角色单价
            if (nullFields.contains("unitPrice")) {
                update.setNull(qdo.unitPrice);
            }
            // 平工期（天
            if (nullFields.contains("days")) {
                update.setNull(qdo.days);
            }
            // 费用总预算
            if (nullFields.contains("totalMoney")) {
                update.setNull(qdo.totalMoney);
            }
            // 使用占比=已使用预算金额/预算总金额
            if (nullFields.contains("proportion")) {
                update.setNull(qdo.proportion);
            }
            // 已使用预算金额=已经报销完成的金额
            if (nullFields.contains("usedMoney")) {
                update.setNull(qdo.usedMoney);
            }
            // 科目主键
            if (nullFields.contains("subjectId")) {
                update.setNull(qdo.subjectId);
            }
            // 科目父主键
            if (nullFields.contains("subjectParentId")) {
                update.setNull(qdo.subjectParentId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 根据预算主键删除
     *
     * @param budgetIds 预算主键
     * @return 结果
     */
    public long deleteByBudgetIds(List<Long> budgetIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.budgetId.in(budgetIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 更新金额
     * @param id
     * @param totalMoney
     * @param usedMoney
     * @param proportion
     * @return
     */
    public long updateMoney(Long id, BigDecimal totalMoney, BigDecimal usedMoney, BigDecimal proportion) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(id));
        update.set(qdo.totalMoney, totalMoney);
        update.set(qdo.usedMoney, usedMoney);
        update.set(qdo.proportion, proportion);
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

}

