package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectWbsQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsListVO;
import com.elitesland.tw.tw5.server.common.constants.ProjectWbsStatusEnum;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QProRelatedPartiesDO;
import com.elitesland.tw.tw5.server.prd.pms.budget.model.entity.QPmsWbsBudgetDetailsDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsResourceDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 活动清单-项目wbs表
 *
 * @author carl
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectActListDAO {

    private final JPAQueryFactory jpaQueryFactory;
    /**
     * wbs
     */
    private final QPmsProjectWbsDO qdo = QPmsProjectWbsDO.pmsProjectWbsDO;
    /**
     * 计划资源
     */
    private static final QPmsProjectWbsResourceDO qPmsProjectWbsResourceDO = QPmsProjectWbsResourceDO.pmsProjectWbsResourceDO;
    /**
     * 相关方
     */
    private static final QProRelatedPartiesDO qProRelatedPartiesDO = QProRelatedPartiesDO.proRelatedPartiesDO;
    /**
     * 活动预算
     */
    private static final QPmsWbsBudgetDetailsDO qPmsWbsBudgetDetailsDO = QPmsWbsBudgetDetailsDO.pmsWbsBudgetDetailsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsListVO> getJpaQuerySelect() {
        return jpaQueryFactory.selectDistinct(Projections.bean(PmsProjectWbsListVO.class,

                // 负责人
                qPmsProjectWbsResourceDO.relatePartiesId.as("managerRelatePartiesId"),
                qProRelatedPartiesDO.name.as("managerRelatePartiesName"),

                // 预算相关
                qPmsWbsBudgetDetailsDO.resCost,
                qPmsWbsBudgetDetailsDO.allocatedResAmt,
                qPmsWbsBudgetDetailsDO.remainingResAmt,
                qPmsWbsBudgetDetailsDO.occupiedResAmt,
                qPmsWbsBudgetDetailsDO.settledResAmt,


                // wbs相关
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 父主键
                qdo.parentId,
                // wbs描述（名称）
                qdo.wbsName,
                // wbs类型(WBS,NET,ACT,MS)
                qdo.wbsType,
                // wbs编码
                qdo.wbsCode,
                // node码
                qdo.nodeCode,
                // 父节点Wbs码
                qdo.parentWbsCode,
                qdo.wbsStatus,
                qdo.cancelShowFlag,
                qdo.managerUserId,
                // 成本对象，勾选为1
                qdo.subjectDist,
                // 实际开始时间
                qdo.startDate,
                // 实际结束时间
                qdo.endDate,
                // 预计开始时间
                qdo.preStartDate,
                // 预计结束时间
                qdo.preEndDate,
                // 预计持续时间（天）
                qdo.preDurationDay,
                // 进度(%)
                qdo.wbsProgress,
                // 实际持续时间（天）
                qdo.durationDay,
                // 最早开始时间
                qdo.earlyStartDate,
                // 最早结束时间
                qdo.earlyEndDate,
                // 最晚开始时间
                qdo.lateStartDate,
                // 最晚结束日期
                qdo.lateEndDate,
                // 总浮时
                qdo.totalFloat,
                // 自由浮时
                qdo.freeFloat,
                // 延时
                qdo.delayLag,
                //关键节点
                qdo.isKeyNode,
                //自动排期表示
                qdo.autoScheduling,
                // 成本类型
                qdo.costType,
                // 开票属性，勾选为1
                qdo.invoiceAttr,
                // 成本计划，勾选为1
                qdo.costPlan,
                // wbs权重
                qdo.wbsWeight,
                // 权重设置方式
                qdo.manualSettingWeight,
                qdo.versionId,
                qdo.versionNo,
                qdo.effRelateId,
                qdo.releaseFlag
            )).from(qdo)
            .leftJoin(qPmsProjectWbsResourceDO)
            .on(qdo.id.eq(qPmsProjectWbsResourceDO.wbsId)
                .and(qPmsProjectWbsResourceDO.managerUserFlag.eq(true)))
            .leftJoin(qProRelatedPartiesDO)
            .on(qProRelatedPartiesDO.id.eq(qPmsProjectWbsResourceDO.relatePartiesId))
            .leftJoin(qPmsWbsBudgetDetailsDO).on(qdo.id.eq(qPmsWbsBudgetDetailsDO.wbsId).and(qPmsWbsBudgetDetailsDO.versionNo.eq(0)))
            ;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectWbsQuery query) {
        long total = jpaQueryFactory
            .select(qdo.countDistinct())
            .from(qdo)
            .leftJoin(qPmsProjectWbsResourceDO)
            .on(qdo.id.eq(qPmsProjectWbsResourceDO.wbsId)
                .and(qPmsProjectWbsResourceDO.managerUserFlag.eq(true)))
            .leftJoin(qProRelatedPartiesDO)
            .on(qProRelatedPartiesDO.id.eq(qPmsProjectWbsResourceDO.relatePartiesId))
            .leftJoin(qPmsWbsBudgetDetailsDO).on(qdo.id.eq(qPmsWbsBudgetDetailsDO.wbsId).and(qPmsWbsBudgetDetailsDO.versionNo.eq(0)))
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsListVO> getJpaQueryWhere(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsListVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectWbsQuery query) {
        List<Predicate> list = new ArrayList<>();

        // 负责人（相关方）
        if (!ObjectUtils.isEmpty(query.getManagerRelatePartiesId())) {
            list.add(qPmsProjectWbsResourceDO.relatePartiesId.eq(query.getManagerRelatePartiesId()));
        }
        // 负责人名称（相关方)
        if (!ObjectUtils.isEmpty(query.getManagerRelatePartiesName())) {
            list.add(qProRelatedPartiesDO.name.like(SqlUtil.toSqlLikeString(query.getManagerRelatePartiesName())));
        }

        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        if (!ObjectUtils.isEmpty(query.getIdList())) {
            list.add(qdo.id.in(query.getIdList()));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsSearch())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsSearch())).or(qdo.wbsCode.like(SqlUtil.toSqlLikeString(query.getWbsSearch()))));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsName())));
        }
        /** wbs类型(WBS,NET,ACT,MS) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsType())) {
            list.add(qdo.wbsType.eq(query.getWbsType()));
        }
        /** wbs编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsCode())) {
            list.add(qdo.wbsCode.like(SqlUtil.toSqlLikeString(query.getWbsCode())));
        }
        /** node码 精确 */
        if (!ObjectUtils.isEmpty(query.getNodeCode())) {
            list.add(qdo.nodeCode.eq(query.getNodeCode()));
        }
        /** 父节点Wbs码 精确 */
        if (!ObjectUtils.isEmpty(query.getParentWbsCode())) {
            list.add(qdo.parentWbsCode.eq(query.getParentWbsCode()));
        }
        if (!ObjectUtils.isEmpty(query.getParentWbsCodes())) {
            list.add(qdo.parentWbsCode.in(query.getParentWbsCodes()));
        }
        /** 成本计划，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getCostPlan())) {
            list.add(qdo.costPlan.eq(query.getCostPlan()));
        }
        /** 科目分配，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectDist())) {
            list.add(qdo.subjectDist.eq(query.getSubjectDist()));
        }
        /** 开票属性，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAttr())) {
            list.add(qdo.invoiceAttr.eq(query.getInvoiceAttr()));
        }
        /** 持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getDurationDay())) {
            list.add(qdo.durationDay.eq(query.getDurationDay()));
        }

        /** 成本类型 精确 */
        if (!ObjectUtils.isEmpty(query.getCostType())) {
            list.add(qdo.costType.eq(query.getCostType()));
        }
        /** 实际开始时间 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 实际结束时间 精确 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.eq(query.getEndDate()));
        }
        /** 预计开始时间 大于等于 */
        if (!ObjectUtils.isEmpty(query.getPreStartDate())) {
            list.add(qdo.preStartDate.goe(query.getPreStartDate()));
        }
        /** 预计结束时间 小于等于 */
        if (!ObjectUtils.isEmpty(query.getPreEndDate())) {
            list.add(qdo.preEndDate.loe(query.getPreEndDate()));
        }
        /** 预计持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getPreDurationDay())) {
            list.add(qdo.preDurationDay.eq(query.getPreDurationDay()));
        }
        /** 进度(%) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsProgress())) {
            list.add(qdo.wbsProgress.eq(query.getWbsProgress()));
        }

        /** 负责人列表 精确 */
        if (!ObjectUtils.isEmpty(query.getManagerUserIds())) {
            list.add(qdo.managerUserId.in(query.getManagerUserIds()));
        }
        /** wbs状态 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsStatus())) {
            list.add(qdo.wbsStatus.eq(query.getWbsStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getWbsStatusPro())) {
            BooleanExpression expression = qdo.wbsStatus.eq(query.getWbsStatusPro());
            if (query.getWbsStatusPro().equals(ProjectWbsStatusEnum.NOGIONG.getCode())) {
                expression = expression.or(qdo.cancelShowFlag.eq(true));
            }
            list.add(expression);
        }

        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getState())) {
            list.add(qdo.state.eq(query.getState()));
        }
        /** 版本号 精确 */
        if (!ObjectUtils.isEmpty(query.getVersionNo())) {
            list.add(qdo.versionNo.eq(query.getVersionNo()));
        }
        /** 版本id 精确 */
        if (!ObjectUtils.isEmpty(query.getVersionId())) {
            list.add(qdo.versionId.eq(query.getVersionId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectWbsListVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectWbsListVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectWbsListVO> queryListDynamic(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsListVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改节点负责人
     *
     * @param projectId      主集合
     * @param managerUserIds 主集合
     * @return 删除的行数
     */
    public long updateManagerUserId(Long projectId, List<Long> managerUserIds, Long managerUserId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.managerUserId, managerUserId)
            .where(qdo.projectId.eq(projectId).and(qdo.managerUserId.in(managerUserIds)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新活动进度 状态
     *
     * @param id
     * @param wbsProgress
     * @return
     */
    public long updateWbsProgress(Long id, BigDecimal wbsProgress, String wbsStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.wbsProgress, wbsProgress).set(qdo.wbsStatus, wbsStatus)
            .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 更新活动进度 状态
     *
     * @param id
     * @param wbsProgress
     * @return
     */
    public long updateWbsProgress(Long id, BigDecimal wbsProgress) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.wbsProgress, wbsProgress)
            .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectWbsListVO> queryPage(PmsProjectWbsQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectWbsListVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectWbsListVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<PmsProjectWbsListVO>builder().records(result).total(total).build();
    }


    /**
     * 更新wbs状态
     *
     * @param id        主键
     * @param wbsStatus wbs状态
     * @return {@link Long}
     */
    public Long updateWbsStatus(Long id, String wbsStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.wbsStatus, wbsStatus)
            .where(qdo.id.eq(id));
        if (StringUtils.hasText(wbsStatus) && wbsStatus.equals(ProjectWbsStatusEnum.FINISHED.getCode())) {
            update.set(qdo.wbsProgress, new BigDecimal(100));
        }
        return update.execute();
    }

    public Long updateReleaseFlag(List<Long> idList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.releaseFlag, true)
            .set(qdo.state, "2")
            .where(qdo.id.in(idList));
        return update.execute();
    }

    public Long updateReleaseFlagByEffRelateId(List<Long> effRelateIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.releaseFlag, true)
            .set(qdo.state, "2")
            .where(qdo.effRelateId.in(effRelateIdList));
        return update.execute();
    }

    public Long updateWbsStatusByVersionIdAndEffRelateId(Long versionId, Long effRelateId, String wbsStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.wbsStatus, wbsStatus)
            .where(qdo.effRelateId.eq(effRelateId).and(qdo.versionId.eq(versionId)));
        return update.execute();
    }
}

