package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectWbsSnapshotQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsSnapshotVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectWbsSnapshotDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsSnapshotDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectWbsSnapshotRepo;

import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目wbs快照
 *
 * @author carl
 * @date 2023-04-17
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectWbsSnapshotDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectWbsSnapshotRepo repo;
    private final QPmsProjectWbsSnapshotDO qdo = QPmsProjectWbsSnapshotDO.pmsProjectWbsSnapshotDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsSnapshotVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectWbsSnapshotVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 快照（名称）
                qdo.snapName,
                // 快照内容
                qdo.snapContent
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsSnapshotVO> getJpaQueryWhere(PmsProjectWbsSnapshotQuery query) {
        JPAQuery<PmsProjectWbsSnapshotVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectWbsSnapshotQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectWbsSnapshotQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 快照（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getSnapName())) {
            list.add(qdo.snapName.like(SqlUtil.toSqlLikeString(query.getSnapName())));
        }
        /** 快照内容 模糊 */
        if (!ObjectUtils.isEmpty(query.getSnapContent())) {
            list.add(qdo.snapContent.like(SqlUtil.toSqlLikeString(query.getSnapContent())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectWbsSnapshotVO queryByKey(Long id) {
        JPAQuery<PmsProjectWbsSnapshotVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectWbsSnapshotVO> queryListDynamic(PmsProjectWbsSnapshotQuery query) {
        JPAQuery<PmsProjectWbsSnapshotVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectWbsSnapshotVO> queryPaging(PmsProjectWbsSnapshotQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectWbsSnapshotVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectWbsSnapshotVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectWbsSnapshotVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectWbsSnapshotDO save(PmsProjectWbsSnapshotDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectWbsSnapshotDO> saveAll(List<PmsProjectWbsSnapshotDO> dos) {
        return repo.saveAll(dos);
    }


    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

