package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsWbsTemplateNodePayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsWbsTemplateNodeQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsWbsTemplateNodeVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsWbsTemplateNodeDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsWbsTemplateNodeDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsWbsTemplateNodeRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * wbs模板节点
 *
 * @author Echo
 * @date 2023-03-29
 */
@Repository
@RequiredArgsConstructor
public class PmsWbsTemplateNodeDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsTemplateNodeRepo repo;
    private final QPmsWbsTemplateNodeDO qdo = QPmsWbsTemplateNodeDO.pmsWbsTemplateNodeDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTemplateNodeVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsTemplateNodeVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 模板主键
                qdo.templateId,
                // 节点描述（名称）
                qdo.wbsName,
                // 节点类型(WBS,NET,ACT,MS)
                qdo.wbsType,
                // 节点编码
                qdo.nodeCode,
                // wbs码
                qdo.wbsCode,
                // 父级wbs码
                qdo.parentWbsCode
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTemplateNodeVO> getJpaQueryWhere(PmsWbsTemplateNodeQuery query) {
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsWbsTemplateNodeQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsWbsTemplateNodeQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模板主键 精确 */
        if (!ObjectUtils.isEmpty(query.getTemplateId())) {
            list.add(qdo.templateId.eq(query.getTemplateId()));
        }
        /** 节点描述（名称） 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.eq(query.getWbsName()));
        }
        /** 节点类型(WBS,NET,ACT,MS) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsType())) {
            list.add(qdo.wbsType.eq(query.getWbsType()));
        }
        /** 节点编码 精确 */
        if (!ObjectUtils.isEmpty(query.getNodeCode())) {
            list.add(qdo.nodeCode.eq(query.getNodeCode()));
        }
        /** wbs码 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsCode())) {
            list.add(qdo.wbsCode.eq(query.getWbsCode()));
        }
        /** 父级wbs码 精确 */
        if (!ObjectUtils.isEmpty(query.getParentWbsCode())) {
            list.add(qdo.parentWbsCode.eq(query.getParentWbsCode()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsTemplateNodeVO queryByKey(Long id) {
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsTemplateNodeVO> queryListDynamic(PmsWbsTemplateNodeQuery query) {
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsWbsTemplateNodeVO> queryPaging(PmsWbsTemplateNodeQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsWbsTemplateNodeVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsWbsTemplateNodeVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsWbsTemplateNodeDO save(PmsWbsTemplateNodeDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsTemplateNodeDO> saveAll(List<PmsWbsTemplateNodeDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsWbsTemplateNodePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模板主键
        if (payload.getTemplateId() != null) {
            update.set(qdo.templateId, payload.getTemplateId());
        }
        // 节点描述（名称）
        if (payload.getWbsName() != null) {
            update.set(qdo.wbsName, payload.getWbsName());
        }
        // 节点类型(WBS,NET,ACT,MS)
        if (payload.getWbsType() != null) {
            update.set(qdo.wbsType, payload.getWbsType());
        }
        // 节点编码
        if (payload.getNodeCode() != null) {
            update.set(qdo.nodeCode, payload.getNodeCode());
        }
        // wbs码
        if (payload.getWbsCode() != null) {
            update.set(qdo.wbsCode, payload.getWbsCode());
        }
        // 父级wbs码
        if (payload.getParentWbsCode() != null) {
            update.set(qdo.parentWbsCode, payload.getParentWbsCode());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模板主键
            if (nullFields.contains("templateId")) {
                update.setNull(qdo.templateId);
            }
            // 节点描述（名称）
            if (nullFields.contains("wbsName")) {
                update.setNull(qdo.wbsName);
            }
            // 节点类型(WBS,NET,ACT,MS)
            if (nullFields.contains("wbsType")) {
                update.setNull(qdo.wbsType);
            }
            // 节点编码
            if (nullFields.contains("nodeCode")) {
                update.setNull(qdo.nodeCode);
            }
            // wbs码
            if (nullFields.contains("wbsCode")) {
                update.setNull(qdo.wbsCode);
            }
            // 父级wbs码
            if (nullFields.contains("parentWbsCode")) {
                update.setNull(qdo.parentWbsCode);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 通过主表ids删除
     *
     * @param templateIds 模板id（主表id）
     */
    public long deleteSoftByTemplateIds(List<Long> templateIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag,1)
                .where(qdo.templateId.in(templateIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update,qdo._super);
        return update.execute();
    }

    /**
     * 通过wbs模板主键查询
     *
     * @param templateIds 模板ids
     * @return 结果
     */
    public List<PmsWbsTemplateNodeVO> queryByTemplateIds(List<Long> templateIds) {
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.templateId.in(templateIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public long deleteSoftByTemplateId(Long templateId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag,1)
                .where(qdo.templateId.eq(templateId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update,qdo._super);
        return update.execute();
    }

    public boolean queryByName(PmsWbsTemplateNodePayload payload){
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuerySelect = getJpaQuerySelect();
        JPAQuery<PmsWbsTemplateNodeVO> jpaQuery = jpaQuerySelect.where(qdo.wbsName.eq(payload.getWbsName()));
        PmsWbsTemplateNodeVO pmsWbsTemplateNodeVO = jpaQuery.fetchOne();
        return pmsWbsTemplateNodeVO != null;
    }
}

