package com.elitesland.tw.tw5.server.prd.pms.deliver.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cas.service.PrdCasSsoService;
import com.elitesland.tw.tw5.api.prd.personplan.service.ProRelatedPartiesService;
import com.elitesland.tw.tw5.api.prd.personplan.vo.ProRelatedPartiesVO;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectWbsAcceptQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectWbsAcceptService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsAcceptVO;
import com.elitesland.tw.tw5.api.yeedocpro.vo.ItemInfoDataFileDTO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.pms.deliver.constants.DeliverItemStatisticsStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.deliver.constants.DeliverItemStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.deliver.convert.PmsDeliverItemConvert;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.entity.PmsDeliverItemDO;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.payload.PmsDeliverItemPayload;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.query.PmsDeliverItemQuery;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.vo.PmsDeliverItemStatisticsVO;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.vo.PmsDeliverItemVO;
import com.elitesland.tw.tw5.server.prd.pms.deliver.model.vo.PmsIframeDocInfoVO;
import com.elitesland.tw.tw5.server.prd.pms.deliver.repo.PmsDeliverItemRepo;
import com.elitesland.tw.tw5.server.prd.pms.deliver.repo.dao.PmsDeliverItemDao;
import com.elitesland.tw.tw5.server.prd.pms.deliver.service.PmsDeliverItemService;
import com.elitesland.tw.tw5.server.yeedocpro.config.YeedocProProperties;
import com.elitesland.tw.tw5.server.yeedocpro.dto.*;
import com.elitesland.tw.tw5.server.yeedocpro.service.YeedocProService;
import com.elitesland.tw.tw5.server.yeedocref.constants.YeedocRefTypeEnum;
import com.elitesland.tw.tw5.server.yeedocref.model.payload.PrdYeedocRefPayload;
import com.elitesland.tw.tw5.server.yeedocref.model.vo.PrdYeedocRefVO;
import com.elitesland.tw.tw5.server.yeedocref.service.PrdYeedocRefService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : duwh
 * @date : 2024-2-27
 * @desc : 交付项Service
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class PmsDeliverItemServiceImpl implements PmsDeliverItemService {


    private final PmsDeliverItemDao pmsDeliverItemDao;
    private final YeedocProProperties yeedocProProperties;
    private final PmsDeliverItemRepo pmsDeliverItemRepo;
    private final ProRelatedPartiesService proRelatedPartiesService;
    private final YeedocProService yeedocProService;
    private final PrdCasSsoService prdCasSsoService;
    @Resource
    @Lazy
    private PmsProjectService pmsProjectService;
    @Resource
    @Lazy
    private PmsProjectWbsAcceptService pmsProjectWbsAcceptService;
    @Resource
    private PrdYeedocRefService prdYeedocRefService;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsDeliverItemVO save(PmsDeliverItemPayload pmsDeliverItemPayload) {
        checkData(pmsDeliverItemPayload);
        PmsDeliverItemDO pmsDeliverItemDO = PmsDeliverItemConvert.INSTANCE.p2d(pmsDeliverItemPayload);
        PmsDeliverItemDO res = pmsDeliverItemRepo.save(pmsDeliverItemDO);
        return PmsDeliverItemConvert.INSTANCE.d2v(res);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsDeliverItemVO updateAll(PmsDeliverItemPayload pmsDeliverItemPayload) {
        Assert.notNull(pmsDeliverItemPayload.getId(), "id is null");
        PmsDeliverItemVO res = save(pmsDeliverItemPayload);
        return res;
    }


    @Override
    public PmsDeliverItemVO get(Long id) {
        if (null == id) {
            return null;
        }
        PmsDeliverItemVO res = pmsDeliverItemDao.get(id);
        Assert.notNull(res, "交付项不存在,id=" + id);
        translateOne(res);
        return res;
    }

    private void translateOne(PmsDeliverItemVO res) {
        String itemId = res.getItemId();
        if (StringUtils.hasText(itemId)) {
            Set<String> itemIdList = new HashSet<>();
            itemIdList.add(itemId);
            // 翻译附件
            ItemInfoPayload itemInfoPayload = new ItemInfoPayload();

            itemInfoPayload.setItemIds(itemIdList);
            Map<String, List<ItemInfoDataFileDTO>> stringListMap = yeedocProService.itemInfoReturn(itemInfoPayload);
            List<PmsDeliverItemPayload> updateStatusList = new ArrayList<>();
            res.setStatus(DeliverItemStatusEnum.NO.getCode());
            List<ItemInfoDataFileDTO> itemInfoDataFileDTOS = stringListMap.get(itemId);
            PmsDeliverItemPayload update = new PmsDeliverItemPayload();
            update.setId(res.getId());
            if (!CollectionUtil.isEmpty(itemInfoDataFileDTOS)) {
                res.setStatus(DeliverItemStatusEnum.OK.getCode());
            }
            update.setStatus(res.getStatus());
            updateStatusList.add(update);
            res.setFileList(itemInfoDataFileDTOS);
        }
    }


    @Override
    @Transactional
    public PagingVO<PmsDeliverItemVO> page(PmsDeliverItemQuery pmsDeliverItemQuery) {
        String submitContent = pmsDeliverItemQuery.getFileList();
        // 提交内容
        if (StringUtils.hasText(submitContent)) {
            PrdYeedocRefVO prdYeedocRefVO = prdYeedocRefService.queryByRefTypeAndRefId(YeedocRefTypeEnum.PMS_PROJECT, pmsDeliverItemQuery.getProjectId());
            if (prdYeedocRefVO == null) {
                return PagingVO.empty();
            }
            FolderListByItemNamePayload payload = new FolderListByItemNamePayload();
            payload.setLeafName(submitContent);
            payload.setLibraryId(prdYeedocRefVO.getLibraryId());
            payload.setFolderId(prdYeedocRefVO.getItemId());
            List<FolderListByItemNameDTO> folderListByItemNameDTOS = yeedocProService.folderListByItemName(payload);
            if (!CollectionUtil.isEmpty(folderListByItemNameDTOS)) {
                Set<String> itemIdSet = folderListByItemNameDTOS.stream().map(item -> item.getFolderId()).collect(Collectors.toSet());
                pmsDeliverItemQuery.setItemIdSet(itemIdSet);
            } else {
                return PagingVO.empty();
            }
        }

        // 里程碑
        if (null != pmsDeliverItemQuery.getWbsId()) {
            PmsProjectWbsAcceptQuery pmsProjectWbsAcceptQuery = new PmsProjectWbsAcceptQuery();
            pmsProjectWbsAcceptQuery.setWbsId(pmsDeliverItemQuery.getWbsId());
            List<PmsProjectWbsAcceptVO> pmsProjectWbsAcceptVOS = pmsProjectWbsAcceptService.getList(pmsProjectWbsAcceptQuery);
            Set<Long> idList = pmsProjectWbsAcceptVOS.stream().filter(pmsProjectWbsAcceptVO ->
                    (null != pmsProjectWbsAcceptVO.getType()
                        && pmsProjectWbsAcceptVO.getType().equals(0))
                        && StringUtils.hasText(pmsProjectWbsAcceptVO.getContent()))
                .map(pmsProjectWbsAcceptVO -> Long.valueOf(pmsProjectWbsAcceptVO.getContent()))
                .collect(Collectors.toSet());
            if (CollectionUtil.isEmpty(idList)) {
                return PagingVO.empty();
            }
            pmsDeliverItemQuery.setIdCollection(idList);
        }

        PagingVO<PmsDeliverItemVO> res = pmsDeliverItemDao.page(pmsDeliverItemQuery);
        // 翻译
        List<PmsDeliverItemPayload> updateStatusList = translate(res.getRecords());
        updateStatusBatch(updateStatusList);
        return res;
    }

    @Override
    @Transactional
    public PagingVO<PmsDeliverItemVO> myPage(PmsDeliverItemQuery pmsDeliverItemQuery) {
        Long projectId = pmsDeliverItemQuery.getProjectId();
        Long loginUserId = GlobalUtil.getLoginUserId();
        ProRelatedPartiesVO proRelatedPartiesVO = proRelatedPartiesService.getListSimpleByProjectIdAndUserId(projectId, loginUserId);
        if (null == proRelatedPartiesVO) {
            return new PagingVO<>();
        }
        Long id = proRelatedPartiesVO.getId();
        pmsDeliverItemQuery.setDirectorRelatedPartiesId(id);
        PagingVO<PmsDeliverItemVO> page = page(pmsDeliverItemQuery);
        List<PmsDeliverItemVO> records = page.getRecords();

        // 翻译易稻壳附件
        List<PmsDeliverItemPayload> updateStatusList = translate(records);
        updateStatusBatch(updateStatusList);
        return page;
    }

    @NotNull
    private List<PmsDeliverItemPayload> translate(List<PmsDeliverItemVO> records) {
        if (CollectionUtil.isEmpty(records)) {
            return Collections.emptyList();
        }
        // 翻译里程碑
        List<String> idList = records.stream().map(item -> item.getId().toString()).collect(Collectors.toList());

        List<PmsProjectWbsAcceptVO> projectWbsAcceptVOS = pmsProjectWbsAcceptService.listByContentAndTypeEqualZero(idList);
        // projectWbsAcceptVOS 转 Map 按 content分组
        Map<String, List<PmsProjectWbsAcceptVO>> contentMap = projectWbsAcceptVOS.stream().collect(Collectors.groupingBy(PmsProjectWbsAcceptVO::getContent));
        if (!contentMap.isEmpty()) {
            records.forEach(pmsDeliverItemVO -> {
                Long id = pmsDeliverItemVO.getId();
                List<PmsProjectWbsAcceptVO> projectWbsAcceptVOS1 = contentMap.get(id + "");
                if (!CollectionUtil.isEmpty(projectWbsAcceptVOS1)) {
                    PmsProjectWbsAcceptVO pmsProjectWbsAcceptVO = projectWbsAcceptVOS1.get(0);
                    Long wbsId = pmsProjectWbsAcceptVO.getWbsId();
                    String wbsName = pmsProjectWbsAcceptVO.getWbsName();
                    pmsDeliverItemVO.setWbsId(wbsId);
                    pmsDeliverItemVO.setWbsName(wbsName);
                }
            });
        }

        // 翻译附件
        ItemInfoPayload itemInfoPayload = new ItemInfoPayload();
        Set<String> itemIdList = records.stream()
            .filter(pmsDeliverItemVO -> StringUtils.hasText(pmsDeliverItemVO.getItemId()))
            .map(PmsDeliverItemVO::getItemId)
            .collect(Collectors.toSet());
        itemInfoPayload.setItemIds(itemIdList);
        Map<String, List<ItemInfoDataFileDTO>> stringListMap = yeedocProService.itemInfoReturn(itemInfoPayload);
        List<PmsDeliverItemPayload> updateStatusList = new ArrayList<>();
        records.stream()
            .filter(pmsDeliverItemVO -> StringUtils.hasText(pmsDeliverItemVO.getItemId()))
            .forEach(pmsDeliverItemVO -> {
                pmsDeliverItemVO.setStatus(DeliverItemStatusEnum.NO.getCode());
                String itemId = pmsDeliverItemVO.getItemId();
                List<ItemInfoDataFileDTO> itemInfoDataFileDTOS = stringListMap.get(itemId);
                PmsDeliverItemPayload update = new PmsDeliverItemPayload();
                update.setId(pmsDeliverItemVO.getId());
                if (!CollectionUtil.isEmpty(itemInfoDataFileDTOS)) {
                    pmsDeliverItemVO.setStatus(DeliverItemStatusEnum.OK.getCode());
                }
                update.setStatus(pmsDeliverItemVO.getStatus());
                updateStatusList.add(update);
                pmsDeliverItemVO.setFileList(itemInfoDataFileDTOS);
            });
        return updateStatusList;
    }

    @Transactional
    @Async
    public void updateStatusBatch(List<PmsDeliverItemPayload> updateStatusList) {
        if (!CollectionUtil.isEmpty(updateStatusList)) {
            updateStatusList.forEach(pmsDeliverItemPayload -> {
                update(pmsDeliverItemPayload);
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long del(List<Long> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return 0L;
        }
        Long res = pmsDeliverItemDao.del(ids);
        return res;

    }

    @Override
    public List<PmsDeliverItemVO> getList(PmsDeliverItemQuery pmsDeliverItemQuery) {
        List<PmsDeliverItemVO> res = pmsDeliverItemDao.getList(pmsDeliverItemQuery);
        return res;
    }

    @Override
    public List<PmsDeliverItemVO> list(PmsDeliverItemQuery pmsDeliverItemQuery) {
        List<PmsDeliverItemVO> res = pmsDeliverItemDao.list(pmsDeliverItemQuery);
        return res;
    }

    /**
     * 数据校验
     *
     * @param pmsDeliverItemPayload
     */
    private void checkData(PmsDeliverItemPayload pmsDeliverItemPayload) {
//        Assert.notEmpty(pmsDeliverItemPayload.getType(), "类型不能为空");
//        Assert.notNull(pmsDeliverItemPayload.getUserId(), "用户id不能为空");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long update(PmsDeliverItemPayload pmsDeliverItemPayload) {
        Assert.notNull(pmsDeliverItemPayload.getId(), "id不能为空");
        Long res = pmsDeliverItemDao.update(pmsDeliverItemPayload);
        return res;
    }

    public Long count(PmsDeliverItemQuery pmsDeliverItemQuery) {
        Long res = pmsDeliverItemDao.count(pmsDeliverItemQuery);
        return res;
    }

    @Override
    public long delByProjectId(Long projectId) {
        return pmsDeliverItemDao.delByProjectId(projectId);
    }

    @Override
    public Long updateShowFlagByRoleId(Long roleId, boolean showFlag) {
        return pmsDeliverItemDao.updateShowFlagByRoleId(roleId, showFlag);
    }

    @Override
    @Transactional
    public PmsIframeDocInfoVO getIframeDocInfo(Long projectId, String appId, HttpServletRequest request, HttpServletResponse response) {
        Assert.notNull(projectId, "项目id不能为空");
        PrdYeedocRefVO prdYeedocRefVO = prdYeedocRefService.queryByRefTypeAndRefId(YeedocRefTypeEnum.PMS_PROJECT, projectId);
        String pmsLibraryId = yeedocProProperties.getPmsLibraryId();
        // 如果不存在的话，同步去易稻壳创建项目文件夹
        if (prdYeedocRefVO == null) {
            // 再次创建项目文件夹
            // 根据项目id查询易稻壳文件夹id
            PmsProjectVO pmsProjectVO = pmsProjectService.queryByKey(projectId);
            String fileName = pmsProjectVO.getProjName() + "-" + pmsProjectVO.getProjNo() + "";
            // String fileName = "测试项目后续删-" + projectId + "";
            CreateFolderTwPayload createFolderTwPayload = new CreateFolderTwPayload();
            createFolderTwPayload.setLibraryId(pmsLibraryId);
            createFolderTwPayload.setPathArry(Arrays.asList("/" + fileName));
            YeedocCreateFolderForTwDTO folderReturn = yeedocProService.createFolderForTw(createFolderTwPayload);

            PrdYeedocRefPayload save = new PrdYeedocRefPayload();
            save.setRefType(YeedocRefTypeEnum.PMS_PROJECT.getCode());
            save.setRefId(projectId);
            save.setLibraryId(pmsLibraryId);
            save.setItemId(folderReturn.getFolderId());
            save.setItemName(folderReturn.getLeafName());
            TwOutputUtil<PrdYeedocRefVO> insert = prdYeedocRefService.insert(save);
            if (insert.isOk()) {
                prdYeedocRefVO = insert.getData();
            }
        }
        PmsIframeDocInfoVO result = new PmsIframeDocInfoVO();
        result.setLibraryId(pmsLibraryId);
        result.setFolder(prdYeedocRefVO.getItemId());
        result.setRootFolder(prdYeedocRefVO.getItemId());
        result.setSource(yeedocProProperties.getSourceIframe());
        Map ticketAndUrl = prdCasSsoService.getTicketAndUrl(appId, request, response);
        if (null != ticketAndUrl) {
            Object objTicket = ticketAndUrl.get("ticket");
            result.setTicket(objTicket.toString());
        } else {
            log.error("ticketAndUrl is null");
        }
        return result;
    }

    /**
     * 数据源：来源于项目内交付清单和里程碑数据，统计到项目开始到目前日期的数据
     * 统计交付物完成情况，使用饼图根据状态区分为已完成和未完成
     * - 已完成
     * - 未完成中数据细分为
     * - 逾期（统计当前日期以前未完成的数据）
     * - 今日需完成（统计当前日期需要完成的数据）
     * - 剩余（统计后面将要完成的数据）
     *
     * @param query 查询
     * @return {@link List}<{@link PmsDeliverItemStatisticsVO}>
     */
    @Override
    public List<PmsDeliverItemStatisticsVO> statistics(PmsDeliverItemQuery query) {
        org.springframework.util.Assert.notNull(query.getProjectId(), "项目主键不能为空");
        List<PmsDeliverItemStatisticsVO> result = new ArrayList<>();

        long okCount = 0l;
        long overdueCount = 0l;
        long todayCount = 0l;
        long surplusCount = 0l;

        if (ObjectUtils.isEmpty(query.getShowFlag())) {
            query.setShowFlag(true);
        }
        List<PmsDeliverItemVO> pmsDeliverItemVOList = list(query);

        if (CollectionUtils.isNotEmpty(pmsDeliverItemVOList)) {
            LocalDate now = LocalDate.now();
            // 统计已完成的
            okCount = pmsDeliverItemVOList.stream().filter(pmsDeliverItemVO -> DeliverItemStatusEnum.OK.getCode().equals(pmsDeliverItemVO.getStatus())).count();
            //逾期（统计当前日期以前未完成的数据）
            // 统计当前日期以前未完成的数据
            overdueCount = pmsDeliverItemVOList.stream()
                .filter(pmsDeliverItemVO ->
                    DeliverItemStatusEnum.NO.getCode().equals(pmsDeliverItemVO.getStatus()) && pmsDeliverItemVO.getPreStartDate().isBefore(now))
                .count();

            // 今日需完成（统计当前日期需要完成的数据）
            todayCount = pmsDeliverItemVOList.stream()
                .filter(pmsDeliverItemVO ->
                    DeliverItemStatusEnum.NO.getCode().equals(pmsDeliverItemVO.getStatus()) && pmsDeliverItemVO.getPreStartDate().isEqual(now))
                .count();
            // 剩余（统计后面将要完成的数据）
            surplusCount = pmsDeliverItemVOList.stream()
                .filter(pmsDeliverItemVO ->
                    DeliverItemStatusEnum.NO.getCode().equals(pmsDeliverItemVO.getStatus()) && pmsDeliverItemVO.getPreStartDate().isAfter(now))
                .count();

            Long countSum = okCount + overdueCount + todayCount + surplusCount;

            result.add(new PmsDeliverItemStatisticsVO(DeliverItemStatisticsStatusEnum.OK.getCode(), DeliverItemStatisticsStatusEnum.OK.getDesc(), okCount, countSum));
            result.add(new PmsDeliverItemStatisticsVO(DeliverItemStatisticsStatusEnum.OVERDUE.getCode(), DeliverItemStatisticsStatusEnum.OVERDUE.getDesc(), overdueCount, countSum));
            result.add(new PmsDeliverItemStatisticsVO(DeliverItemStatisticsStatusEnum.TO_BE_COMPLETED_TODAY.getCode(), DeliverItemStatisticsStatusEnum.TO_BE_COMPLETED_TODAY.getDesc(), todayCount, countSum));
            result.add(new PmsDeliverItemStatisticsVO(DeliverItemStatisticsStatusEnum.SURPLUS.getCode(), DeliverItemStatisticsStatusEnum.SURPLUS.getDesc(), surplusCount, countSum));
        }


        return result;
    }

}

