package com.elitesland.tw.tw5.server.prd.pms.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetDetailPayload;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetDetailQuery;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetSubjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsBudgetDetailService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsBudgetDetailSubjectService;
import com.elitesland.tw.tw5.api.prd.pms.vo.*;
import com.elitesland.tw.tw5.api.prd.task.vo.PmsWbsTaskVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.BudgetStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.ProjectWbsTypeEnum;
import com.elitesland.tw.tw5.server.prd.pms.convert.PmsBudgetConvert;
import com.elitesland.tw.tw5.server.prd.pms.convert.PmsBudgetDetailConvert;
import com.elitesland.tw.tw5.server.prd.pms.dao.*;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetDetailDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetDetailRepo;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetRepo;
import com.elitesland.workflow.enums.ProcInstStatus;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 预算明细
 *
 * @author xxb
 * @date 2023-07-10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsBudgetDetailServiceImpl extends BaseServiceImpl implements PmsBudgetDetailService {

    private final PmsBudgetDetailRepo pmsBudgetDetailRepo;
    private final PmsBudgetDetailDAO pmsBudgetDetailDAO;

    private final PmsBudgetDetailSubjectService subjectService;

    private final PmsBudgetDAO pmsBudgetDAO;
    private final PmsBudgetRepo pmsBudgetRepo;

    private final PmsProjectDAO projectDAO;

    private final PmsBudgetSubjectDAO budgetSubjectDAO;

    private final CacheUtil cacheUtil;
    private final PmsProjectWbsDAO pmsProjectWbsDAO;


    @Override
    public PagingVO<PmsBudgetDetailVO> queryPaging(PmsBudgetDetailQuery query) {
        return pmsBudgetDetailDAO.queryPaging(query);
    }

    @Override
    public List<PmsBudgetDetailVO> queryListDynamic(PmsBudgetDetailQuery query) {
        return pmsBudgetDetailDAO.queryListDynamic(query);
    }

    @Override
    public PmsBudgetDetailVO queryByKey(Long key) {
        PmsBudgetDetailDO entity = pmsBudgetDetailRepo.findById(key).orElseGet(PmsBudgetDetailDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsBudgetDetailVO vo = PmsBudgetDetailConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void insert(PmsBudgetPayload budgetPayload) {
        //验证数据
        checkData(budgetPayload);
        // 获取预算明细
        List<PmsBudgetDetailPayload> detailPayloads = budgetPayload.getBudgetDetailList();
        PmsBudgetDetailPayload detailPayload = detailPayloads.get(0);
        BigDecimal totalMoney = detailPayload.getUnitPrice().multiply(detailPayload.getDays());
        detailPayload.setTotalMoney(totalMoney);

        // 保存预算
        Long budgetId = budgetPayload.getId();
        if (ObjectUtils.isEmpty(budgetId)) {
            //新增 预算基本信息
            budgetPayload.setTotalMoney(totalMoney);
            budgetPayload.setUsedMoney(BigDecimal.ZERO);
            budgetPayload.setProportion(BigDecimal.ZERO);
            budgetPayload.setBudgetStatus(BudgetStatusEnum.CREATE.getCode());
            PmsBudgetDO budgetDO = PmsBudgetConvert.INSTANCE.toDo(budgetPayload);
            PmsBudgetVO budgetVO = PmsBudgetConvert.INSTANCE.toVo(pmsBudgetRepo.save(budgetDO));
            budgetId = budgetVO.getId();

            // 保存预算明细
            detailPayload.setBudgetId(budgetId);
            insert(detailPayload);
        } else {
            // 逻辑判断
            PmsBudgetVO budgetVO = pmsBudgetDAO.queryByKey(budgetId);
            if (ObjectUtils.isEmpty(budgetVO)) {
                throw TwException.error("", "预算基本信息不存在，请核验！");
            }
            // 如果预算时激活状态 不让再编辑了
            if (BudgetStatusEnum.APPROVED.getCode().equals(budgetVO.getBudgetStatus())) {
                throw TwException.error("", "预算处于激活状态，不允许再编辑，请核验！");
            }

            // 新增或更新 预算明细
            if (ObjectUtils.isEmpty(detailPayload.getId())) {
                PmsBudgetDetailVO detailVO = pmsBudgetDetailDAO.queryByBudgetIdAndRole(budgetId, detailPayload.getProjectRoleId());
                if (!ObjectUtils.isEmpty(detailVO)) {
                    throw TwException.error("", "相同节点下，一个角色只能对应一个预算，请核验！");
                }
                detailPayload.setBudgetId(budgetId);
                insert(detailPayload);
            } else {
                PmsBudgetDetailVO detailVO = pmsBudgetDetailDAO.queryByKey(detailPayload.getId());
                if (!detailPayload.getProjectRoleId().equals(detailVO.getProjectRoleId())) {
                    detailVO = pmsBudgetDetailDAO.queryByBudgetIdAndRole(budgetId, detailPayload.getProjectRoleId());
                    if (!ObjectUtils.isEmpty(detailVO)) {
                        throw TwException.error("", "相同节点下，一个角色只能对应一个预算，请核验！");
                    }
                }
                update(detailPayload);
            }
            //更新 预算基本信息
            updateBudgetMoney(budgetId);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsBudgetDetailVO insert(PmsBudgetDetailPayload payload) {
        payload.setUsedMoney(BigDecimal.ZERO);
        payload.setProportion(BigDecimal.ZERO);
        PmsBudgetDetailDO entityDo = PmsBudgetDetailConvert.INSTANCE.toDo(payload);
        return PmsBudgetDetailConvert.INSTANCE.toVo(pmsBudgetDetailRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsBudgetDetailVO update(PmsBudgetDetailPayload payload) {
        PmsBudgetDetailDO entity = pmsBudgetDetailRepo.findById(payload.getId()).orElseGet(PmsBudgetDetailDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsBudgetDetailDO entityDo = PmsBudgetDetailConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return PmsBudgetDetailConvert.INSTANCE.toVo(pmsBudgetDetailRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUsedMoney(Long wbsId, List<PmsWbsTaskVO> list) {
        //判断 预算基本信息 是否存在
        PmsBudgetVO budgetVO = pmsBudgetDAO.queryByWBS(wbsId);
        if (ObjectUtils.isEmpty(budgetVO)) {
            throw TwException.error("", "预算基本信息不存在，请核验！");
        }
        if (!BudgetStatusEnum.APPROVED.getCode().equals(budgetVO.getBudgetStatus())) {
            throw TwException.error("", "只有预算处于激活状态才可以进行任务派发，请核验！");
        }
        Long budgetId = budgetVO.getId();
        for (PmsWbsTaskVO vo : list) {
            // 更新 预算明细
            PmsBudgetDetailVO detailVO = pmsBudgetDetailDAO.queryByBudgetIdAndRole(budgetId, vo.getProjectRoleId());
            if (ObjectUtils.isEmpty(detailVO)) {
                throw TwException.error("", "预算明细不存在，请核验！");
            }
            BigDecimal addUse = detailVO.getUnitPrice().multiply(vo.getDays());
            BigDecimal usedMoney = detailVO.getUsedMoney().add(addUse);
            if (usedMoney.doubleValue() > detailVO.getTotalMoney().doubleValue()) {
                throw TwException.error("", "超出预算啦，请核验！");
            }
            BigDecimal proportion = usedMoney.divide(detailVO.getTotalMoney(), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
            pmsBudgetDetailDAO.updateMoney(detailVO.getId(), detailVO.getTotalMoney(), usedMoney, proportion);
        }

        //更新 预算基本信息
        updateBudgetMoney(budgetId);

    }

    /**
     * 更新 预算基本信息 中金额
     *
     * @param budgetId
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateBudgetMoney(Long budgetId) {
        List<PmsBudgetDetailVO> detailVOS = pmsBudgetDetailDAO.queryByBudgetId(budgetId);
        BigDecimal totalMoney = detailVOS.stream().map(PmsBudgetDetailVO::getTotalMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal usedMoney = detailVOS.stream().map(PmsBudgetDetailVO::getUsedMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal proportion = BigDecimal.ZERO;
        if (totalMoney.doubleValue() > 0) {
            proportion = usedMoney.divide(totalMoney, 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
        }
        pmsBudgetDAO.updateMoney(budgetId, totalMoney, usedMoney, proportion);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(Long budgetId, List<Long> keys) {
        if (ObjectUtils.isEmpty(keys)) {
            throw TwException.error("", "预算明细主键不能为空，请核验！");
        }
        PmsBudgetVO budgetVO = pmsBudgetDAO.queryByKey(budgetId);
        if (ObjectUtils.isEmpty(budgetVO)) {
            throw TwException.error("", "预算基本信息不存在，请核验！");
        }
        if (BudgetStatusEnum.APPROVED.getCode().equals(budgetVO.getBudgetStatus())) {
            throw TwException.error("", "预算处于激活状态，不允许再编辑，请核验！");
        }

        // 删除预算明细
        List<PmsBudgetDetailVO> list = pmsBudgetDetailDAO.queryByKeys(keys);
        for (PmsBudgetDetailVO vo : list) {
            if (vo.getUsedMoney().doubleValue() > 0) {
                throw TwException.error("", " 已经使用过的预算明细不能再被编辑，请核验！");
            }
        }
        pmsBudgetDetailDAO.deleteSoft(keys);

        // 更新预算基本信息  总预算金额、使用占比
        updateBudgetMoney(budgetId);
    }

    @Override
    public PmsBudgetVO queryByWbs(Long projectId, Long wbsId) {
        if (ObjectUtils.isEmpty(projectId)) {
            throw TwException.error("", " projectId不能为空，请核验！");
        }
        PmsBudgetVO budgetVO = new PmsBudgetVO();
        String budgetStatus = null;
        List<PmsBudgetVO> list = pmsBudgetDAO.queryListByProjectId(projectId);
        if (!ObjectUtils.isEmpty(list)) {
            budgetStatus = list.get(0).getBudgetStatus();
        }

        if (!ObjectUtils.isEmpty(wbsId)) {
            // 判断当前节点是 活动 还是元素  如果是活动 直接根据节点查询  如果是元素，根据 元素下的子节点 查询
            PmsProjectWbsVO pmsProjectWbsVO = pmsProjectWbsDAO.queryByKey(wbsId);
            if (!ObjectUtils.isEmpty(pmsProjectWbsVO)) {
                if (ProjectWbsTypeEnum.ACT.getCode().equals(pmsProjectWbsVO.getWbsType())) {
                    budgetVO = pmsBudgetDAO.queryByWBS(wbsId);
                    if (!ObjectUtils.isEmpty(budgetVO)) {
                        List<PmsBudgetDetailVO> detailVOS = pmsBudgetDetailDAO.queryByBudgetId(budgetVO.getId());
                        transferData(detailVOS);
                        budgetVO.setBudgetDetailList(detailVOS);
                    }else {
                        budgetVO = new PmsBudgetVO();
                    }
                } else if (ProjectWbsTypeEnum.WBS.getCode().equals(pmsProjectWbsVO.getWbsType())) {
                    //WBS元素的预算信息为其下级节点的预算总和，WBS元素本身没有预算
                    list = new ArrayList<>();
                    //获取 该节点下的  所有预算信息
                    getBudgetData(projectId, list, Arrays.asList(pmsProjectWbsVO.getWbsCode()));

                    budgetVO.setTotalMoney(list.stream()
                            .map(PmsBudgetVO::getTotalMoney).reduce(BigDecimal.ZERO, BigDecimal::add));
                    budgetVO.setUsedMoney(list.stream()
                            .map(PmsBudgetVO::getUsedMoney).reduce(BigDecimal.ZERO, BigDecimal::add));
                    BigDecimal proportion = BigDecimal.ZERO;
                    if (budgetVO.getTotalMoney().doubleValue() > 0) {
                        proportion = budgetVO.getUsedMoney().divide(budgetVO.getTotalMoney(), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
                    }
                    budgetVO.setProportion(proportion);
                }
            }
        }
        budgetVO.setBudgetStatus(budgetStatus);
        return budgetVO;
    }

    /**
     * 获取 该节点下的  所有预算信息
     *
     * @param list
     * @param parentWbsCodes
     */
    private void getBudgetData(Long projectId, List<PmsBudgetVO> list, List<String> parentWbsCodes) {
        List<PmsProjectWbsVO> wbsList = pmsProjectWbsDAO.queryByParentWbsCodes(projectId, parentWbsCodes);
        if (!ObjectUtils.isEmpty(wbsList)) {
            List<Long> wbsIdList = wbsList.stream().map(v -> v.getId()).collect(Collectors.toList());
            List<PmsBudgetVO> budgetVOS = pmsBudgetDAO.queryByWbsList(wbsIdList);
            if (!ObjectUtils.isEmpty(budgetVOS)) {
                list.addAll(budgetVOS);
            }
            parentWbsCodes = wbsList.stream().map(v -> v.getWbsCode()).collect(Collectors.toList());
            getBudgetData(projectId, list, parentWbsCodes);
        }
    }

    @Override
    public String queryProjectBudgetStatus(Long projectId, Long wbsId) {
        if (ObjectUtils.isEmpty(projectId)) {
            throw TwException.error("", " projectId不能为空，请核验！");
        }
        PmsProjectWbsVO pmsProjectWbsVO = pmsProjectWbsDAO.queryByKey(wbsId);
        if (!ObjectUtils.isEmpty(pmsProjectWbsVO)) {
            // 如果此节点是 活动 ，判断它 下面有没有激活预算
            if (ProjectWbsTypeEnum.ACT.getCode().equals(pmsProjectWbsVO.getWbsType())) {
                PmsBudgetVO budgetVO = pmsBudgetDAO.queryByWBS(wbsId);
                if (!ObjectUtils.isEmpty(budgetVO)) {
                    return budgetVO.getBudgetStatus();
                }
            } else if (ProjectWbsTypeEnum.WBS.getCode().equals(pmsProjectWbsVO.getWbsType())) {
                //WBS元素的预算信息为其下级节点的预算总和，WBS元素本身没有预算
                List<PmsBudgetVO> budgetList = new ArrayList<>();
                //获取 该节点下的  所有预算信息
                getBudgetData(projectId, budgetList, Arrays.asList(pmsProjectWbsVO.getWbsCode()));
                if (!ObjectUtils.isEmpty(budgetList)) {
                    return budgetList.get(0).getBudgetStatus();
                }
            }
        }
        return null;
    }

    @Override
    public List<PmsBudgetVO> queryByWbsList(List<Long> wbsIds) {
        return pmsBudgetDAO.queryByWbsList(wbsIds);
    }

    @Override
    public PmsBudgetVO queryByProject(Long projectId) {
        //判断项目是否存在
        PmsProjectVO projectVO = projectDAO.queryByKey(projectId);
        if (ObjectUtils.isEmpty(projectVO)) {
            throw TwException.error("", "关联项目不存在，请核验！");
        }
        PmsBudgetVO budgetVO = new PmsBudgetVO();
        budgetVO.setBudgetControlFlag(projectVO.getBudgetControlFlag());
        budgetVO.setBudgetControlStrategyFlag(projectVO.getBudgetControlStrategyFlag());

        PmsBudgetDetailSubjectVO sv = new PmsBudgetDetailSubjectVO();
        sv.setSubjectLevel(1);
        sv.setSubjectCode("项目总计");

        // 获取预算中的科目
        List<PmsBudgetDetailSubjectVO> subjectList = subjectService.queryByProjectId(projectId);
        // 获取预算基本信息
        List<PmsBudgetVO> list = pmsBudgetDAO.queryListByProjectId(projectId);
        if (!ObjectUtils.isEmpty(list)) {
            List<Long> budgetIds = list.stream().map(v -> v.getId()).collect(Collectors.toList());
            // 获取预算明细
            List<PmsBudgetDetailVO> detailVOS = pmsBudgetDetailDAO.queryByBudgetIds(budgetIds);

            //求算每个科目的值
            for (PmsBudgetDetailSubjectVO subjectVO : subjectList) {
                if (1 == subjectVO.getSubjectLevel()) {
                    subjectVO.setTotalMoney(detailVOS.stream()
                            .filter(vo -> subjectVO.getSubjectId().equals(vo.getSubjectParentId()))
                            .map(PmsBudgetDetailVO::getTotalMoney).reduce(BigDecimal.ZERO, BigDecimal::add));
                } else {
                    subjectVO.setTotalMoney(detailVOS.stream()
                            .filter(vo -> !ObjectUtils.isEmpty(vo.getSubjectId()) && subjectVO.getSubjectId().equals(vo.getSubjectId()))
                            .map(PmsBudgetDetailVO::getTotalMoney).reduce(BigDecimal.ZERO, BigDecimal::add));
                }
            }
            budgetVO.setBudgetStatus(list.get(0).getBudgetStatus());
            sv.setTotalMoney(subjectList.stream().filter(vo -> 1 == vo.getSubjectLevel() && !ObjectUtils.isEmpty(vo.getTotalMoney())).map(PmsBudgetDetailSubjectVO::getTotalMoney).reduce(BigDecimal.ZERO, BigDecimal::add));
        }
        // 科目编号排序
        List<PmsBudgetDetailSubjectVO> subjectDetails = new LinkedList<>();
        subjectDetails.add(sv);
        subjectDetails.addAll(listByTree(subjectList));
        budgetVO.setTemplateSubjectList(subjectDetails);

        return budgetVO;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitApprovalProcess(Long projectId) {
        //判断项目是否存在
        PmsProjectVO projectVO = projectDAO.queryByKey(projectId);
        if (ObjectUtils.isEmpty(projectVO)) {
            throw TwException.error("", "关联项目不存在，请核验！");
        }
        // TODO  当前自动审批通过
        List<PmsBudgetVO> list = pmsBudgetDAO.queryListByProjectId(projectId);
        List<Long> keys = list.stream().map(v -> v.getId()).collect(Collectors.toList());
        pmsBudgetDAO.updateWorkFlow(keys, LocalDateTime.now(), "11", BudgetStatusEnum.APPROVED.getCode(), LocalDateTime.now(), ProcInstStatus.APPROVED);
    }

    @Override
    public void deleteSoftByWbsIds(List<Long> keys) {
        // 算出预算基本信息
        pmsBudgetDAO.deleteSoft(keys);
        // 删除预算明细
        pmsBudgetDetailDAO.deleteByBudgetIds(keys);
    }


    void checkData(PmsBudgetPayload budgetPayload) {
        // 判断 非空
        if (ObjectUtils.isEmpty(budgetPayload.getProjectId())) {
            throw TwException.error("", "关联项目ID不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(budgetPayload.getWbsId())) {
            throw TwException.error("", "wbsID不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(budgetPayload.getTemplateId())) {
            throw TwException.error("", "模板ID不能为空，请核验！");
        }
        List<PmsBudgetDetailPayload> detailPayloads = budgetPayload.getBudgetDetailList();
        if (ObjectUtils.isEmpty(detailPayloads)) {
            throw TwException.error("", "预算明细不能为空，请核验！");
        }

        PmsBudgetDetailPayload detailPayload = detailPayloads.get(0);
        if (ObjectUtils.isEmpty(detailPayload.getSourceType())) {
            throw TwException.error("", "资源类型不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(detailPayload.getProjectRoleId())) {
            throw TwException.error("", "角色不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(detailPayload.getUnitPrice())) {
            throw TwException.error("", "角色单价不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(detailPayload.getDays())) {
            throw TwException.error("", "工期不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(detailPayload.getSubjectParentId())) {
            throw TwException.error("", "一级科目不能为空，请核验！");
        }

        // 根据模板 去查询 此模板下的科目  为了判断 如果一级科目有子集科目，必须选二级科目
        if (ObjectUtils.isEmpty(detailPayload.getSubjectId())) {
            long num = subjectService.getCountByProjectAndParent(budgetPayload.getTemplateId(), detailPayload.getSubjectParentId());
            if (num > 0) {
                throw TwException.error("", "如果一级科目有子集科目，必须选二级科目，请核验！");
            }
        }
    }

    /**
     * 翻译数据
     *
     * @param detailVOS
     */
    void transferData(List<PmsBudgetDetailVO> detailVOS) {
        List<Long> subjectIds = new ArrayList<>();
        for (PmsBudgetDetailVO detailVO : detailVOS) {
            subjectIds.add(detailVO.getSubjectParentId());
            if (!ObjectUtils.isEmpty(detailVO.getSubjectId())) {
                subjectIds.add(detailVO.getSubjectId());
            }
        }
        PmsBudgetSubjectQuery query = new PmsBudgetSubjectQuery();
        query.setSubjectIds(subjectIds);
        List<PmsBudgetSubjectVO> subjectList = budgetSubjectDAO.queryListDynamic(query);
        Map<Long, String> nameMap = subjectList.stream().collect(Collectors.toMap(PmsBudgetSubjectVO::getId, PmsBudgetSubjectVO::getSubjectName));
        for (PmsBudgetDetailVO detailVO : detailVOS) {
            detailVO.setSubjectParentName(nameMap.get(detailVO.getSubjectParentId()));
            if (!ObjectUtils.isEmpty(detailVO.getSubjectId())) {
                detailVO.setSubjectName(nameMap.get(detailVO.getSubjectId()));
            }
            detailVO.setSourceTypeName(cacheUtil.transferSystemSelection("PMS:WBS:RESOURCE:TYPE", detailVO.getSourceType()));
        }
    }

    /**
     * 科目排序
     *
     * @param list
     * @return
     */
    private List<PmsBudgetDetailSubjectVO> listByTree(List<PmsBudgetDetailSubjectVO> list) {
        List<PmsBudgetDetailSubjectVO> resultList = new LinkedList<>();
        //找出 父级 并排序
        List<PmsBudgetDetailSubjectVO> parentList = list.stream()
                .filter(vo -> ObjectUtils.isEmpty(vo.getSubjectParentId())).collect(Collectors.toList());
        List<Long> parentIds = parentList.stream().map(PmsBudgetDetailSubjectVO::getSubjectId).collect(Collectors.toList());
        //找出 没有父科目的子科目  集合
        List<PmsBudgetDetailSubjectVO> noparentlist = list.stream()
                .filter(vo -> !ObjectUtils.isEmpty(vo.getSubjectParentId()) && !parentIds.contains(vo.getSubjectParentId())).collect(Collectors.toList());
        // 针对有 没有父科目的子科目  当做 父科目
        parentList.addAll(noparentlist);

        sortBySubjectCode(parentList);
        // 找出父子关系
        Map<Long, List<PmsBudgetDetailSubjectVO>> groupedAndSorted = list.stream()
                .filter(vo -> !ObjectUtils.isEmpty(vo.getSubjectParentId()))
                .collect(Collectors.groupingBy(
                        PmsBudgetDetailSubjectVO::getSubjectParentId,
                        Collectors.toList()
                ));
        // 展示父子关系
        for (PmsBudgetDetailSubjectVO vo : parentList) {
            resultList.add(vo);
            List<PmsBudgetDetailSubjectVO> subset = groupedAndSorted.get(vo.getSubjectId());
            if (!ObjectUtils.isEmpty(subset)) {
                groupedAndSorted.remove(vo.getSubjectId());
                // 子集排序
                sortBySubjectCode(subset);
                resultList.addAll(subset);
            }
        }
        return resultList;
    }

    /**
     * 根据科目编号排序
     *
     * @param list
     */
    void sortBySubjectCode(List<PmsBudgetDetailSubjectVO> list) {
        list.sort(new Comparator<PmsBudgetDetailSubjectVO>() {
            @Override
            public int compare(PmsBudgetDetailSubjectVO obj1, PmsBudgetDetailSubjectVO obj2) {
                String str1 = obj1.getSubjectCode();
                String str2 = obj2.getSubjectCode();

                // 将字符串转换为整数进行比较
                int int1 = Integer.parseInt(str1);
                int int2 = Integer.parseInt(str2);

                // 检查是否以0开头
                boolean isStr1StartingWithZero = str1.startsWith("0");
                boolean isStr2StartingWithZero = str2.startsWith("0");

                // 如果一个字符串以0开头而另一个不是，则将以0开头的字符串排在前面
                if (isStr1StartingWithZero && !isStr2StartingWithZero) {
                    return -1;
                } else if (!isStr1StartingWithZero && isStr2StartingWithZero) {
                    return 1;
                } else {
                    // 否则，使用整数值进行比较
                    return Integer.compare(int1, int2);
                }
            }
        });
    }

}
