package com.elitesland.tw.tw5.server.prd.pms.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetSubjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetSubjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsBudgetSubjectService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetSubjectVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetTemplateSubjectVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.pms.convert.PmsBudgetSubjectConvert;
import com.elitesland.tw.tw5.server.prd.pms.dao.PmsBudgetSubjectDAO;
import com.elitesland.tw.tw5.server.prd.pms.dao.PmsBudgetTemplateSubjectDAO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetTemplateSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetSubjectRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 预算科目
 *
 * @author xxb
 * @date 2023-04-23
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsBudgetSubjectServiceImpl extends BaseServiceImpl implements PmsBudgetSubjectService {

    private final PmsBudgetSubjectRepo pmsBudgetSubjectRepo;
    private final PmsBudgetSubjectDAO pmsBudgetSubjectDAO;

    private final PmsBudgetTemplateSubjectDAO pmsBudgetTemplateSubjectDAO;

    @Override
    public PagingVO<PmsBudgetSubjectVO> queryPaging(PmsBudgetSubjectQuery query) {
        return pmsBudgetSubjectDAO.queryPaging(query);
    }

    @Override
    public PagingVO<PmsBudgetSubjectVO> pagingByTree(PmsBudgetSubjectQuery query) {
        List<PmsBudgetSubjectVO> list = listByTree(query);
        List<PmsBudgetSubjectVO> result = list.stream()
                .skip(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .collect(Collectors.toList());
        return PagingVO.<PmsBudgetSubjectVO>builder().records(result).total(list.size()).build();
    }

    @Override
    public List<PmsBudgetSubjectVO> listByTree(PmsBudgetSubjectQuery query) {
        // 返回结果
        List<PmsBudgetSubjectVO> resultList = new LinkedList<>();
        List<PmsBudgetSubjectVO> list = pmsBudgetSubjectDAO.queryListDynamic(query);
        List<Long> subjectIds = query.getSubjectIds();
        if (!ObjectUtils.isEmpty(subjectIds)) {
            list = list.stream().filter(vo -> subjectIds.contains(vo.getId())).toList();
        }
        //找出 父级 并排序
        List<PmsBudgetSubjectVO> parentList = list.stream()
                .filter(vo -> ObjectUtils.isEmpty(vo.getParentId())).collect(Collectors.toList());
        List<Long> parentIds = parentList.stream().map(PmsBudgetSubjectVO::getId).collect(Collectors.toList());
        if (!ObjectUtils.isEmpty(query.getSubjectCode()) || !ObjectUtils.isEmpty(query.getSubjectName())) {
            //找出 没有父科目的子科目  集合
            List<PmsBudgetSubjectVO> noparentlist = list.stream()
                    .filter(vo -> !ObjectUtils.isEmpty(vo.getParentId()) && !parentIds.contains(vo.getParentId())).collect(Collectors.toList());
            // 针对有 没有父科目的子科目  当做 父科目
            parentList.addAll(noparentlist);
        }
        //  Collections.sort(parentList, Comparator.comparing((PmsBudgetSubjectVO vo) -> vo.getSubjectCode().startsWith("0") ? 0 : 1));
        sortBySubjectCode(parentList);
        // 找出父子关系
        Map<Long, List<PmsBudgetSubjectVO>> groupedAndSorted = list.stream()
                .filter(vo -> !ObjectUtils.isEmpty(vo.getParentId()))
                .collect(Collectors.groupingBy(
                        PmsBudgetSubjectVO::getParentId,
                        Collectors.toList()
                ));
        // 展示父子关系
        for (PmsBudgetSubjectVO vo : parentList) {
            resultList.add(vo);
            List<PmsBudgetSubjectVO> subset = groupedAndSorted.get(vo.getId());
            if (!ObjectUtils.isEmpty(subset)) {
                groupedAndSorted.remove(vo.getId());
                // 子集排序
                sortBySubjectCode(subset);
                resultList.addAll(subset);
            }
        }
        return resultList;
    }

    @Override
    public List<PmsBudgetSubjectVO> queryListDynamic(PmsBudgetSubjectQuery query) {
        return pmsBudgetSubjectDAO.queryListDynamic(query);
    }

    @Override
    public PmsBudgetSubjectVO queryByKey(Long key) {
        PmsBudgetSubjectDO entity = pmsBudgetSubjectRepo.findById(key).orElseGet(PmsBudgetSubjectDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsBudgetSubjectVO vo = PmsBudgetSubjectConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsBudgetSubjectVO insert(PmsBudgetSubjectPayload payload) {
        //核验数据并赋值
        checkData(payload);

        PmsBudgetSubjectDO entityDo = PmsBudgetSubjectConvert.INSTANCE.toDo(payload);
        PmsBudgetSubjectVO vo = PmsBudgetSubjectConvert.INSTANCE.toVo(pmsBudgetSubjectRepo.save(entityDo));

        // 一级科目，每增加一个新的二级科目，自动更新到预算模板中
        if (!ObjectUtils.isEmpty(entityDo.getParentId())) {
            List<PmsBudgetTemplateSubjectVO> list = pmsBudgetTemplateSubjectDAO.queryByBySubjectId(entityDo.getParentId());
            List<Long> templateIds = list.stream().map(PmsBudgetTemplateSubjectVO::getTemplateId).collect(Collectors.toList());
            List<PmsBudgetTemplateSubjectDO> dDetails = new ArrayList<>();
            for (Long templateId : templateIds) {
                PmsBudgetTemplateSubjectDO templateSubject = new PmsBudgetTemplateSubjectDO();
                templateSubject.setSubjectId(vo.getId());
                templateSubject.setTemplateId(templateId);
                dDetails.add(templateSubject);
            }
            pmsBudgetTemplateSubjectDAO.saveAll(dDetails);
        }

        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            //删除科目表数据
            pmsBudgetSubjectDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSubjectStatus(Long[] keys, Integer subjectStatus) {
        if (ObjectUtils.isEmpty(keys)) {
            throw TwException.error("", "科目主键为空，请核验！");
        }
        if (ObjectUtils.isEmpty(subjectStatus)) {
            throw TwException.error("", "科目状态为空，请核验！");
        }
        pmsBudgetSubjectDAO.updateSubjectStatus(keys, subjectStatus);
    }


    /**
     * 核验数据并赋值
     *
     * @param payload
     */
    void checkData(PmsBudgetSubjectPayload payload) {
        payload.setSubjectStatus(0);
        if (!ObjectUtils.isEmpty(payload.getId())) {
            PmsBudgetSubjectVO vo = pmsBudgetSubjectDAO.queryByKey(payload.getId());
            if (ObjectUtils.isEmpty(vo)) {
                throw TwException.error("", "科目数据不存在");
            }
            if (!vo.getSubjectCode().equals(payload.getSubjectCode())) {
                PmsBudgetSubjectVO byCodeVo = pmsBudgetSubjectDAO.queryByCode(payload.getSubjectCode());
                if (!ObjectUtils.isEmpty(byCodeVo)) {
                    throw TwException.error("", "科目编号已经存在！");
                }
            }
            if (!vo.getSubjectName().equals(payload.getSubjectName())) {
                PmsBudgetSubjectVO byCodeVo = pmsBudgetSubjectDAO.queryByName(payload.getSubjectName());
                if (!ObjectUtils.isEmpty(byCodeVo)) {
                    throw TwException.error("", "科目名称已经存在！");
                }
            }
        } else {
            PmsBudgetSubjectVO vo = pmsBudgetSubjectDAO.queryByCode(payload.getSubjectCode());
            if (!ObjectUtils.isEmpty(vo)) {
                throw TwException.error("", "科目编号已经存在！");
            }
            vo = pmsBudgetSubjectDAO.queryByName(payload.getSubjectName());
            if (!ObjectUtils.isEmpty(vo)) {
                throw TwException.error("", "科目名称已经存在！");
            }
        }
        if (payload.getParentId() != null) {
            PmsBudgetSubjectVO pmsBudgetSubjectVO = pmsBudgetSubjectDAO.queryByKey(payload.getParentId());
            if (pmsBudgetSubjectVO != null) {
                if (1 == pmsBudgetSubjectVO.getSubjectLevel()) {
                    payload.setSubjectLevel(2);
                    payload.setParentName(pmsBudgetSubjectVO.getSubjectName());
                } else {
                    throw TwException.error("", "上级科目必须是一级科目，请核验！");
                }
            } else {
                throw TwException.error("", "上级科目不存在，请核验！");
            }
        } else {
            payload.setSubjectLevel(1);
        }
    }


    /**
     * 根据科目编号排序
     * @param list
     */
    void sortBySubjectCode(List<PmsBudgetSubjectVO> list) {
        list.sort(new Comparator<PmsBudgetSubjectVO>() {
            @Override
            public int compare(PmsBudgetSubjectVO obj1, PmsBudgetSubjectVO obj2) {
                String str1 = obj1.getSubjectCode();
                String str2 = obj2.getSubjectCode();

                // 将字符串转换为整数进行比较
                int int1 = Integer.parseInt(str1);
                int int2 = Integer.parseInt(str2);

                // 检查是否以0开头
                boolean isStr1StartingWithZero = str1.startsWith("0");
                boolean isStr2StartingWithZero = str2.startsWith("0");

                // 如果一个字符串以0开头而另一个不是，则将以0开头的字符串排在前面
                if (isStr1StartingWithZero && !isStr2StartingWithZero) {
                    return -1;
                } else if (!isStr1StartingWithZero && isStr2StartingWithZero) {
                    return 1;
                } else {
                    // 否则，使用整数值进行比较
                    return Integer.compare(int1, int2);
                }
            }
        });
    }
}
