package com.elitesland.tw.tw5.server.prd.prj.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.prj.payload.PrjProjectMemberPayload;
import com.elitesland.tw.tw5.api.prd.prj.payload.PrjProjectPayload;
import com.elitesland.tw.tw5.api.prd.prj.query.PrjProjectQuery;
import com.elitesland.tw.tw5.api.prd.prj.vo.PrjProjectVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.prj.entity.PrjProjectDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.PrjProjectMemberDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.QPrjProjectDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.QPrjProjectMemberDO;
import com.elitesland.tw.tw5.server.prd.prj.repo.PrjProjectMemberRepo;
import com.elitesland.tw.tw5.server.prd.prj.repo.PrjProjectRepo;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.Arrays;
import java.util.List;

/**
 * @author zoey
 * @Description:
 * @date 2022/5/20 - 9:17
 */
@Repository
@RequiredArgsConstructor
public class PrjProjectDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrjProjectRepo repo;
    private final PrjProjectMemberRepo repoMember;
    private final QPrjProjectDO qdo = QPrjProjectDO.prjProjectDO;

    private final QPrjProjectMemberDO qdoMember = QPrjProjectMemberDO.prjProjectMemberDO;

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrjProjectDO save(PrjProjectDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrjProjectMemberDO saveMember(PrjProjectMemberDO ado) {
        return repoMember.save(ado);
    }

    /**
     * 调用jpa的保存
     *
     * @param ados do对象
     * @return 保存后的对象
     */
    public List<PrjProjectMemberDO> saveMemberAll(List<PrjProjectMemberDO> ados) {
        return repoMember.saveAll(ados);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrjProjectDO> saveAll(List<PrjProjectDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrjProjectVO queryByKey(Long id) {
        JPAQuery<PrjProjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    public PrjProjectVO queryByNo(String projectNo) {
        JPAQuery<PrjProjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectNo.eq(projectNo));
        return jpaQuery.fetchFirst();
    }


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrjProjectVO> getJpaQuerySelect() {
        JPAQuery<PrjProjectVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PrjProjectVO.class,
                qdo.id,
                qdo.projectName,
                qdo.projectNo,
                qdo.projectType
        )).from(qdo);
        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrjProjectVO> getJpaQueryWhere(PrjProjectQuery query) {
        JPAQuery<PrjProjectVO> jpaQuery = getJpaQuerySelect();

        if (!ObjectUtils.isEmpty(query.getProjectName())) {
            jpaQuery.where(qdo.projectName.eq(query.getProjectName()));
        }
        if (!ObjectUtils.isEmpty(query.getProjectNo())) {
            jpaQuery.where(qdo.projectNo.eq(query.getProjectNo()));
        }
        if (!ObjectUtils.isEmpty(query.getProjectType())) {
            jpaQuery.where(qdo.projectType.like(SqlUtil.toSqlLikeString(query.getProjectType())));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 动态查询集合(不分页查询)
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrjProjectVO> queryListDynamic(PrjProjectQuery query) {
        //默认按照时间倒叙排序
        OrderItem orderItem = OrderItem.asc("createTime");
        query.setOrders(Arrays.asList(orderItem));
        JPAQuery<PrjProjectVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }


    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrjProjectVO> queryPaging(PrjProjectQuery query) {
        JPAQuery<PrjProjectVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PrjProjectVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        System.out.println(result.getTotal());
        return PagingVO.<PrjProjectVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrjProjectPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getProjectName() != null) {
            update.set(qdo.projectName, payload.getProjectName());
        }
        if (payload.getProjectNo() != null) {
            update.set(qdo.projectNo, payload.getProjectNo());
        }
        if (payload.getProjectType() != null) {
            update.set(qdo.projectType, payload.getProjectType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {

        }
        // 执行修改
        return update.execute();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateMemberByKeyDynamic(PrjProjectMemberPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoMember)
                .where(qdoMember.id.eq(payload.getId()));
        if (payload.getProjectId() != null) {
            update.set(qdoMember.projectId, payload.getProjectId());
        }
        if (payload.getUserId() != null) {
            update.set(qdoMember.userId, payload.getUserId());
        }
//        if (payload.getTaskPackageId() != null) {
//            update.set(qdoMember.taskPackageId, payload.getTaskPackageId());
//        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {

        }
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param key    主键
     * @param userId 负责人id
     * @return 删除的行数
     */
    public long updateProjectManger(Long key, Long userId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.manageUserId, userId)
                .where(qdo.id.eq(key));
        return update.execute();
    }

    /**
     * 逻辑删除项目成员
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteMemberSoft(List<Long> keys, List<Long> userIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoMember)
                .set(qdoMember.deleteFlag, 1)
                .where(qdoMember.id.in(keys))
                .where(qdoMember.userId.notIn(userIds));
        return update.execute();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    public List<Long> getUserIds(List<Long> keys) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdoMember.userId).from(qdoMember).where(qdoMember.id.in(keys)).where(qdoMember.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 逻辑删除项目成员
     *
     * @param key     项目主键
     * @param userIds 项目成员ids
     * @return 删除的行数
     */
    public long deleteMemberByUserId(Long key, List<Long> userIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoMember)
                .set(qdoMember.deleteFlag, 1).where(qdoMember.projectId.eq(key))
                .where(qdoMember.userId.in(userIds));
        return update.execute();
    }
}
