package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseAgreementResPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseAgreementResQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseAgreementResVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchaseAgreementResRepo;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchaseAgreementResDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseAgreementResDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 采购协议关联资源
 *
 * @author lemon
 * @date 2023-11-02
 */
@Repository
@RequiredArgsConstructor
public class PurchaseAgreementResDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchaseAgreementResRepo repo;
    private final QPurchaseAgreementResDO qdo = QPurchaseAgreementResDO.purchaseAgreementResDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementResVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchaseAgreementResVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 资源ID
                qdo.resId,
                // 采购协议单据ID
                qdo.documentId,
                // 采购协议单据编号
                qdo.documentNo,
                // 资源开始日期
                qdo.resStartDate,
                // 资源结束日期
                qdo.resEndDate,
                // 合作方式
                qdo.cooperationWay,
                // 结算单价
                qdo.settlePrice,
                // 出差补助
                qdo.travelAllowance
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementResVO> getJpaQueryWhere(PurchaseAgreementResQuery query) {
        JPAQuery<PurchaseAgreementResVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
//        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseAgreementResQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseAgreementResQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 资源ID 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        }
        /** 采购协议单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDocumentId())) {
            list.add(qdo.documentId.eq(query.getDocumentId()));
        }
        /** 采购协议单据编号 精确 */
        if (!ObjectUtils.isEmpty(query.getDocumentNo())) {
            list.add(qdo.documentNo.eq(query.getDocumentNo()));
        }
        /** 资源开始日期 精确 */
        if (!ObjectUtils.isEmpty(query.getResStartDate())) {
            list.add(qdo.resStartDate.eq(query.getResStartDate()));
        }
        /** 资源结束日期 精确 */
        if (!ObjectUtils.isEmpty(query.getResEndDate())) {
            list.add(qdo.resEndDate.eq(query.getResEndDate()));
        }
        /** 资源结束日期 精确 */
        if (!ObjectUtils.isEmpty(query.getCooperationWay())) {
            list.add(qdo.cooperationWay.eq(query.getCooperationWay()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseAgreementResVO queryByKey(Long id) {
        JPAQuery<PurchaseAgreementResVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseAgreementResVO> queryListDynamic(PurchaseAgreementResQuery query) {
        JPAQuery<PurchaseAgreementResVO> jpaQuery = getJpaQueryWhere(query);
        // 按照资源人、资源日期升序排序
        jpaQuery.orderBy(qdo.resId.asc(), qdo.resStartDate.asc());
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseAgreementResVO> queryPaging(PurchaseAgreementResQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PurchaseAgreementResVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseAgreementResVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseAgreementResVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchaseAgreementResDO save(PurchaseAgreementResDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchaseAgreementResDO> saveAll(List<PurchaseAgreementResDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchaseAgreementResPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 资源ID
        if (payload.getResId() != null) {
            update.set(qdo.resId, payload.getResId());
        }
        // 采购协议单据ID
        if (payload.getDocumentId() != null) {
            update.set(qdo.documentId, payload.getDocumentId());
        }
        // 采购协议单据编号
        if (payload.getDocumentNo() != null) {
            update.set(qdo.documentNo, payload.getDocumentNo());
        }
        // 资源开始日期
        if (payload.getResStartDate() != null) {
            update.set(qdo.resStartDate, payload.getResStartDate());
        }
        // 资源结束日期
        if (payload.getResEndDate() != null) {
            update.set(qdo.resEndDate, payload.getResEndDate());
        }
        if (payload.getRemark()!=null) {
            update.set(qdo.remark, payload.getRemark());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 合作方式
        if (payload.getCooperationWay() != null) {
            update.set(qdo.cooperationWay, payload.getCooperationWay());
        }
        // 结算单价
        if (payload.getSettlePrice() != null) {
            update.set(qdo.settlePrice, payload.getSettlePrice());
        }
        // 出差补助
        if (payload.getTravelAllowance() != null) {
            update.set(qdo.travelAllowance, payload.getTravelAllowance());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 资源ID
            if (nullFields.contains("resId")) {
                update.setNull(qdo.resId);
            }
            // 采购协议单据ID
            if (nullFields.contains("documentId")) {
                update.setNull(qdo.documentId);
            }
            // 采购协议单据编号
            if (nullFields.contains("documentNo")) {
                update.setNull(qdo.documentNo);
            }
            // 资源开始日期
            if (nullFields.contains("resStartDate")) {
                update.setNull(qdo.resStartDate);
            }
            // 资源结束日期
            if (nullFields.contains("resEndDate")) {
                update.setNull(qdo.resEndDate);
            }
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
            // 合作方式
            if (nullFields.contains("cooperationWay")) {
                update.setNull(qdo.cooperationWay);
            }
            // 结算单价
            if (nullFields.contains("settlePrice")) {
                update.setNull(qdo.settlePrice);
            }
            // 出差补助
            if (nullFields.contains("travelAllowance")) {
                update.setNull(qdo.travelAllowance);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据协议id逻辑删除
     * @param documentIdList 协议idList
     */
    public void deleteSoftByDocumentIdList(List<Long> documentIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.documentId.in(documentIdList));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }
}

