package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitesland.tw.tw5.api.prd.salecon.payload.ConAchievePayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConAchieveQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConAchieveVO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConAchieveDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConAchieveDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConAchieveRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 业绩统计
 *
 * @author likunpeng
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class ConAchieveDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConAchieveRepo repo;
    private final QConAchieveDO qdo = QConAchieveDO.conAchieveDO;
    private final QSaleConContractDO qdoCon = QSaleConContractDO.saleConContractDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConAchieveVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConAchieveVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 合同ID
                qdo.contractId,
                qdo.achieveIdV4,
                // 平台提成比例
                qdo.percentage,
                // 折算后有效销售额
                qdo.obversionEffectiveAmt,
                // 业绩统计状态
                qdo.achieveStatus,
                // 签单额度
                qdoCon.amt,
                // 有效合同金额
                qdoCon.effectiveAmt
        )).from(qdo)
                .leftJoin(qdoCon).on(qdoCon.id.longValue().eq(qdo.contractId.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConAchieveVO> getJpaQueryWhere(ConAchieveQuery query) {
        JPAQuery<ConAchieveVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConAchieveQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConAchieveQuery query){
        List<Predicate> list = new ArrayList<>();
        list.add(qdo.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 合同ID 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 折算后有效销售额 精确 */
        if (!ObjectUtils.isEmpty(query.getObversionEffectiveAmt())) {
            list.add(qdo.obversionEffectiveAmt.eq(query.getObversionEffectiveAmt()));
        }
        /** 业绩统计状态 精确 */
        if (!ObjectUtils.isEmpty(query.getAchieveStatus())) {
            list.add(qdo.achieveStatus.eq(query.getAchieveStatus()));
        }
        /** 平台提成比例 精确 */
        if (!ObjectUtils.isEmpty(query.getPercentage())) {
            list.add(qdo.percentage.eq(query.getPercentage()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConAchieveVO queryByKey(Long id) {
        JPAQuery<ConAchieveVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConAchieveVO> queryListDynamic(ConAchieveQuery query) {
        JPAQuery<ConAchieveVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConAchieveVO> queryPaging(ConAchieveQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConAchieveVO> jpaQuery = getJpaQueryWhere(query);
        List<ConAchieveVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConAchieveVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConAchieveDO save(ConAchieveDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConAchieveDO> saveAll(List<ConAchieveDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConAchievePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 合同ID
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 折算后有效销售额
        if (payload.getObversionEffectiveAmt() != null) {
            update.set(qdo.obversionEffectiveAmt, payload.getObversionEffectiveAmt());
        }
        // 业绩统计状态
        if (payload.getAchieveStatus() != null) {
            update.set(qdo.achieveStatus, payload.getAchieveStatus());
        }
        // 业绩统计状态
        if (payload.getAchieveIdV4() != null) {
            update.set(qdo.achieveIdV4, payload.getAchieveIdV4());
        }
        // 平台提成比例
        if (payload.getPercentage() != null) {
            update.set(qdo.percentage, payload.getPercentage());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 合同ID
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 折算后有效销售额
            if (nullFields.contains("obversionEffectiveAmt")) {
                update.setNull(qdo.obversionEffectiveAmt);
            }
            // 业绩统计状态
            if (nullFields.contains("achieveStatus")) {
                update.setNull(qdo.achieveStatus);
            }
            // 业绩统计状态
            if (nullFields.contains("achieveIdV4")) {
                update.setNull(qdo.achieveIdV4);
            }
            // 平台提成比例
            if (nullFields.contains("percentage")) {
                update.setNull(qdo.percentage);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据合同ID查询业绩统计
     * @param contractId
     * @return ConAchieveVO
     */
    public ConAchieveVO queryByContractId(Long contractId) {
        JPAQuery<ConAchieveVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.contractId.eq(contractId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.groupBy(qdo.id);
        return jpaQuery.fetchFirst();
    }

    /**
     * 修改合同的含税与不含税金额
     * @param amt
     * @param contractId
     */
    public void updateConAmt(BigDecimal amt,BigDecimal effectiveAmt, Long contractId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoCon);
        update.set(qdoCon.amt,amt).set(qdoCon.effectiveAmt,effectiveAmt);
        update.where(qdoCon.id.eq(contractId)).execute();
    }
}

