package com.elitesland.tw.tw5.server.prd.system.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.org.query.PrdOrgEmployeeQuery;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemNewFunctionPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemNewFunctionQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemNewFunctionService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemBusinessObjectVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemNewFunctionVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionTypeEnum;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemNewFunctionConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemBusinessObjectDAO;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemNewFunctionDAO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemNewFunctionDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemNewFunctionRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 业务对象 功能列表
 *
 * @Author Bill
 * @Date 2023/9/18 13:15
 **/
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemNewFunctionServiceImpl implements PrdSystemNewFunctionService {

    private final PrdSystemNewFunctionRepo repo;
    private final PrdSystemNewFunctionDAO dao;
    private final PrdSystemBusinessObjectDAO businessObjectDAO;

    /**
     * 业务对象 功能列表 新增
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemNewFunctionVO insert(PrdSystemNewFunctionPayload payload) {
        //校验
        check(payload);
        //对象转换
        PrdSystemNewFunctionDO entityDo = PrdSystemNewFunctionConvert.INSTANCE.toDo(payload);
        repo.save(entityDo);
        return PrdSystemNewFunctionConvert.INSTANCE.toVo(entityDo);
    }

    /**
     * 业务对象 功能列表 编辑
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemNewFunctionVO update(PrdSystemNewFunctionPayload payload) {
        //校验
        checkUpdate(payload);
        //查询对应的数据是否已经存在
        PrdSystemNewFunctionDO entity = repo.findById(payload.getId()).orElseGet(PrdSystemNewFunctionDO::new);
        Assert.notNull(entity.getId(), "不存在");
        //查询编号是否重复
        if (StringUtils.hasText(payload.getFunctionCode()) && !entity.getFunctionCode().equals(payload.getFunctionCode())){
                Assert.isNull(dao.queryByFunctionCode(payload.getFunctionCode()),"functionCode不可重复");
        }
        PrdSystemNewFunctionDO entityDo = PrdSystemNewFunctionConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final PrdSystemNewFunctionDO save = repo.save(entity);
        return PrdSystemNewFunctionConvert.INSTANCE.toVo(save);
    }

    /**
     * 校验更新数据
     * @param payload
     */
    private void checkUpdate(PrdSystemNewFunctionPayload payload) {
        if (payload.getId() == null){
            throw TwException.error("", "id不能为空");
        }
    }

    /**
     * 业务对象 功能列表 逻辑删除
     * @param keys
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!CollectionUtils.isEmpty(keys)){
            repo.deleteSoft(keys);
        }
    }

    /**
     * 功能列表主键查询
     * @param id
     * @return
     */
    @Override
    public PrdSystemNewFunctionVO get(Long id) {
        PrdSystemNewFunctionDO entity = repo.findById(id).orElseGet(PrdSystemNewFunctionDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PrdSystemNewFunctionVO prdSystemNewFunctionVO = PrdSystemNewFunctionConvert.INSTANCE.toVo(entity);
        return prdSystemNewFunctionVO;
    }

    /**
     * 业务对象 功能列表分页查询
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdSystemNewFunctionVO> paging(PrdSystemNewFunctionQuery query) {
        //校验 参数
        checkQuery(query);
        Page<PrdSystemNewFunctionDO> pageDo = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        Page<PrdSystemNewFunctionVO> pageVo = pageDo.map(PrdSystemNewFunctionConvert.INSTANCE::toVo);
        return PageUtil.toPageVo(pageVo);
    }

    /**
     * 业务对象 功能列表分页查询 keyword多列匹配
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdSystemNewFunctionVO> pagingKeyword(PrdSystemNewFunctionQuery query) {
        //校验参数
        checkQuery(query);
        return dao.queryByKeyword(query);
    }

    /**
     * 查询参数校验
     * @param query
     */
    private void checkQuery(PrdSystemNewFunctionQuery query) {
        if (query.getObjectId() == null){
            throw TwException.error("", "objectId不能为空");
        }
    }

    /**
     * 新增功能列表 校验
     * @param payload
     */
    private void check(PrdSystemNewFunctionPayload payload) {
        //校验
        if (!StringUtils.hasText(payload.getFunctionCode())){
            throw TwException.error("", "functionCode不能为空");
        }
        if (!StringUtils.hasText(payload.getFunctionName())){
            throw TwException.error("", "functionName不能为空");
        }
        if (!StringUtils.hasText(payload.getClientType())){
            throw TwException.error("", "clientType不能为空");
        }
        if (StringUtils.hasText(payload.getFunctionCode())){
            //查此编号
            Assert.isNull(dao.queryByFunctionCode(payload.getFunctionCode()),"objectCode已存在");
        }
        if (!StringUtils.hasText(payload.getFunctionType())){
            throw TwException.error("", "functionType不能为空");
        }
        if (!FunctionTypeEnum.LINK_BUTTON.getName().equals(payload.getFunctionType())) {
            throw TwException.error("", "只能添加LINK_BUTTON功能");
        }
        if (payload.getFunctionStatus() == null){
            throw TwException.error("", "functionStatus不能为空");
        }
        if (payload.getObjectId() == null){
            throw TwException.error("", "objectId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoCreate(Map<String, List<PrdSystemNewFunctionPayload>> functionMap) {
        List<String> objectCodes = new ArrayList<>(functionMap.keySet());

        List<String> referDomainCodes = functionMap.values().stream()
                .flatMap(Collection::stream)
                .map(PrdSystemNewFunctionPayload::getReferDomainCodes).filter(Objects::nonNull)
                .flatMap(Collection::stream).filter(Objects::nonNull).distinct().toList();

        objectCodes.addAll(referDomainCodes);

        // 查询业务对象
        List<PrdSystemBusinessObjectVO> objectList = businessObjectDAO.queryByObjectCodes(objectCodes);
        Map<String, PrdSystemBusinessObjectVO> objectMap = objectList.stream().collect(Collectors.toMap(PrdSystemBusinessObjectVO::getObjectCode, e -> e, (l, r) -> r));

        // 查询已有功能
        List<Long> objectIds = objectList.stream().map(PrdSystemBusinessObjectVO::getId).toList();
        List<PrdSystemNewFunctionVO> existFunctionList = dao.queryByObjectIds(objectIds);
        Map<String, PrdSystemNewFunctionVO> existFunctionMap = existFunctionList.stream().collect(Collectors.toMap(PrdSystemNewFunctionVO::getFunctionCode, e -> e, (l, r) -> r));

        List<PrdSystemNewFunctionDO> insertList = new ArrayList<>();
        List<PrdSystemNewFunctionDO> updateList = new ArrayList<>();

        functionMap.forEach((k,v) -> {
            if (objectMap.containsKey(k)) {
                PrdSystemBusinessObjectVO businessObject = objectMap.get(k);
                v.forEach(f -> {
                    this.buildNewFunctionEntity(existFunctionMap, insertList, updateList, businessObject, f, f.getFunctionCode());
                    // 外部引用功能
                    if(!CollectionUtils.isEmpty(f.getReferDomainCodes())){
                        f.getReferDomainCodes().forEach(referDomainCode ->{
                            if (objectMap.containsKey(referDomainCode)) {
                                PrdSystemBusinessObjectVO referDomainBusiness = objectMap.get(referDomainCode);
                                this.buildNewFunctionEntity(existFunctionMap, insertList, updateList, referDomainBusiness, f, referDomainCode + "/" + f.getFunctionCode());
                            }
                        });
                    }
                });
            }
        });

        List<String> nowFunctionCodes = functionMap.values().stream().flatMap(Collection::stream).map(PrdSystemNewFunctionPayload::getFunctionCode).distinct().toList();
        List<String> referFunctionCodes = functionMap.values().stream().flatMap(Collection::stream)
                .filter(f -> !CollectionUtils.isEmpty(f.getReferDomainCodes()))
                .map(f -> f.getReferDomainCodes().stream().map(referDomainCode -> referDomainCode + "/" + f.getFunctionCode()).toList())
                .flatMap(Collection::stream).filter(Objects::nonNull).distinct().toList();
        List<Long> deleteIds = existFunctionList.stream().filter(e -> !nowFunctionCodes.contains(e.getFunctionCode()) && !referFunctionCodes.contains(e.getFunctionCode()) && !FunctionTypeEnum.LINK_BUTTON.getName().equals(e.getFunctionType())).map(PrdSystemNewFunctionVO::getId).toList();

        if(!CollectionUtils.isEmpty(insertList)) {
            repo.saveAll(insertList);
        }
        if(!CollectionUtils.isEmpty(updateList)) {
            repo.saveAll(updateList);
        }
        if(!CollectionUtils.isEmpty(deleteIds)) {
            dao.deleteByIds(deleteIds);
        }
    }

    private void buildNewFunctionEntity(Map<String, PrdSystemNewFunctionVO> existFunctionMap,
                                        List<PrdSystemNewFunctionDO> insertList,
                                        List<PrdSystemNewFunctionDO> updateList,
                                        PrdSystemBusinessObjectVO businessObject,
                                        PrdSystemNewFunctionPayload f,
                                        String functionCode) {
        // 没有指定功能编码则取功能
        if(functionCode == null){
            functionCode = f.getFunctionCode();
        }
        if (existFunctionMap.containsKey(functionCode)) {
            PrdSystemNewFunctionVO prdSystemNewFunctionVO = existFunctionMap.get(functionCode);
            PrdSystemNewFunctionDO entityDo = PrdSystemNewFunctionConvert.INSTANCE.toEntity(prdSystemNewFunctionVO);

            entityDo.setFunctionName(f.getFunctionName());
            entityDo.setFunctionType(f.getFunctionType());
            entityDo.setClientType(f.getClientType());

            updateList.add(entityDo);
        }else{
            PrdSystemNewFunctionDO entityDo = PrdSystemNewFunctionConvert.INSTANCE.toDo(f);
            entityDo.setObjectId(businessObject.getId());
            entityDo.setFunctionCode(functionCode);

            insertList.add(entityDo);
        }
    }

    /**
     * 业务对象 那些用户拥有该功能
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdOrgEmployeeVO> pagingUser(PrdOrgEmployeeQuery query) {
        //校验
        checkListUser(query);
        PagingVO<PrdOrgEmployeeVO> pagingVO = dao.queryByListUser(query);
        List<PrdOrgEmployeeVO> records = pagingVO.getRecords();
        for (PrdOrgEmployeeVO record : records) {
            record.setRoleDatas(dao.queryRoleByUserId(record.getUserId()));
        }
        return pagingVO;
    }

    /**
     * 业务对象 功能下拉框
     * @param objectId
     * @return
     */
    @Override
    public List<PrdSystemNewFunctionVO> list(Long objectId) {
        //检查
        checkList(objectId);
        return dao.queryByObjectIds(Arrays.asList(objectId));
    }

    /**
     * 根据功能编码查询功能
     * @param functionCode
     * @return
     */
    @Override
    public PrdSystemNewFunctionVO getByFunctionCode(String functionCode) {
        return dao.queryByFunctionCode(functionCode);
    }

    /**
     * 业务对象 可配置功能列表
     *
     * @param objectId
     * @return
     */
    @Override
    public List<PrdSystemNewFunctionVO> listFieldConfig(Long objectId) {
        return dao.queryByObjectId(objectId);
    }

    /**
     * 通过功能编码查询 业务对象主键
     *
     * @param codeRegx
     * @return
     */
    @Override
    public Long getObjectIdByFunctionCode(String codeRegx) {
        return dao.getObjectIdByFunctionCode(codeRegx);
    }

    /**
     * 检查功能下拉框 参数
     * @param objectId
     */
    private void checkList(Long objectId) {
        if (ObjectUtils.isEmpty(objectId)) {
            throw TwException.error("", "objectId不能为空");
        }
    }

    /**
     * 校验用户列表查询参数
     * @param query
     */
    private void checkListUser(PrdOrgEmployeeQuery query) {
        if (ObjectUtils.isEmpty(query.getFunctionId())) {
            throw TwException.error("", "functionId不能为空");
        }
    }

}
