package com.elitesland.tw.tw5.server.prd.task.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.task.payload.TaskPackagePayload;
import com.elitesland.tw.tw5.api.prd.task.query.TaskPackageQuery;
import com.elitesland.tw.tw5.api.prd.task.vo.TaskPackageVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.task.entity.QTaskPackageDO;
import com.elitesland.tw.tw5.server.prd.task.entity.TaskPackageDO;
import com.elitesland.tw.tw5.server.prd.task.repo.TaskPackageRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 任务
 *
 * @author xxb
 * @date 2023-11-16
 */
@Repository
@RequiredArgsConstructor
public class TaskPackageDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TaskPackageRepo repo;
    private final QTaskPackageDO qdo = QTaskPackageDO.taskPackageDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TaskPackageVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TaskPackageVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                qdo.reasonId,
                qdo.reasonName,
                qdo.reasonType,
                qdo.taskPackageName,
                qdo.disterResId,
                qdo.receiverResId,
                qdo.receiverBuId,
                qdo.compositeAbilityId,
                qdo.startDate,
                qdo.endDate,
                qdo.taskPackageStatus,
                qdo.taskPackageType,
                qdo.taskPackageDes,
                qdo.acceptorId,
                qdo.acceptMethod,
                qdo.pricingMethod,
                qdo.autoSettleFlag,
                qdo.guaranteeRate,
                qdo.settlePrice,
                qdo.incomePrice,
                qdo.jobType1,
                qdo.jobType2,
                qdo.taskPackageNo,
                qdo.eqvaRatio
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TaskPackageVO> getJpaQueryWhere(TaskPackageQuery query) {
        JPAQuery<TaskPackageVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TaskPackageQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        Long total = jpaQuery.fetchOne();
        total=total==null?0L:total;
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TaskPackageQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonType())) {
            list.add(qdo.reasonType.eq(query.getReasonType()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonId())) {
            list.add(qdo.reasonId.eq(query.getReasonId()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonTypes())) {
            list.add(qdo.reasonType.in(query.getReasonTypes()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonName())) {
            list.add(qdo.reasonName.like(SqlUtil.toSqlLikeString(query.getReasonName())));
        }
        if (!ObjectUtils.isEmpty(query.getTaskPackageName())) {
            list.add(qdo.taskPackageName.like(SqlUtil.toSqlLikeString(query.getTaskPackageName())));
        }
        if (!ObjectUtils.isEmpty(query.getDisterResId())) {
            list.add(qdo.disterResId.eq(query.getDisterResId()));
        }

        if (!ObjectUtils.isEmpty(query.getReceiverResId())) {
            list.add(qdo.receiverResId.eq(query.getReceiverResId()));
        }
        if (!ObjectUtils.isEmpty(query.getReceiverResIds())) {
            list.add(qdo.receiverResId.in(query.getReceiverResIds()));
        }
        if (!ObjectUtils.isEmpty(query.getReceiverBuId())) {
            list.add(qdo.receiverBuId.eq(query.getReceiverBuId()));
        }
        if (!ObjectUtils.isEmpty(query.getCompositeAbilityId())) {
            list.add(qdo.compositeAbilityId.eq(query.getCompositeAbilityId()));
        }
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.eq(query.getEndDate()));
        }
        if (!ObjectUtils.isEmpty(query.getTaskPackageStatus())) {
            list.add(qdo.taskPackageStatus.eq(query.getTaskPackageStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getTaskPackageType())) {
            list.add(qdo.taskPackageType.eq(query.getTaskPackageType()));
        }
        if (!ObjectUtils.isEmpty(query.getTaskPackageDes())) {
            list.add(qdo.taskPackageDes.eq(query.getTaskPackageDes()));
        }
        if (!ObjectUtils.isEmpty(query.getAcceptorId())) {
            list.add(qdo.acceptorId.eq(query.getAcceptorId()));
        }
        if (!ObjectUtils.isEmpty(query.getAcceptMethod())) {
            list.add(qdo.acceptMethod.eq(query.getAcceptMethod()));
        }
        if (!ObjectUtils.isEmpty(query.getPricingMethod())) {
            list.add(qdo.pricingMethod.eq(query.getPricingMethod()));
        }
        if (!ObjectUtils.isEmpty(query.getAutoSettleFlag())) {
            list.add(qdo.autoSettleFlag.eq(query.getAutoSettleFlag()));
        }
        if (!ObjectUtils.isEmpty(query.getGuaranteeRate())) {
            list.add(qdo.guaranteeRate.eq(query.getGuaranteeRate()));
        }
        if (!ObjectUtils.isEmpty(query.getSettlePrice())) {
            list.add(qdo.settlePrice.eq(query.getSettlePrice()));
        }
        if (!ObjectUtils.isEmpty(query.getIncomePrice())) {
            list.add(qdo.incomePrice.eq(query.getIncomePrice()));
        }
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        }
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        }
        //主键批量查询
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TaskPackageVO queryByKey(Long id) {
        JPAQuery<TaskPackageVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键集合 查询
     *
     * @param ids
     * @return
     */
    public List<TaskPackageVO> queryByKeys(List<Long> ids) {
        JPAQuery<TaskPackageVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskPackageVO> queryListDynamic(TaskPackageQuery query) {
        JPAQuery<TaskPackageVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TaskPackageVO> queryPaging(TaskPackageQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<TaskPackageVO> jpaQuery = getJpaQueryWhere(query);
        List<TaskPackageVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TaskPackageVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TaskPackageDO save(TaskPackageDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TaskPackageDO> saveAll(List<TaskPackageDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TaskPackagePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        if (payload.getReasonId() != null) {
            update.set(qdo.reasonId, payload.getReasonId());
        }
        if (payload.getReasonType() != null) {
            update.set(qdo.reasonType, payload.getReasonType());
        }
        if (payload.getReasonName() != null) {
            update.set(qdo.reasonName, payload.getReasonName());
        }
        if (payload.getTaskPackageName() != null) {
            update.set(qdo.taskPackageName, payload.getTaskPackageName());
        }
        if (payload.getTaskPackageStatus() != null) {
            update.set(qdo.taskPackageStatus, payload.getTaskPackageStatus());
        }
        if (payload.getTaskPackageType() != null) {
            update.set(qdo.taskPackageType, payload.getTaskPackageType());
        }
        if (payload.getTaskPackageDes() != null) {
            update.set(qdo.taskPackageDes, payload.getTaskPackageDes());
        }
        if (payload.getAcceptorId() != null) {
            update.set(qdo.acceptorId, payload.getAcceptorId());
        }
        if (payload.getAcceptMethod() != null) {
            update.set(qdo.acceptMethod, payload.getAcceptMethod());
        }
        if (payload.getPricingMethod() != null) {
            update.set(qdo.pricingMethod, payload.getPricingMethod());
        }
        if (payload.getAutoSettleFlag() != null) {
            update.set(qdo.autoSettleFlag, payload.getAutoSettleFlag());
        }
        if (payload.getGuaranteeRate() != null) {
            update.set(qdo.guaranteeRate, payload.getGuaranteeRate());
        }
        if (payload.getSettlePrice() != null) {
            update.set(qdo.settlePrice, payload.getSettlePrice());
        }
        if (payload.getIncomePrice() != null) {
            update.set(qdo.incomePrice, payload.getIncomePrice());
        }
        if (payload.getJobType1() != null) {
            update.set(qdo.jobType1, payload.getJobType1());
        }
        if (payload.getJobType2() != null) {
            update.set(qdo.jobType2, payload.getJobType2());
        }
        if (payload.getReceiverResId() != null) {
            update.set(qdo.receiverResId, payload.getReceiverResId());
        }
        if (payload.getEqvaRatio() != null) {
            update.set(qdo.eqvaRatio, payload.getEqvaRatio());
        }
        if (payload.getReceiverBuId() != null) {
            update.set(qdo.receiverBuId, payload.getReceiverBuId());
        }

        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

