package com.elitesland.tw.tw5.server.prd.taskpro.repo.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.personplan.vo.TaskMemberRefVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.QTaskMemberRefDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.TaskMemberRefDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.payload.TaskMemberRefPayload;
import com.elitesland.tw.tw5.server.prd.taskpro.model.query.TaskMemberRefQuery;
import com.elitesland.tw.tw5.server.prd.taskpro.repo.TaskMemberRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 通用任务成员关系
 *
 * @author duwh
 * @date 2024-02-01
 */
@Repository
@RequiredArgsConstructor
public class TaskMemberRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TaskMemberRefRepo repo;
    private final QTaskMemberRefDO qdo = QTaskMemberRefDO.taskMemberRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TaskMemberRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TaskMemberRefVO.class,
                qdo.id,
                qdo.taskId,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 关联主键类型 USER、RELATED
                qdo.type,
                // 关联类型 FOLLOW:关注人 PART:参与人
                qdo.objType,
                // 关联主键
                qdo.objId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TaskMemberRefVO> getJpaQueryWhere(TaskMemberRefQuery query) {
        JPAQuery<TaskMemberRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TaskMemberRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TaskMemberRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getTaskId())) {
            list.add(qdo.taskId.eq(query.getTaskId()));
        }
        /** 关联主键类型 USER、RELATED 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 关联类型 FOLLOW:关注人 PART:参与人  精确 */
        if (!ObjectUtils.isEmpty(query.getObjType())) {
            list.add(qdo.objType.eq(query.getObjType()));
        }
        /** 关联主键 精确 */
        if (!ObjectUtils.isEmpty(query.getObjId())) {
            list.add(qdo.objId.eq(query.getObjId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TaskMemberRefVO queryByKey(Long id) {
        JPAQuery<TaskMemberRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskMemberRefVO> queryListDynamic(TaskMemberRefQuery query) {
        JPAQuery<TaskMemberRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TaskMemberRefVO> queryPaging(TaskMemberRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TaskMemberRefVO> jpaQuery = getJpaQueryWhere(query);
        List<TaskMemberRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TaskMemberRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TaskMemberRefDO save(TaskMemberRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TaskMemberRefDO> saveAll(List<TaskMemberRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TaskMemberRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        if (payload.getTaskId() != null) {
            update.set(qdo.taskId, payload.getTaskId());
        }
        // 关联主键类型 USER、RELATED
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 关联类型 FOLLOW:关注人 PART:参与人
        if (payload.getObjType() != null) {
            update.set(qdo.objType, payload.getObjType());
        }
        // 关联主键
        if (payload.getObjId() != null) {
            update.set(qdo.objId, payload.getObjId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            if (nullFields.contains("taskId")) {
                update.setNull(qdo.taskId);
            }
            // 关联主键类型 USER、RELATED
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 关联类型 FOLLOW:关注人 PART:参与人
            if (nullFields.contains("objType")) {
                update.setNull(qdo.objType);
            }
            // 关联主键
            if (nullFields.contains("objId")) {
                update.setNull(qdo.objId);
            }
        }
        //拼装更新
        // SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        // SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long deleteSoftByTaskId(Long taskId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.taskId.eq(taskId));
        //拼装更新
        // SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

