package com.elitesland.tw.tw5.server.prd.taskpro.repo.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;

import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.QTaskSceneCategoryRefDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.TaskSceneCategoryRefDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.payload.TaskSceneCategoryRefPayload;
import com.elitesland.tw.tw5.server.prd.taskpro.model.query.TaskSceneCategoryRefQuery;
import com.elitesland.tw.tw5.server.prd.taskpro.model.vo.TaskSceneCategoryRefVO;
import com.elitesland.tw.tw5.server.prd.taskpro.repo.TaskSceneCategoryRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 任务场景类别关系表
 *
 * @author duwh
 * @date 2024-01-31
 */
@Repository
@RequiredArgsConstructor
public class TaskSceneCategoryRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TaskSceneCategoryRefRepo repo;
    private final QTaskSceneCategoryRefDO qdo = QTaskSceneCategoryRefDO.taskSceneCategoryRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSceneCategoryRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TaskSceneCategoryRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 任务类别主键 pms_task_scene.id
                qdo.sceneId,
                // 任务类别主键 pms_task_category.id
                qdo.categoryId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSceneCategoryRefVO> getJpaQueryWhere(TaskSceneCategoryRefQuery query) {
        JPAQuery<TaskSceneCategoryRefVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TaskSceneCategoryRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TaskSceneCategoryRefQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 任务类别主键 pms_task_scene.id 精确 */
        if (!ObjectUtils.isEmpty(query.getSceneId())) {
            list.add(qdo.sceneId.eq(query.getSceneId()));
        }
        /** 任务类别主键 pms_task_category.id 精确 */
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TaskSceneCategoryRefVO queryByKey(Long id) {
        JPAQuery<TaskSceneCategoryRefVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskSceneCategoryRefVO> queryListDynamic(TaskSceneCategoryRefQuery query) {
        JPAQuery<TaskSceneCategoryRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TaskSceneCategoryRefVO> queryPaging(TaskSceneCategoryRefQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TaskSceneCategoryRefVO> jpaQuery = getJpaQueryWhere(query);
        List<TaskSceneCategoryRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TaskSceneCategoryRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TaskSceneCategoryRefDO save(TaskSceneCategoryRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TaskSceneCategoryRefDO> saveAll(List<TaskSceneCategoryRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TaskSceneCategoryRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 任务类别主键 pms_task_scene.id
        if (payload.getSceneId() != null) {
            update.set(qdo.sceneId, payload.getSceneId());
        }
        // 任务类别主键 pms_task_category.id
        if (payload.getCategoryId() != null) {
            update.set(qdo.categoryId, payload.getCategoryId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 任务类别主键 pms_task_scene.id
            if (nullFields.contains("sceneId")) {
                update.setNull(qdo.sceneId);
            }
            // 任务类别主键 pms_task_category.id
            if (nullFields.contains("categoryId")) {
                update.setNull(qdo.categoryId);
            }
        }
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        return update.execute();
    }

    public long deleteSoftBySceneId(Long sceneId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.sceneId.eq(sceneId));
        return update.execute();
    }

    public long deleteSoftBySceneIdList(List<Long> sceneIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.sceneId.in(sceneIdList));
        return update.execute();
    }
}

