package com.elitesland.tw.tw5.server.prd.taskpro.repo.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.QTaskCategoryDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.QTaskSceneCategoryRefDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.QTaskSceneDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.entity.TaskSceneDO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.payload.TaskScenePayload;
import com.elitesland.tw.tw5.server.prd.taskpro.model.query.TaskCategoryQuery;
import com.elitesland.tw.tw5.server.prd.taskpro.model.query.TaskSceneQuery;
import com.elitesland.tw.tw5.server.prd.taskpro.model.vo.TaskCategoryVO;
import com.elitesland.tw.tw5.server.prd.taskpro.model.vo.TaskSceneVO;
import com.elitesland.tw.tw5.server.prd.taskpro.repo.TaskSceneRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 任务场景
 *
 * @author duwh
 * @date 2024-01-31
 */
@Repository
@RequiredArgsConstructor
public class TaskSceneDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TaskSceneRepo repo;
    private final QTaskSceneDO qdo = QTaskSceneDO.taskSceneDO;
    private final QTaskCategoryDO qTaskCategoryDO = QTaskCategoryDO.taskCategoryDO;
    private final QTaskSceneCategoryRefDO qTaskSceneCategoryRefDO = QTaskSceneCategoryRefDO.taskSceneCategoryRefDO;

    /**
     * 拼装查询字段
     * ss
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSceneVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TaskSceneVO.class,
                qdo.id,
                qdo.remark,
                qdo.categoryId,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 任务场景名称
                qdo.name,
                // 任务场景备注
                qdo.content,
                // 是否启用
                qdo.enabled,
                // 排序
                qdo.sort,
                // 拓展字段1
                qdo.ext1,
                // 拓展字段2
                qdo.ext2,
                // 拓展字段3
                qdo.ext3,
                // 拓展字段4
                qdo.ext4,
                // 拓展字段5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSceneVO> getJpaQueryWhere(TaskSceneQuery query) {
        JPAQuery<TaskSceneVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TaskSceneQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TaskSceneQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getIdNeq())) {
            list.add(qdo.id.ne(query.getIdNeq()));
        }
        /** 任务场景名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getName())) {
             list.add(qdo.name.like(SqlUtil.toSqlLikeString(query.getName())));
        }
        if (!ObjectUtils.isEmpty(query.getNameEq())) {
            list.add(qdo.name.eq(query.getNameEq()));
        }
        if (!ObjectUtils.isEmpty(query.getCategoryId())) {
            list.add(qdo.categoryId.eq(query.getCategoryId()));
        }
        /** 任务场景备注 精确 */
        if (!ObjectUtils.isEmpty(query.getContent())) {
            list.add(qdo.content.eq(query.getContent()));
        }
        /** 是否启用 精确 */
        if (!ObjectUtils.isEmpty(query.getEnabled())) {
            list.add(qdo.enabled.eq(query.getEnabled()));
        }
        /** 排序 精确 */
        if (!ObjectUtils.isEmpty(query.getSort())) {
            list.add(qdo.sort.eq(query.getSort()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 拓展字段4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 拓展字段5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TaskSceneVO queryByKey(Long id) {
        JPAQuery<TaskSceneVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskSceneVO> queryListDynamic(TaskSceneQuery query) {
        JPAQuery<TaskSceneVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TaskSceneVO> queryPaging(TaskSceneQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<TaskSceneVO> jpaQuery = getJpaQueryWhere(query);
        List<TaskSceneVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TaskSceneVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TaskSceneDO save(TaskSceneDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TaskSceneDO> saveAll(List<TaskSceneDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TaskScenePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 任务场景名称
        if (payload.getName() != null) {
            update.set(qdo.name, payload.getName());
        }
        // 任务场景备注
        if (payload.getContent() != null) {
            update.set(qdo.content, payload.getContent());
        }
        // 是否启用
        if (payload.getEnabled() != null) {
            update.set(qdo.enabled, payload.getEnabled());
        }
        // 排序
        if (payload.getSort() != null) {
            update.set(qdo.sort, payload.getSort());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 拓展字段4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 拓展字段5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 任务场景名称
            if (nullFields.contains("name")) {
                update.setNull(qdo.name);
            }
            // 任务场景备注
            if (nullFields.contains("content")) {
                update.setNull(qdo.content);
            }
            // 是否启用
            if (nullFields.contains("enabled")) {
                update.setNull(qdo.enabled);
            }
            // 排序
            if (nullFields.contains("sort")) {
                update.setNull(qdo.sort);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 拓展字段4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 拓展字段5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        return update.execute();
    }

    public List<TaskCategoryVO> listByRef(TaskCategoryQuery query) {
        return jpaQueryFactory.select(Projections.bean(TaskCategoryVO.class,
            qTaskCategoryDO.id,
            //qTaskCategoryDO.remark,
            //qTaskCategoryDO.createUserId,
            //qTaskCategoryDO.creator,
            //qTaskCategoryDO.createTime,
            //qTaskCategoryDO.modifyUserId,
            //qTaskCategoryDO.updater,
            //qTaskCategoryDO.modifyTime,
            // 任务类别名称
            qTaskCategoryDO.name,
            // 任务类别说明
            qTaskCategoryDO.content,
            // 是否启用
            qTaskCategoryDO.enabled,
            // 排序
            qTaskCategoryDO.sort,
            // 拓展字段1
            qTaskCategoryDO.ext1,
            // 拓展字段2
            qTaskCategoryDO.ext2,
            // 拓展字段3
            qTaskCategoryDO.ext3,
            // 拓展字段4
            qTaskCategoryDO.ext4,
            // 拓展字段5
            qTaskCategoryDO.ext5
        )).from(qTaskCategoryDO)
            .where(qTaskCategoryDO.id.notIn(jpaQueryFactory.select(qTaskSceneCategoryRefDO.categoryId).from(qTaskSceneCategoryRefDO).where(qTaskSceneCategoryRefDO.deleteFlag.eq(0))))
            .fetch()
            ;

    }
}

