package com.elitesland.tw.tw5.server.yeedocref.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.yeedocref.constants.YeedocRefTypeEnum;
import com.elitesland.tw.tw5.server.yeedocref.convert.PrdYeedocRefConvert;
import com.elitesland.tw.tw5.server.yeedocref.dao.repo.PrdYeedocRefRepo;
import com.elitesland.tw.tw5.server.yeedocref.model.entity.PrdYeedocRefDO;
import com.elitesland.tw.tw5.server.yeedocref.model.entity.QPrdYeedocRefDO;
import com.elitesland.tw.tw5.server.yeedocref.model.payload.PrdYeedocRefPayload;
import com.elitesland.tw.tw5.server.yeedocref.model.query.PrdYeedocRefQuery;
import com.elitesland.tw.tw5.server.yeedocref.model.vo.PrdYeedocRefVO;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.Collections;
import java.util.List;


/**
 * 通用易稻壳文件夹关系表dao
 * @author duwh
 * @date 2024/3/22
*/
@RequiredArgsConstructor
@Component
public class PrdYeedocRefDao  {

    private static final QPrdYeedocRefDO qdo = QPrdYeedocRefDO.prdYeedocRefDO;
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdYeedocRefRepo repo;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdYeedocRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdYeedocRefVO.class,
                qdo.id,
                qdo.tenantId,
                qdo.belongOrgId,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                qdo.tenantOrgId,
                qdo.remark,
                qdo.deleteFlag,
                qdo.refType,
                qdo.refId,
                qdo.itemId,
                qdo.libraryId,
                qdo.itemName,
                qdo.extString1,
                qdo.extString3,
                qdo.extString2
        )).from(qdo);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<Long> getJpaQuerySelectCount() {
        return jpaQueryFactory.select(qdo.id.count()).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     */
    private void jpaQueryWhere(JPAQuery<?> jpaQuery, PrdYeedocRefQuery query) {
        if (!ObjectUtils.isEmpty(query.getRemark())) {
            jpaQuery.where(qdo.remark.eq(query.getRemark()));
        }
        if (!ObjectUtils.isEmpty(query.getRefType())) {
            jpaQuery.where(qdo.refType.eq(query.getRefType()));
        }
        if (!ObjectUtils.isEmpty(query.getRefId())) {
            jpaQuery.where(qdo.refId.eq(query.getRefId()));
        }
        if (!ObjectUtils.isEmpty(query.getItemId())) {
            jpaQuery.where(qdo.itemId.eq(query.getItemId()));
        }
        if (!ObjectUtils.isEmpty(query.getLibraryId())) {
            jpaQuery.where(qdo.libraryId.eq(query.getLibraryId()));
        }
        if (!ObjectUtils.isEmpty(query.getItemName())) {
            jpaQuery.where(qdo.itemName.eq(query.getItemName()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString1())) {
            jpaQuery.where(qdo.extString1.eq(query.getExtString1()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString3())) {
            jpaQuery.where(qdo.extString3.eq(query.getExtString3()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString2())) {
            jpaQuery.where(qdo.extString2.eq(query.getExtString2()));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
    }

    /**
     * 分页查询
     * @param query 查询参数
     * @return result
     */
    public PagingVO<PrdYeedocRefVO> queryPage(PrdYeedocRefQuery query) {
        // 查询总行数total
        JPAQuery<Long> jpaQuerySelectCount = getJpaQuerySelectCount();
        jpaQueryWhere(jpaQuerySelectCount, query);
        Long nullAbleCount = jpaQuerySelectCount.fetchOne();
        long count = nullAbleCount == null ? 0 : nullAbleCount;
        // 如果总行数为0则直接返回
        if(count == 0){
            return PagingVO.<PrdYeedocRefVO>builder()
                    .total(count)
                    .records(Collections.emptyList())
                    .build();
        }
        JPAQuery<PrdYeedocRefVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQueryWhere(jpaQuerySelect, query);
        // 动态排序
        jpaQuerySelect.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        // 分页
        SqlUtil.handlePage(jpaQuerySelect, query);
        return PagingVO.<PrdYeedocRefVO>builder()
                .total(count)
                .records(jpaQuerySelect.fetch())
                .build();
    }

    /**
     * 查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdYeedocRefVO> queryList(PrdYeedocRefQuery query) {
        JPAQuery<PrdYeedocRefVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQueryWhere(jpaQuerySelect, query);
        // 动态排序
        jpaQuerySelect.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuerySelect.fetch();
    }

    /**
     * 查询count
     *
     * @param query 查询参数
     * @return count
     */
    public long queryCount(PrdYeedocRefQuery query) {
        JPAQuery<Long> jpaQuerySelectCount = getJpaQuerySelectCount();
        jpaQueryWhere(jpaQuerySelectCount, query);
        Long count = jpaQuerySelectCount.fetchOne();
        return count==null?0:count;
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdYeedocRefVO queryByKey(Long id) {
        JPAQuery<PrdYeedocRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 调用jpa的保存
     *
     * @param payload payload
     * @return 保存后的对象
     */
    public PrdYeedocRefVO save(PrdYeedocRefPayload payload) {
        PrdYeedocRefDO ado = PrdYeedocRefConvert.INSTANCE.toEntity(payload);
        // 创建单据的时候审计字段会被前端传的值影响，强制清空相关字段
        SqlUtil.clearAuditFields(ado);
        repo.save(ado);
        return PrdYeedocRefConvert.INSTANCE.toVO(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdYeedocRefDO> saveAll(List<PrdYeedocRefDO> dos) {
        // 创建单据的时候审计字段会被前端传的值影响，强制清空相关字段
        SqlUtil.clearAuditFields(dos);
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrdYeedocRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        if (payload.getRefType() != null) {
            update.set(qdo.refType, payload.getRefType());
        }
        if (payload.getRefId() != null) {
            update.set(qdo.refId, payload.getRefId());
        }
        if (payload.getItemId() != null) {
            update.set(qdo.itemId, payload.getItemId());
        }
        if (payload.getLibraryId() != null) {
            update.set(qdo.libraryId, payload.getLibraryId());
        }
        if (payload.getItemName() != null) {
            update.set(qdo.itemName, payload.getItemName());
        }
        if (payload.getExtString1() != null) {
            update.set(qdo.extString1, payload.getExtString1());
        }
        if (payload.getExtString3() != null) {
            update.set(qdo.extString3, payload.getExtString3());
        }
        if (payload.getExtString2() != null) {
            update.set(qdo.extString2, payload.getExtString2());
        }
        // 处理要设置成空的字段(nullField处理)
        SqlUtil.handleNullFieldsUpdate(payload,qdo,update);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 按引用类型和引用id查询
     *
     * @param yeedocRefTypeEnum yeedoc-ref类型枚举
     * @param projectId         项目id
     * @return {@link PrdYeedocRefVO}
     */
    public PrdYeedocRefVO queryByRefTypeAndRefId(YeedocRefTypeEnum yeedocRefTypeEnum, Long projectId) {
        JPAQuery<PrdYeedocRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.refId.eq(projectId));
        jpaQuery.where(qdo.refType.eq(yeedocRefTypeEnum.getCode()));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}
