package com.elitesland.tw.tw5.server.common;

import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.util.BeanUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.apache.poi.xssf.usermodel.XSSFDataValidationConstraint;
import org.apache.poi.xssf.usermodel.XSSFDataValidationHelper;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.List;

@Component
@RequiredArgsConstructor
@Slf4j
public class ExcelUtil {

    /**
     * 将文件模板写入响应中
     *
     * @param response 响应
     * @param fileName 文件名
     * @param workbook 模板数据
     */
    public static void writeResponse(HttpServletResponse response, String fileName, Workbook workbook) {
        try {
            ServletOutputStream servletOutputStream = response.getOutputStream();
            response.setContentType("application/vnd.ms-excel");
            fileName = URLEncoder.encode(fileName, "utf-8");
            response.addHeader("Content-Disposition", "attachment;filename=\""
                    + fileName + ".xls\"");
            BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(servletOutputStream);
            workbook.write(servletOutputStream);
            bufferedOutputStream.flush();
            servletOutputStream.close();
            bufferedOutputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage());
        }
    }

    //该方法将excell的单元格值转化为字符串类型
    public static String getCellFormatValue(Cell cell) {
        if (cell == null) {
            return "";
        }
        CellType cellType = cell.getCellTypeEnum();
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        StringBuffer cellValue = new StringBuffer();
        switch (cellType) {
            case STRING://文本
                cellValue = new StringBuffer(cell.getStringCellValue());
                break;
            case NUMERIC://数字，日期
                if (DateUtil.isCellDateFormatted(cell)) {
                    cellValue = new StringBuffer(
                            df.format(cell.getDateCellValue()));

                } else {
                    cellValue = new StringBuffer(NumberToTextConverter.toText(cell.getNumericCellValue()));
                    // cellValue = new StringBuffer(BigDecimal.valueOf(cell.getNumericCellValue()).toPlainString());
//                    cellValue = new StringBuffer(cell.getNumericCellValue() + "");
                }
                break;
            case BLANK://空白
            case ERROR://错误
                cellValue = new StringBuffer();
                break;
            case BOOLEAN://布尔
                cellValue = new StringBuffer(String.valueOf(cell.getBooleanCellValue()));
                break;
            case FORMULA://公式
                //公式里面是计算日期格式的，先设置成数字格式，走正常获取数字格式内容方法
                //如果公式结算结果不是数字就会报错，try catch为String格式获取内容
                //没有考虑有公式的单元格最后结果是其他数据类型的情况，原理都是一样
                cell.setCellType(CellType.STRING);
                cellValue = new StringBuffer(cell.getStringCellValue());
                if ("#N/A".equals(cellValue.toString())) {
                    cellValue = new StringBuffer("");
                }
                break;
            default:
                break;
        }
        return cellValue.toString().trim();
    }

    public Cell setCellValue(Row row, int column, Object value) {
        Cell cell = row.createCell(column);

        if (value instanceof BigDecimal) {
            cell.setCellValue(((BigDecimal) value).doubleValue());
//            cell.setCellType(CellType.NUMERIC);
        } else {
            cell.setCellValue(value == null ? "" : value.toString());
        }
        return cell;
    }

    public Cell setCellValueAndStyle(Row row, CellStyle style, int column, Object value) {
        Cell cell = row.createCell(column);
        if (value instanceof BigDecimal) {
            cell.setCellValue(((BigDecimal) value).doubleValue());
        } else {
            cell.setCellValue(value == null ? "" : value.toString());
        }
        cell.setCellStyle(style);
        return cell;
    }

    /**
     * 不影响原来格子的格式，避免直接创建带来的问题
     *
     * @param row
     * @param column
     * @param value
     * @return
     */
    public Cell setCellValueNew(Row row, int column, Object value) {
        Cell cell = row.getCell(column);
        if (cell == null) {
            cell = row.createCell(column);
        }
        if (value instanceof BigDecimal) {
            cell.setCellValue(((BigDecimal) value).doubleValue());
        } else {
            cell.setCellValue(value == null ? "" : value.toString());
        }
        return cell;
    }


    public static final String formulasReplace = ":rowNo";

    public String replaceFormulasRowNo(String formulas, int rowNo) {
        return formulas.replaceAll(formulasReplace, rowNo + "");
    }

    public Cell setCellFormulas(Row row, int column, String formulas) {

        Cell cell = row.createCell(column);
        cell.setCellFormula(replaceFormulasRowNo(formulas, row.getRowNum() + 1));
        return cell;
    }

    /**
     * 对一列数据进行VLOOKUP
     *
     * @param sheet
     * @param column
     * @param formulas
     * @return
     */
    public void setColumnFormulas(XSSFSheet sheet, int column, String formulas) {
        for (int i = 1; i < 1000; i++) {
            Row row = sheet.getRow(i);
            if (row == null) {
                row = sheet.createRow(i);
            }
            Cell cell = row.createCell(column);
            cell.setCellFormula(replaceFormulasRowNo(formulas, row.getRowNum() + 1));
        }
    }


    /**
     * 构造数据有效性约束
     *
     * @param sheet              sheet名称
     * @param column             列序号
     * @param startRow           开始行
     * @param validDataSheetName 有效数据sheet名称
     * @param validStartRow      有效数据开始行 从1开始，而不是从0
     * @param validColStr        有效数据列名
     */
    public void generateRangeList(XSSFSheet sheet, int column, int startRow, String validDataSheetName,
                                  int validStartRow, String validColStr) {
        String strFormula = validDataSheetName + "!$" + validColStr + "$" + validStartRow + ":$" + validColStr + "$65535";
        XSSFDataValidationConstraint constraint = new XSSFDataValidationConstraint(DataValidationConstraint.ValidationType.LIST, strFormula);
        // 设置数据有效性加载在哪个单元格上,四个参数分别是：起始行、终止行、起始列、终止列
        CellRangeAddressList regions = new CellRangeAddressList(startRow, 65535, column, column);
        // 数据有效性对象
        DataValidationHelper help = new XSSFDataValidationHelper(sheet);
        DataValidation validation = help.createValidation(constraint, regions);
        sheet.addValidationData(validation);
    }

    /**
     * 构造数据有效性约束
     *
     * @param sheet              sheet名称
     * @param column             列序号
     * @param startRow           开始行
     * @param validDataSheetName 有效数据sheet名称
     * @param validStartRow      有效数据开始行 从1开始，而不是从0
     * @param validColStr        有效数据列名
     */
    public void generateRangeMultipleChoice(XSSFSheet sheet, int column, int startRow, String validDataSheetName,
                                            int validStartRow, String validColStr) {
        String strFormula = validDataSheetName + "!$" + validColStr + "$" + validStartRow + ":$" + validColStr + "$65535";
        // 数据有效性对象
        DataValidationHelper dvHelper = new XSSFDataValidationHelper(sheet);

        String[] strs = {"A", "B", "C"};
        XSSFDataValidationConstraint dvConstraint = (XSSFDataValidationConstraint) dvHelper.createExplicitListConstraint(strs);//strs下拉菜单的数据数组例如:String[]

        XSSFDataValidationConstraint constraint = new XSSFDataValidationConstraint(DataValidationConstraint.ValidationType.LIST, strFormula);
        // 设置数据有效性加载在哪个单元格上,四个参数分别是：起始行、终止行、起始列、终止列
        CellRangeAddressList regions = new CellRangeAddressList(startRow, 65535, column, column);

//        DataValidation validation = help.createValidation(constraint, regions);
//        sheet.addValidationData(validation);
//        addressList = new CellRangeAddressList(1,endRow, i, i);// 第一个参数是开始行，第二个是结束行，第三个是开始列，第四个是结束列

        DataValidation validation = dvHelper.createValidation(dvConstraint, regions);

        //设置只能选下拉菜单里的值不能随便输入
        validation.setSuppressDropDownArrow(true);

        validation.setShowErrorBox(true);
        sheet.addValidationData(validation);
    }


    /**
     * 往lov中添加选择项数据
     *
     * @param lovSheet
     * @param data
     * @param columnNum
     */
    public void insertLOVdata(Sheet lovSheet, List<PrdSystemSelectionVO> data, int columnNum) {
        if (!CollectionUtils.isEmpty(data) && lovSheet != null) {
            int nextRow = 1;
            for (PrdSystemSelectionVO selection : data) {
                String selectionValue = selection.getSelectionValue();
                String selectionName = selection.getSelectionName();
                if (StringUtils.hasText(selectionName) && selectionValue != null) {
                    //判断当前row是否存在
                    Row row = lovSheet.getRow(nextRow);
                    if (row == null) {
                        row = lovSheet.createRow(nextRow);
                    }
                    setCellValue(row, columnNum, selectionName);
                    setCellValue(row, columnNum + 1, selectionValue + "");
                    nextRow++;
                }
            }
        }
    }

    /**
     * 往lov中添加普通下拉列表数据
     *
     * @param listSheet 目标sheet
     * @param data      数据列表
     * @param columnNum 值的属性名
     * @param columnNum 名称的属性名
     * @param columnNum 插入列
     */
    public void insertListData(Sheet listSheet, List<? extends Object> data, String valueAttributeName, String nameAttributeName, int columnNum) {
        if (!CollectionUtils.isEmpty(data) && listSheet != null) {
            int nextRow = 1;
            for (Object selection : data) {
                Object selectionValue = BeanUtil.getAllFieldValueByFieldName(valueAttributeName, selection);
                Object selectionName = BeanUtil.getAllFieldValueByFieldName(nameAttributeName, selection);
                if (selectionName != null && selectionValue != null) {
                    //判断当前row是否存在
                    Row row = listSheet.getRow(nextRow);
                    if (row == null) {
                        row = listSheet.createRow(nextRow);
                    }
                    setCellValue(row, columnNum, selectionName);
                    setCellValue(row, columnNum + 1, selectionValue + "");
                    nextRow++;
                }
            }
        }
    }


    /**
     * 将workbook转为file
     *
     * @param wb
     * @param name
     * @return
     */
    public static File workbookToFile(Workbook wb, String name) {
        File toFile = null;
        try {
            toFile = File.createTempFile(name, ".xlsx");
            OutputStream os = new FileOutputStream(toFile);
            wb.write(os);
            os.close();
            return toFile;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return toFile;
    }


}
