package com.elitesland.tw.tw5.server.common.permission.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.zhxu.bs.FieldOps;
import cn.zhxu.bs.util.MapBuilder;
import cn.zhxu.bs.util.MapUtils;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgOrganizationService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgRoleVO;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemBusinessObjectConfigService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemNewFunctionService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemPermissonRuleService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRoleFunctionService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionFieldObjRoleFunctionVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionRuleVO;
import com.elitesland.tw.tw5.server.common.permission.enums.*;
import com.elitesland.tw.tw5.server.common.permission.service.PermissionRuleService;
import com.elitesland.tw.tw5.server.common.permission.strategy.context.BusinessObjectTypeStrategyContext;
import com.elitesland.tw.tw5.server.common.permission.strategy.context.OrgUserStrategyContext;
import com.elitesland.tw.tw5.server.common.permission.strategy.context.RoleStrategyContext;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

import static com.elitesland.tw.tw5.server.common.permission.contants.PermissionContants.*;

/**
 * 权限规则
 *
 * @author : JS
 * @date 2023/09/25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PermissionRuleServiceImpl implements PermissionRuleService {

    private final CacheUtil cacheUtil;

    private final PrdSystemPermissonRuleService permissionRuleService;

    private final PrdOrgOrganizationService prdOrgOrganizationService;

    private final PrdSystemRoleFunctionService prdSystemRoleFunctionService;

    private final PrdSystemNewFunctionService prdSystemNewFunctionService;

    private final PrdSystemBusinessObjectConfigService prdSystemBusinessObjectConfigService;

    private final OrgUserStrategyContext orgUserStrategyContext;

    private final BusinessObjectTypeStrategyContext businessObjectTypeStrategyContext;

    private final RoleStrategyContext roleStrategyContext;

    private final Map<String, Object> map = new HashMap<>();

    @Override
    public MapBuilder searcherHandle(Class<?> classBean, Map<String, Object> params, String permissionDomainCode) {
        String functionCode = (String) params.get(FUNCTION_CODE);
        MapBuilder mapBuilder = MapUtils.builder(params);

        // 清掉缓存
        map.clear();

        Long loginUserId = GlobalUtil.getLoginUserId();
        // 获取人员登录信息
        List<String> systemRoleCodes = cacheUtil.getSystemRoleCodes(loginUserId);

        // 获取权限配置信息
        List<PrdSystemPermissionRuleVO> rules = this.getRules(functionCode);

        // 拿到用户组织角色信息
        List<PrdOrgRoleVO> prdOrgRoleVOS = prdOrgOrganizationService.queryUserOrgRoles(loginUserId);
        Set<String> systemOrgRoleCodes = new HashSet<>();
        Set<Long> systemOrgRoleIds = new HashSet<>();
        prdOrgRoleVOS.forEach(prdOrgRoleVO -> {
            systemOrgRoleCodes.add(prdOrgRoleVO.getRoleCode());
            systemOrgRoleIds.add(prdOrgRoleVO.getId());
        });

        // 拼接数据权限规则 （特殊条件）
        List<String> ruleGroupSpecialExprList = new ArrayList<>();

        // 处理团队成员特殊情况 如果业务对象拥有团队成员面板添加条件
        String[] codeRegx = functionCode.split(REGX);

        log.info("功能编码 functionCode {} ： 拆分后未 【{}】", functionCode, codeRegx);

//        Long objectId = cacheUtil.getObjectIdByFunctionCode(codeRegx[codeRegx.length - 1]);
//        // 查询该业务对象是否拥有团队成员面板
//        String isMember = prdSystemBusinessObjectConfigService.getIsMemberByObjectId(objectId);
//        if (BusinessObjectConfigEnum.MEMBER.getName().equals(isMember)) {
//            mapBuilder.group(MEMBER_GROUP)
//                    .field(MEMBER_FIELD)
//                    .sql("exists (select ppm.user_id from prj_project_member ppm where ppm.project_id = $1 and ppm.user_id = ? and ppm.delete_flag = 0)", loginUserId);
//            ruleGroupSpecialExprList.add(MEMBER_GROUP);
//        }
        // 商机定制团队成员权限
        if(FunctionCodeEnum.BUSINESS_PAGE.name().equals(functionCode)){
            mapBuilder.group(MEMBER_GROUP)
                    .field(MEMBER_FIELD)
                    .sql("exists (select ppm.user_id from prj_project_member ppm where ppm.project_id = $1 and ppm.user_id = ? and ppm.delete_flag = 0)", loginUserId);
            ruleGroupSpecialExprList.add(MEMBER_GROUP);
        }
        // 创建分组表达式
        Object bsGroupExpr = params.get(MapBuilder.GROUP_EXPR);
        StringBuilder groupExpr = new StringBuilder();

        if (!ObjectUtils.isEmpty(bsGroupExpr)) {
            groupExpr.append(bsGroupExpr);
        }
        ruleGroupSpecialExprList.forEach(expr -> {
            if (groupExpr.isEmpty()) {
                groupExpr.append(expr);
            }else {
                groupExpr.append("|(").append(expr).append(")");
            }
        });

        // 判断当前功能有权限，如果没有 并且可以做/拆分使用其默认权限
        if (CollectionUtils.isEmpty(rules)) {
            // 如果可以拆分，则使用默认权限
            if (codeRegx.length > 1) {
                // 拆分后， 分情况 1. 我拥有该功能， 并且该功能有权限规则（使用该规则）； 2. 我拥有该功能， 并且该功能没有权限规则；（直接原来是啥显示啥） 3. 我没有该功能 （直接返回就行）
                Long roleFunctionId = prdSystemRoleFunctionService.listByFunctionCodeAndRoleCodes(functionCode, systemRoleCodes);
                if (ObjectUtils.isEmpty(roleFunctionId)) {
                    mapBuilder.groupExpr(groupExpr.toString());
                    return mapBuilder;
                }
                rules = this.getRules(codeRegx[codeRegx.length - 1]);
                if (CollectionUtils.isEmpty(rules)) return mapBuilder;
            }else {
                // 如果这里要返回了就将规则表达式写入
                mapBuilder.groupExpr(groupExpr.toString());
                return mapBuilder;
            }
        }

        // 拼接数据权限规则 (配置条件)
        List<String> ruleGroupExprList = new ArrayList<>();
        this.parsePermissionRule(functionCode, mapBuilder, loginUserId, systemRoleCodes, systemOrgRoleIds, systemOrgRoleCodes, rules, ruleGroupExprList, classBean);

        // 发现该用户不需要权限直接放行
        if (permissionHasAdmin() || cacheUtil.hasSystemRolePermission(RoleEnum.SYS.getCode())) return mapBuilder;

        // 所有规则用或组合
        String permissionGroupExpr = ruleGroupExprList.stream().map(expr -> "(" + expr + ")").collect(Collectors.joining("|"));

        if (!groupExpr.isEmpty()) {
            groupExpr.append("|(").append(permissionGroupExpr).append(")");
        }else {
            groupExpr.append(permissionGroupExpr);
        }
        mapBuilder.groupExpr(groupExpr.toString());

        // 字段权限 ，过滤不可见字段
        List<String> permissionFieldRule = getPermissionFieldRule(permissionDomainCode, classBean, systemRoleCodes, functionCode);
        if (!CollectionUtils.isEmpty(permissionFieldRule)) {
            mapBuilder.onlySelect(permissionFieldRule.toArray(new String[0]));
        }

        // 设置权限校验标识，防止重复进入权限控制
        mapBuilder.put(PERMISSION_CHECK, true);

        return mapBuilder;
    }

    /**
     * 判断是否不需要进行权限处理
     *
     * @return
     */
    private boolean permissionHasAdmin() {
        return !ObjectUtils.isEmpty(map.get(ADMIN)) && (boolean) map.get(ADMIN);
    }

    private void parsePermissionRule(String functionCode,
                                     MapBuilder mapBuilder,
                                     Long loginUserId,
                                     List<String> systemRoleCodes,
                                     Set<Long> systemOrgRoleIds,
                                     Set<String> systemOrgRoleCodes,
                                     List<PrdSystemPermissionRuleVO> rules,
                                     List<String> ruleGroupExprList,
                                     Class<?> classBean) {
        // 获取部门信息
        Long defaultOrgId = cacheUtil.getDefaultOrgIdByUserId(loginUserId);
        List<Long> childOrgIds = cacheUtil.getAllChildOrgIds(defaultOrgId);
        childOrgIds.add(defaultOrgId);
        Set<Long> childOrgIdSet = new HashSet<>(childOrgIds);

        for (PrdSystemPermissionRuleVO rule : rules) {

            // 先判断当前登录人是否符合规则范围，不符合拼接id=-1，用来屏蔽数据，所有条件都不满足，则无法查询到数据
            if (!this.checkRuleScope(rule, mapBuilder, ruleGroupExprList, loginUserId, defaultOrgId, childOrgIdSet, systemRoleCodes, systemOrgRoleIds, systemOrgRoleCodes)) {

                // 走到这里说明不匹配规则
                log.info("当前登录人不匹配规则范围【{}】，规则值【{}】，规则编码【{}】，功能代码【{}】，当前登录人【{}】", rule.getRuleType(), rule.getRuleScope(), rule.getRuleCode(), functionCode, loginUserId);
                mapBuilder.group(rule.getRuleCode())
                        .field("id", -1).op(FieldOps.Equal);
                ruleGroupExprList.add(rule.getRuleCode());
                continue;
            }

            log.info("验证该用户是否直接放行，结果为【{}】", permissionHasAdmin());

            // 发现不需要进行权限控制放行
            if (permissionHasAdmin()) {
                break;
            };

            log.info("当前规则【{}】已生效，功能代码【{}】", rule.getRuleCode(), functionCode);

            // 之后如果无条件直接跳过
            if (PermissionScopeTypeEnum.SCOPE_ALL.getName().equals(rule.getScopeType())) {

                log.info("当前规则无规则条件， 进行无规则情况下处理");

                // 如果发现是业务对象类型结束的时候 要把编码加进去
                if (!PermissionRuleType.BUSINESS_OBJECT_TYPE.getName().equals(rule.getRuleType())) {
                    mapBuilder.group(rule.getRuleCode());
                    mapBuilder.field("id").sql("1 = 1");
                }
                ruleGroupExprList.add(rule.getRuleCode());
                continue;
            }

            rule.getDetailList().forEach(detail -> {
                String field = detail.getRuleField();
                String value = detail.getRuleValue();

                // 自定义类型采用sql拼接
                if (PermissionRuleType.CUSTOM.getName().equals(rule.getRuleType())) {

                    log.info("当前规则属于自定义类型， 进行sql自定义处理");

                    if (!StringUtils.hasText(detail.getRuleSql())) {
                        return;
                    }

                    mapBuilder.group(rule.getRuleCode() + "_" + detail.getRuleDetailCode());
                    if (StringUtils.hasText(value)) {
                        mapBuilder.field("id").sql(detail.getRuleSql(), value.split(","));
                    } else {
                        mapBuilder.field("id").sql(detail.getRuleSql());
                    }
                } else {
                    // 其他类型按照配置 拼接条件
                    mapBuilder.group(rule.getRuleCode() + "_" + detail.getRuleDetailCode());
                    // 条件筛选 则 直接拼接条件
                    if (PermissionRuleConditionType.FILTER.name().equals(detail.getRuleConditionType())) {
                        if (StringUtils.hasText(value)) {
                            String[] split = value.split(",");

                            log.info("[{}] 当前规则中范围存在值信息", detail);

                            if (!ObjectUtils.isEmpty(detail.getDeep()) && detail.getDeep().equals(1)) {
                                Set<Long> orgAlls = new HashSet<>();
                                for (String s : split) {

                                    log.info("[{}] 获取缓存中的子组织", s);

                                    Long orgId = Long.valueOf(s);
                                    List<Long> allChildOrgIds = cacheUtil.getAllChildOrgIds(orgId);
                                    allChildOrgIds.add(orgId);
                                    if (!CollectionUtils.isEmpty(allChildOrgIds)) {
                                        orgAlls.addAll(allChildOrgIds);
                                    }

                                    log.info("[{}] 获取当前组织的信息成功", s);
                                }

                                log.info("[{}] [{}] 已经获取到所有穿透组织信息", rule.getRuleCode(), detail.getRuleDetailCode());
                                mapBuilder.field(CharSequenceUtil.toCamelCase(field), orgAlls).op(detail.getRuleCondition());
                            } else {
                                mapBuilder.field(CharSequenceUtil.toCamelCase(field), split).op(detail.getRuleCondition());
                            }
                        } else {
                            mapBuilder.field(CharSequenceUtil.toCamelCase(field), value).op(detail.getRuleCondition());
                        }
                    }
                }

            });

            // 构建新的分组表达式 分组为 规则编码_规则明细编码
            String ruleGroupExpr = this.buildRuleGroupExpr(rule.getGroupExpr(), rule.getRuleCode());
            if (PermissionRuleType.BUSINESS_OBJECT_TYPE.getName().equals(rule.getRuleType())) {
                ruleGroupExpr =  rule.getRuleCode() + "&" + ruleGroupExpr;
            }
            ruleGroupExprList.add(ruleGroupExpr);
        };

    }

    /**
     * 校验规则范围
     */
    private boolean checkRuleScope(PrdSystemPermissionRuleVO rule, MapBuilder mapBuilder, List<String> ruleGroupExprList, Long loginUserId, Long defaultOrgId, Set<Long> childOrgIdSet, List<String> systemRoleCodes, Set<Long> systemOrgRoleIds, Set<String> systemOrgRoleCodes) {
        return switch (PermissionRuleType.valueOf(rule.getRuleType())) {
            case USER -> rule.getRuleScope().equals(loginUserId.toString());
            case ORG -> orgUserStrategyContext.getStrategyBean(PermissionRuleType.ORG.getName() + "_" + rule.getLevelScope()).execute(rule, defaultOrgId, childOrgIdSet);
            case ROLE -> roleStrategyContext.getStrategyBean(PermissionRuleType.ROLE.getName() + "_" + rule.getRuleSubType(), PermissionRuleType.ROLE.getName() + "_" + rule.getLevelScope()).execute(rule, systemRoleCodes, systemOrgRoleIds, systemOrgRoleCodes);
            case BUSINESS_OBJECT_TYPE -> businessObjectTypeStrategyContext.getStrategyBean(PermissionRuleType.BUSINESS_OBJECT_TYPE.getName() + "_" + rule.getLevelScope()).execute(rule, mapBuilder, ruleGroupExprList, loginUserId, defaultOrgId, childOrgIdSet, map);
            case CUSTOM -> true;
            default -> false;
        };
    }

    /**
     * 获取权限规则
     */
    private List<PrdSystemPermissionRuleVO> getRules(String functionCode) {

        List<PrdSystemPermissionRuleVO> allPermissionRule = permissionRuleService.getAllByFunctionCode(functionCode);
        if (allPermissionRule.isEmpty()) {
            return Collections.emptyList();
        }
        return allPermissionRule;
    }

    /**
     * 获取字段权限
     */
    private List<String> getPermissionFieldRule(String permissionDomainCode, Class<?> classBean, List<String> systemRoleCodes, String functionCode) {
        List<PrdSystemPermissionFieldObjRoleFunctionVO> permissionFieldRule = permissionRuleService.getPermissionFieldRule(permissionDomainCode, classBean.getName(), functionCode);

        // 过滤不可见字段
        return permissionFieldRule.stream()
                .filter(item -> systemRoleCodes.contains(item.getRoleCode()) && 0 == item.getIsVisible())
                .map(PrdSystemPermissionFieldObjRoleFunctionVO::getFieldName)
                .distinct().toList();
    }

    /**
     * 将规则表达式拼接上分组编码
     *
     * @param groupExpr 原规则表达式
     * @param ruleCode  规则编码
     * @return 拼接后的表达式
     */
    private String buildRuleGroupExpr(String groupExpr, String ruleCode) {
        int left = 0;
        int right = 0;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < groupExpr.length(); i++) {
            if (groupExpr.charAt(i) != '&' && groupExpr.charAt(i) != '|') {
                right++;
            } else {
                sb.append(ruleCode).append("_").append(groupExpr, left, right).append(groupExpr.charAt(i));
                right++;
                left = right;
            }
        }
        if (left != right) {
            sb.append(ruleCode).append("_").append(groupExpr, left, right);
        }
        return sb.toString();
    }


}
