package com.elitesland.tw.tw5.server.prd.personplan.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.personplan.query.PmsProjectRoleAssignmentQuery;
import com.elitesland.tw.tw5.api.prd.personplan.vo.PmsProjectRoleAssignmentVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeDO;
import com.elitesland.tw.tw5.server.prd.personplan.entity.PmsProjectRoleAssignmentDO;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QPmsProjectRoleAssignmentDO;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QPmsProjectRoleDO;
import com.elitesland.tw.tw5.server.prd.personplan.repo.PmsProjectRoleAssignmentRepo;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目角色分配
 *
 * @author xxb
 * @date 2023-07-07
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectRoleAssignmentDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectRoleAssignmentRepo repo;
    private final QPmsProjectRoleAssignmentDO qdo = QPmsProjectRoleAssignmentDO.pmsProjectRoleAssignmentDO;

    private final QPmsProjectDO projectqdo = QPmsProjectDO.pmsProjectDO;

    private final QPrdOrgEmployeeDO employeeqdo = new QPrdOrgEmployeeDO("prdOrgEmployeeDO");

    private final QPmsProjectRoleDO roleqdo = QPmsProjectRoleDO.pmsProjectRoleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectRoleAssignmentVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectRoleAssignmentVO.class,
                        qdo.id,
                        //qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        //qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 角色主键
                        qdo.projectRoleId,
                        // 角色名称
                        roleqdo.roleName,
                        // 项目主键
                        qdo.projectId,
                        // 项目名称
                        projectqdo.projName.as("projectName") ,
                        // 项目编号
                        projectqdo.projNo.as("projectCode") ,
                        // 员工ID
                        qdo.userId,
                        // 员工编号
                        employeeqdo.employeeNo,
                        // 员工名称
                        employeeqdo.employeeName
                )).from(qdo).leftJoin(projectqdo).on(qdo.projectId.eq(projectqdo.id))
                .leftJoin(employeeqdo).on(qdo.userId.eq(employeeqdo.userId))
                .leftJoin(roleqdo).on(qdo.projectRoleId.eq(roleqdo.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectRoleAssignmentVO> getJpaQueryWhere(PmsProjectRoleAssignmentQuery query) {
        JPAQuery<PmsProjectRoleAssignmentVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectRoleAssignmentQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(projectqdo).on(qdo.projectId.eq(projectqdo.id))
                .leftJoin(employeeqdo).on(qdo.userId.eq(employeeqdo.userId));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectRoleAssignmentQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目角色主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectRoleId())) {
            list.add(qdo.projectRoleId.eq(query.getProjectRoleId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 项目名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjectName())) {
            list.add(projectqdo.projName.like(SqlUtil.toSqlLikeString(query.getProjectName())));
        }
        /** 项目编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjectCode())) {
            list.add(projectqdo.projNo.like(SqlUtil.toSqlLikeString(query.getProjectCode())));
        }
        /** 项目名称、编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjectKeyWord())) {
            list.add(projectqdo.projName.like(SqlUtil.toSqlLikeString(query.getProjectKeyWord()))
                    .or(projectqdo.projNo.like(SqlUtil.toSqlLikeString(query.getProjectKeyWord()))));
        }
        /** 员工ID 精确 */
        if (!ObjectUtils.isEmpty(query.getUserId())) {
            list.add(qdo.userId.eq(query.getUserId()));
        }
        if (!ObjectUtils.isEmpty(query.getUserIds())) {
            list.add(qdo.userId.in(query.getUserIds()));
        }
        /** 员工编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getEmployeeNo())) {
            list.add(employeeqdo.employeeNo.like(SqlUtil.toSqlLikeString(query.getEmployeeNo())));
        }
        /** 员工名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getEmployeeName())) {
            list.add(employeeqdo.employeeName.like(SqlUtil.toSqlLikeString(query.getEmployeeName())));
        }
        /** 员工名称、编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getEmployeeKeyWord())) {
            list.add(employeeqdo.employeeNo.like(SqlUtil.toSqlLikeString(query.getEmployeeKeyWord()))
                    .or(employeeqdo.employeeName.like(SqlUtil.toSqlLikeString(query.getEmployeeKeyWord()))));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectRoleAssignmentVO queryByKey(Long id) {
        JPAQuery<PmsProjectRoleAssignmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据 项目id 和 用户资源 查询
     *
     * @param projectId
     * @param userId
     * @return
     */
    public List<PmsProjectRoleAssignmentVO> queryByProjectAndUser(Long projectId, Long userId) {
        JPAQuery<PmsProjectRoleAssignmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.userId.eq(userId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectRoleAssignmentVO> queryListDynamic(PmsProjectRoleAssignmentQuery query) {
        JPAQuery<PmsProjectRoleAssignmentVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectRoleAssignmentVO> queryPaging(PmsProjectRoleAssignmentQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectRoleAssignmentVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectRoleAssignmentVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectRoleAssignmentVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectRoleAssignmentDO save(PmsProjectRoleAssignmentDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectRoleAssignmentDO> saveAll(List<PmsProjectRoleAssignmentDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 根据角色id 删除
     *
     * @param roleIds
     */
    public long deleteByRoleIds(List<Long> roleIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.projectRoleId.in(roleIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据 项目id 和 用户资源 删除
     * @param projectId
     * @param userId
     * @return
     */
    public long deleteByProjectAndUser(Long projectId, Long userId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.projectId.eq(projectId).and(qdo.userId.eq(userId)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

