package com.elitesland.tw.tw5.server.prd.pms.budget.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.pms.budget.payload.PmsWbsBudgetVersionPayload;
import com.elitesland.tw.tw5.api.prd.pms.budget.query.PmsWbsBudgetVersionQuery;
import com.elitesland.tw.tw5.api.prd.pms.budget.vo.PmsWbsBudgetVersionVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.budget.convert.PmsWbsBudgetVersionConvert;
import com.elitesland.tw.tw5.server.prd.pms.budget.dao.repo.PmsWbsBudgetVersionRepo;
import com.elitesland.tw.tw5.server.prd.pms.budget.model.entity.PmsWbsBudgetVersionDO;
import com.elitesland.tw.tw5.server.prd.pms.budget.model.entity.QPmsWbsBudgetVersionDO;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.Collections;
import java.util.List;


/**
 * 预算版本表dao
 *
 * @author duwh
 * @date 2024/3/7
 */
@RequiredArgsConstructor
@Component
public class PmsWbsBudgetVersionDao {

    private static final QPmsWbsBudgetVersionDO qdo = QPmsWbsBudgetVersionDO.pmsWbsBudgetVersionDO;
    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsBudgetVersionRepo repo;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsBudgetVersionVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsBudgetVersionVO.class,
            qdo.id,
            qdo.tenantId,
            qdo.belongOrgId,
            qdo.createUserId,
            qdo.creator,
            qdo.createTime,
            qdo.modifyUserId,
            qdo.updater,
            qdo.modifyTime,
            qdo.tenantOrgId,
            qdo.remark,
            qdo.deleteFlag,
            qdo.proId,
            qdo.versionNo,
            qdo.stepNum,
            qdo.objectStatus
        )).from(qdo);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<Long> getJpaQuerySelectCount() {
        return jpaQueryFactory.select(qdo.id.count()).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     */
    private void jpaQueryWhere(JPAQuery<?> jpaQuery, PmsWbsBudgetVersionQuery query) {
        if (!ObjectUtils.isEmpty(query.getRemark())) {
            jpaQuery.where(qdo.remark.eq(query.getRemark()));
        }
        if (!ObjectUtils.isEmpty(query.getProId())) {
            jpaQuery.where(qdo.proId.eq(query.getProId()));
        }
        if (!ObjectUtils.isEmpty(query.getVersionNo())) {
            jpaQuery.where(qdo.versionNo.eq(query.getVersionNo()));
        }
        if (!ObjectUtils.isEmpty(query.getStepNum())) {
            jpaQuery.where(qdo.stepNum.eq(query.getStepNum()));
        }
        if (!ObjectUtils.isEmpty(query.getObjectStatus())) {
            jpaQuery.where(qdo.objectStatus.eq(query.getObjectStatus()));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return result
     */
    public PagingVO<PmsWbsBudgetVersionVO> queryPage(PmsWbsBudgetVersionQuery query) {
        // 查询总行数total
        JPAQuery<Long> jpaQuerySelectCount = getJpaQuerySelectCount();
        jpaQueryWhere(jpaQuerySelectCount, query);
        Long nullAbleCount = jpaQuerySelectCount.fetchOne();
        long count = nullAbleCount == null ? 0 : nullAbleCount;
        // 如果总行数为0则直接返回
        if (count == 0) {
            return PagingVO.<PmsWbsBudgetVersionVO>builder()
                .total(count)
                .records(Collections.emptyList())
                .build();
        }
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQueryWhere(jpaQuerySelect, query);
        // 动态排序
        jpaQuerySelect.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        SqlUtil.handlePage(jpaQuerySelect, query);
        jpaQuerySelect.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize());
        return PagingVO.<PmsWbsBudgetVersionVO>builder()
            .total(count)
            .records(jpaQuerySelect.fetch())
            .build();
    }

    /**
     * 查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsBudgetVersionVO> queryList(PmsWbsBudgetVersionQuery query) {
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQueryWhere(jpaQuerySelect, query);
        // 动态排序
        jpaQuerySelect.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuerySelect.fetch();
    }

    /**
     * 查询count
     *
     * @param query 查询参数
     * @return count
     */
    public long queryCount(PmsWbsBudgetVersionQuery query) {
        JPAQuery<Long> jpaQuerySelectCount = getJpaQuerySelectCount();
        jpaQueryWhere(jpaQuerySelectCount, query);
        Long count = jpaQuerySelectCount.fetchOne();
        return count == null ? 0 : count;
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsBudgetVersionVO queryByKey(Long id) {
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 按项目id和版本号查询
     *
     * @param projectId 项目id
     * @param versionNo 版本号
     * @return {@link PmsWbsBudgetVersionVO}
     */
    public PmsWbsBudgetVersionVO queryByProjectIdAndVersionNo(Long projectId, int versionNo) {
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.proId.eq(projectId));
        jpaQuery.where(qdo.versionNo.eq(versionNo));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 调用jpa的保存
     *
     * @param payload payload
     * @return 保存后的对象
     */
    public PmsWbsBudgetVersionVO save(PmsWbsBudgetVersionPayload payload) {
        PmsWbsBudgetVersionDO ado = PmsWbsBudgetVersionConvert.INSTANCE.toEntity(payload);
        // 创建单据的时候审计字段会被前端传的值影响，强制清空相关字段
        SqlUtil.clearAuditFields(ado);
        repo.save(ado);
        return PmsWbsBudgetVersionConvert.INSTANCE.toVO(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsBudgetVersionDO> saveAll(List<PmsWbsBudgetVersionDO> dos) {
        // 创建单据的时候审计字段会被前端传的值影响，强制清空相关字段
        SqlUtil.clearAuditFields(dos);
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PmsWbsBudgetVersionPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        if (payload.getProId() != null) {
            update.set(qdo.proId, payload.getProId());
        }
        if (payload.getVersionNo() != null) {
            update.set(qdo.versionNo, payload.getVersionNo());
        }
        if (payload.getStepNum() != null) {
            update.set(qdo.stepNum, payload.getStepNum());
        }
        if (payload.getObjectStatus() != null) {
            update.set(qdo.objectStatus, payload.getObjectStatus());
        }
        // 处理要设置成空的字段(nullField处理)
        SqlUtil.handleNullFieldsUpdate(payload, qdo, update);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        return update.execute();
    }

    public PmsWbsBudgetVersionVO getVersion(Long proId, String status) {
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuerySelect = getJpaQuerySelect();
        PmsWbsBudgetVersionQuery pmsWbsVersionQuery = new PmsWbsBudgetVersionQuery();
        pmsWbsVersionQuery.setProId(proId);
        pmsWbsVersionQuery.setObjectStatus(status);
        jpaQueryWhere(jpaQuerySelect, pmsWbsVersionQuery);
        // 动态排序 versionNo desc,state asc
        OrderItem orderItem = new OrderItem();
        orderItem.setColumn("versionNo");
        orderItem.setAsc(false);
        OrderItem orderItem1 = new OrderItem();
        orderItem1.setColumn("objectStatus");
        pmsWbsVersionQuery.setOrders(List.of(orderItem, orderItem1));
        jpaQuerySelect.orderBy(com.elitesland.tw.tw5.server.common.util.SqlUtil.getSortedColumn(qdo, pmsWbsVersionQuery.getOrders()));
        PmsWbsBudgetVersionVO pmsWbsVersionVO = jpaQuerySelect.fetchFirst();
        return pmsWbsVersionVO;
    }

    public PmsWbsBudgetVersionVO getProIdAndVersionNo(Long proId, int lastVersionNo) {
        JPAQuery<PmsWbsBudgetVersionVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.proId.eq(proId));
        jpaQuery.where(qdo.versionNo.eq(lastVersionNo));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}
