package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetTemplateSubjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetTemplateSubjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetTemplateSubjectVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetTemplateSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetTemplateSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetTemplateSubjectRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算科目明细
 *
 * @author Echo
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetTemplateSubjectDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetTemplateSubjectRepo repo;
    private final QPmsBudgetTemplateSubjectDO qdo = QPmsBudgetTemplateSubjectDO.pmsBudgetTemplateSubjectDO;
    private final QPmsBudgetSubjectDO qSdo = QPmsBudgetSubjectDO.pmsBudgetSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetTemplateSubjectVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetTemplateSubjectVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 模板主键
                qdo.templateId,
                // 科目id
                qdo.subjectId,
                /*
                科目明细字段
                 */
                qSdo.subjectCode,
                qSdo.subjectName,
                qSdo.parentName,
                qSdo.parentId,
                qSdo.subjectStatus,
                qSdo.subjectLevel,
                qSdo.remark
        )).from(qdo).leftJoin(qSdo).on(qSdo.id.eq(qdo.subjectId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetTemplateSubjectVO> getJpaQueryWhere(PmsBudgetTemplateSubjectQuery query) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        jpaQuery.where(qSdo.deleteFlag.eq(0));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetTemplateSubjectQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetTemplateSubjectQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 模板主键 精确 */
        if (!ObjectUtils.isEmpty(query.getTemplateId())) {
            list.add(qdo.templateId.eq(query.getTemplateId()));
        }
        /** 科目id 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectId())) {
            list.add(qdo.subjectId.eq(query.getSubjectId()));
        }
        /** 科目层级 */
        if (!ObjectUtils.isEmpty(query.getSubjectLevel())) {
            list.add(qSdo.subjectLevel.eq(query.getSubjectLevel()));
        }
        /** 父级科目  */
        if (!ObjectUtils.isEmpty(query.getParentId())) {
            list.add(qSdo.parentId.eq(query.getParentId()));
        }
        /** 科目状态  */
        if (!ObjectUtils.isEmpty(query.getSubjectStatus())) {
            list.add(qSdo.subjectStatus.eq(query.getSubjectStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetTemplateSubjectVO queryByKey(Long id) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetTemplateSubjectVO> queryListDynamic(PmsBudgetTemplateSubjectQuery query) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetTemplateSubjectVO> queryPaging(PmsBudgetTemplateSubjectQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetTemplateSubjectVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetTemplateSubjectVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetTemplateSubjectDO save(PmsBudgetTemplateSubjectDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetTemplateSubjectDO> saveAll(List<PmsBudgetTemplateSubjectDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetTemplateSubjectPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 模板主键
        if (payload.getTemplateId() != null) {
            update.set(qdo.templateId, payload.getTemplateId());
        }
        // 预算科目编号
        if (payload.getSubjectId() != null) {
            update.set(qdo.subjectId, payload.getSubjectId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 模板主键
            if (nullFields.contains("templateId")) {
                update.setNull(qdo.templateId);
            }
            // 科目id
            if (nullFields.contains("subjectId")) {
                update.setNull(qdo.subjectId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 通过主表ids删除
     *
     * @param templateIds 主表ids
     */
    public long deleteSoftByTemplateIds(List<Long> templateIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.templateId.in(templateIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 通过科目ids删除
     *
     * @param subjectIds 科目ids
     */
    public long deleteSoftBySubjectIds(List<Long> subjectIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.subjectId.in(subjectIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 通过科目ids删除
     *
     * @param templateId 模板ID
     */
    public long deleteSoftByTemplateId(Long templateId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.templateId.eq(templateId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 通过模板id查询
     *
     * @param templateId 主表id
     * @return List<vo>
     */
    public List<PmsBudgetTemplateSubjectVO> queryByTemplateId(Long templateId) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.templateId.eq(templateId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qSdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    public List<PmsBudgetTemplateSubjectVO> queryByBySubjectId(Long subjectId) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.subjectId.eq(subjectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据科目id查询
     *
     * @param payload 单据数据
     * @return
     */
    public PmsBudgetTemplateSubjectVO queryBySubjectNameAndTemplateId(PmsBudgetTemplateSubjectPayload payload) {
        JPAQuery<PmsBudgetTemplateSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.subjectId.eq(payload.getSubjectId()));
        jpaQuery.where(qdo.templateId.eq(payload.getTemplateId()));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qSdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }
}

