package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectWbsRelyPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectWbsRelyQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsRelyVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectWbsRelyDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsRelyDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectWbsRelyRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目wbs前置依赖关系
 *
 * @author carl
 * @date 2023-07-19
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectWbsRelyDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectWbsRelyRepo repo;
    private final QPmsProjectWbsRelyDO qdo = QPmsProjectWbsRelyDO.pmsProjectWbsRelyDO;
    private final QPmsProjectWbsDO qPmsProjectWbsDO = QPmsProjectWbsDO.pmsProjectWbsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsRelyVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectWbsRelyVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // wbs主键(依赖建立着，后置任务)
                qdo.wbsId,
                // wbs名称(依赖建立着，后置任务，冗余字段)
                qdo.wbsName,
                // wbs主键(依赖前置任务)
                qdo.wbsRelyId,
                // wbs名称(前置依赖名称，冗余字段）
                qdo.wbsRelyName,
                //关系类型
                qdo.relationType,
                // 前置类型（FS）
                qdo.relyType,
                qdo.versionNo,
                qdo.versionId,
                qdo.effRelateId,
                qdo.wbsCode
        )).from(qdo);
    }

    public List<PmsProjectWbsRelyVO> listLeftJoinWbs(PmsProjectWbsRelyQuery query){
        return jpaQueryFactory.selectDistinct(Projections.bean(PmsProjectWbsRelyVO.class,
            qPmsProjectWbsDO.wbsStatus,
            qdo.id,
            //qdo.remark,
            //qdo.createUserId,
            //qdo.creator,
            //qdo.createTime,
            //qdo.modifyUserId,
            //qdo.updater,
            //qdo.modifyTime,
            // 项目主键
            qdo.projectId,
            // wbs主键(依赖建立着，后置任务)
            qdo.wbsId,
            // wbs名称(依赖建立着，后置任务，冗余字段)
            qdo.wbsName,
            // wbs主键(依赖前置任务)
            qdo.wbsRelyId,
            // wbs名称(前置依赖名称，冗余字段）
            qdo.wbsRelyName,
            //关系类型
            qdo.relationType,
            // 前置类型（FS）
            qdo.relyType
        )).from(qdo)
            .leftJoin(qPmsProjectWbsDO).on(qdo.wbsId.eq(qPmsProjectWbsDO.id))
            .where(whereLeftJoinWbs(query))
            .fetch();
    }

    private Predicate whereLeftJoinWbs(PmsProjectWbsRelyQuery query) {
        Predicate where = where(query);
        return where;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsRelyVO> getJpaQueryWhere(PmsProjectWbsRelyQuery query) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectWbsRelyQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectWbsRelyQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** wbs主键(依赖建立着，后置任务) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsId())) {
            list.add(qdo.wbsId.eq(query.getWbsId()));
        }
        /** wbs主键(依赖建立着，后置任务) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsIdList())) {
            list.add(qdo.wbsId.in(query.getWbsIdList()));
        }
        /** wbs名称(依赖建立着，后置任务，冗余字段) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.eq(query.getWbsName()));
        }
        /** wbs主键(依赖前置任务) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsRelyId())) {
            list.add(qdo.wbsRelyId.eq(query.getWbsRelyId()));
        }
        /** wbs名称(前置依赖名称，冗余字段） 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsRelyName())) {
            list.add(qdo.wbsRelyName.eq(query.getWbsRelyName()));
        }
        /** 前置类型（FS） 精确 */
        if (!ObjectUtils.isEmpty(query.getRelyType())) {
            list.add(qdo.relyType.eq(query.getRelyType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectWbsRelyVO queryByKey(Long id) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据项目主键查询
     *
     * @param projectId 项目主键
     * @return 结果
     */
    public List<PmsProjectWbsRelyVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public List<PmsProjectWbsRelyVO> queryByProjectIdAndVersionNo(Long projectId,Integer versionNo) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qdo.versionNo.eq(versionNo));
        return jpaQuery.fetch();
    }

    /**
     * 根据wbs主键查询
     *
     * @param wbsIds 主键
     * @return 结果
     */
    public List<PmsProjectWbsRelyVO> queryByWbsIds(List<Long> wbsIds) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.in(wbsIds).or(qdo.wbsRelyId.in(wbsIds)));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据当前节点和前置节点查询
     *
     * @param wbsId
     * @param relyIds
     * @return
     */
    public PmsProjectWbsRelyVO queryWbsIdAndWbsRelyIds(Long wbsId, List<Long> relyIds) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.eq(wbsId));
        jpaQuery.where(qdo.wbsRelyId.in(relyIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据当前节点和前置节点查询
     *
     * @param wbsIds
     * @param relyId
     * @return
     */
    public PmsProjectWbsRelyVO queryWbsIdsAndWbsRelyId(List<Long> wbsIds, Long relyId) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.in(wbsIds));
        jpaQuery.where(qdo.wbsRelyId.eq(relyId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectWbsRelyVO> queryListDynamic(PmsProjectWbsRelyQuery query) {
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectWbsRelyVO> queryPaging(PmsProjectWbsRelyQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectWbsRelyVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectWbsRelyVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectWbsRelyVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectWbsRelyDO save(PmsProjectWbsRelyDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectWbsRelyDO> saveAll(List<PmsProjectWbsRelyDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectWbsRelyPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // wbs主键(依赖建立着，后置任务)
        if (payload.getWbsId() != null) {
            update.set(qdo.wbsId, payload.getWbsId());
        }
        // wbs名称(依赖建立着，后置任务，冗余字段)
        if (payload.getWbsName() != null) {
            update.set(qdo.wbsName, payload.getWbsName());
        }
        // wbs主键(依赖前置任务)
        if (payload.getWbsRelyId() != null) {
            update.set(qdo.wbsRelyId, payload.getWbsRelyId());
        }
        // wbs名称(前置依赖名称，冗余字段）
        if (payload.getWbsRelyName() != null) {
            update.set(qdo.wbsRelyName, payload.getWbsRelyName());
        }
        // 前置类型（FS）
        if (payload.getRelyType() != null) {
            update.set(qdo.relyType, payload.getRelyType());
        }
        if (payload.getEffRelateId() != null) {
            update.set(qdo.effRelateId, payload.getEffRelateId());
        }
        if (payload.getWbsCode() != null) {
            update.set(qdo.wbsCode, payload.getWbsCode());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // wbs主键(依赖建立着，后置任务)
            if (nullFields.contains("wbsId")) {
                update.setNull(qdo.wbsId);
            }
            // wbs名称(依赖建立着，后置任务，冗余字段)
            if (nullFields.contains("wbsName")) {
                update.setNull(qdo.wbsName);
            }
            // wbs主键(依赖前置任务)
            if (nullFields.contains("wbsRelyId")) {
                update.setNull(qdo.wbsRelyId);
            }
            // wbs名称(前置依赖名称，冗余字段）
            if (nullFields.contains("wbsRelyName")) {
                update.setNull(qdo.wbsRelyName);
            }
            // 前置类型（FS）
            if (nullFields.contains("relyType")) {
                update.setNull(qdo.relyType);
            }
        }
        //拼装更新
//        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 根据wbsId逻辑删除
     *
     * @param wbsIds 主集合
     * @return 删除的行数
     */
    public long deleteSoftByWbsIds(List<Long> wbsIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.wbsId.in(wbsIds).or(qdo.wbsRelyId.in(wbsIds)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public void delByWbsId(Long wbsId, Integer versionNo) {
        long res = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.wbsId.eq(wbsId))
                .where(qdo.versionNo.eq(versionNo))
                .execute();

    }

    public void delByV0(List<Long> relyDelIds,Long wbsId) {
        long res = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.notIn(relyDelIds))
                .where(qdo.versionNo.eq(0))
                .where(qdo.wbsId.eq(wbsId))
                .execute();
    }
}

