package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsResourcePlanPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsResourcePlanQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsResourcePlanVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsResourcePlanDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsResourcePlanDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsResourcePlanRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 资源规划
 *
 * @author xxb
 * @date 2023-08-11
 */
@Repository
@RequiredArgsConstructor
public class PmsResourcePlanDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsResourcePlanRepo repo;
    private final QPmsResourcePlanDO qdo = QPmsResourcePlanDO.pmsResourcePlanDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsResourcePlanVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsResourcePlanVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 计划类型
                qdo.planType,
                // 关联对象ID
                qdo.objId,
                // 关联对象
                qdo.objName,
                // 开始日期
                qdo.startDate,
                // 结束日期
                qdo.endDate,
                // 持续数
                qdo.duration,
                // 资源规划状态 0正常
                qdo.planningStatus,
                // 商机销售阶段  --- 给商机类型资源规划用的
                qdo.salePhase,
                // 成单概率交付角度 --- 给商机类型资源规划用的
                qdo.probability,
                // 版本号
                qdo.versionNo,
                // 变更原因
                qdo.changeReason,
                // 交付负责人
                qdo.deliResId,
                // BUID
                qdo.buId,
                // 资源规划周期: WEEK周 MONTH月
                qdo.resourcePlanningCycle
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsResourcePlanVO> getJpaQueryWhere(PmsResourcePlanQuery query) {
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsResourcePlanQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        Long total = jpaQuery.fetchOne();
        total=total==null?0L:total;
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsResourcePlanQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 计划类型 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanType())) {
            list.add(qdo.planType.eq(query.getPlanType()));
        }
        /** 关联对象ID 精确 */
        if (!ObjectUtils.isEmpty(query.getObjId())) {
            list.add(qdo.objId.eq(query.getObjId()));
        }
        if (!ObjectUtils.isEmpty(query.getObjIds())) {
            list.add(qdo.objId.in(query.getObjIds()));
        }
        /** 关联对象 精确 */
        if (!ObjectUtils.isEmpty(query.getObjName())) {
            list.add(qdo.objName.eq(query.getObjName()));
        }
        /** 开始日期 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 结束日期 精确 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.eq(query.getEndDate()));
        }
        /** 资源规划状态 0正常 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanningStatus())) {
            list.add(qdo.planningStatus.eq(query.getPlanningStatus()));
        }
        // 持续数
        if (!ObjectUtils.isEmpty(query.getDuration())) {
            list.add(qdo.duration.eq(query.getDuration()));
        }
        // 资源规划周期
        if (!ObjectUtils.isEmpty(query.getResourcePlanningCycle())) {
            list.add(qdo.resourcePlanningCycle.eq(query.getResourcePlanningCycle()));
        }

//        /** 商机销售阶段  --- 给商机类型资源规划用的 精确 */
//        if (!ObjectUtils.isEmpty(query.getSalePhase())) {
//            list.add(qdo.salePhase.eq(query.getSalePhase()));
//        }
//        /** 成单概率交付角度 --- 给商机类型资源规划用的 精确 */
//        if (!ObjectUtils.isEmpty(query.getProbability())) {
//            list.add(qdo.probability.eq(query.getProbability()));
//        }
//        /** 版本号  精确 */
//        if (!ObjectUtils.isEmpty(query.getVersionNo())) {
//            list.add(qdo.versionNo.eq(query.getVersionNo()));
//        }
//        /** 变更原因 精确 */
//        if (!ObjectUtils.isEmpty(query.getChangeReason())) {
//            list.add(qdo.changeReason.eq(query.getChangeReason()));
//        }
//        /** 交付负责人 精确 */
//        if (!ObjectUtils.isEmpty(query.getDeliResId())) {
//            list.add(qdo.deliResId.eq(query.getDeliResId()));
//        }
//        /** BUID 精确 */
//        if (!ObjectUtils.isEmpty(query.getBuId())) {
//            list.add(qdo.buId.eq(query.getBuId()));
//        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsResourcePlanVO queryByKey(Long id) {
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<PmsResourcePlanVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        return jpaQuery.fetch();
    }

    /**
     * 资源规划-关联对象查询
     *
     * @param objId
     * @return
     */
    public PmsResourcePlanVO queryByObjId(Long objId, String planType) {
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.objId.eq(objId));
        jpaQuery.where(qdo.planType.eq(planType));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsResourcePlanVO> queryListDynamic(PmsResourcePlanQuery query) {
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsResourcePlanVO> queryPaging(PmsResourcePlanQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsResourcePlanVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsResourcePlanVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsResourcePlanVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsResourcePlanDO save(PmsResourcePlanDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsResourcePlanDO> saveAll(List<PmsResourcePlanDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsResourcePlanPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 计划类型
        if (payload.getPlanType() != null) {
            update.set(qdo.planType, payload.getPlanType());
        }
        // 关联对象ID
        if (payload.getObjId() != null) {
            update.set(qdo.objId, payload.getObjId());
        }
        // 关联对象
        if (payload.getObjName() != null) {
            update.set(qdo.objName, payload.getObjName());
        }
        // 开始日期
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 结束日期
        if (payload.getEndDate() != null) {
            update.set(qdo.endDate, payload.getEndDate());
        }
        // 资源规划周期
        if (payload.getResourcePlanningCycle() != null) {
            update.set(qdo.resourcePlanningCycle, payload.getResourcePlanningCycle());
        }
        // 持续数
        if (payload.getDuration() != null) {
            update.set(qdo.duration, payload.getDuration());
        }

        // 资源规划状态 0正常
        if (payload.getPlanningStatus() != null) {
            update.set(qdo.planningStatus, payload.getPlanningStatus());
        }
        /** 资源规划周期: WEEK周 MONTH月 */
        if (!ObjectUtils.isEmpty(payload.getResourcePlanningCycle())) {
            update.set(qdo.resourcePlanningCycle, payload.getResourcePlanningCycle());
        }
//        // 商机销售阶段  --- 给商机类型资源规划用的
//        if (payload.getSalePhase() != null) {
//            update.set(qdo.salePhase, payload.getSalePhase());
//        }
//        // 成单概率交付角度 --- 给商机类型资源规划用的
//        if (payload.getProbability() != null) {
//            update.set(qdo.probability, payload.getProbability());
//        }
//        // 版本号
//        if (payload.getVersionNo() != null) {
//            update.set(qdo.versionNo, payload.getVersionNo());
//        }
//        // 变更原因
//        if (payload.getChangeReason() != null) {
//            update.set(qdo.changeReason, payload.getChangeReason());
//        }
//        // 交付负责人
//        if (payload.getDeliResId() != null) {
//            update.set(qdo.deliResId, payload.getDeliResId());
//        }
//        // BUID
//        if (payload.getBuId() != null) {
//            update.set(qdo.buId, payload.getBuId());
//        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 计划类型
            if (nullFields.contains("planType")) {
                update.setNull(qdo.planType);
            }
            // 关联对象ID
            if (nullFields.contains("objId")) {
                update.setNull(qdo.objId);
            }
            // 关联对象
            if (nullFields.contains("objName")) {
                update.setNull(qdo.objName);
            }
            // 开始日期
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 结束日期
            if (nullFields.contains("endDate")) {
                update.setNull(qdo.endDate);
            }
            // 资源规划状态 0正常
            if (nullFields.contains("planningStatus")) {
                update.setNull(qdo.planningStatus);
            }
            // 商机销售阶段  --- 给商机类型资源规划用的
            if (nullFields.contains("salePhase")) {
                update.setNull(qdo.salePhase);
            }
            // 成单概率交付角度 --- 给商机类型资源规划用的
            if (nullFields.contains("probability")) {
                update.setNull(qdo.probability);
            }
            // 版本号
            if (nullFields.contains("versionNo")) {
                update.setNull(qdo.versionNo);
            }
            // 变更原因
            if (nullFields.contains("changeReason")) {
                update.setNull(qdo.changeReason);
            }
            // 交付负责人
            if (nullFields.contains("deliResId")) {
                update.setNull(qdo.deliResId);
            }
            // BUID
            if (nullFields.contains("buId")) {
                update.setNull(qdo.buId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

