package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsWbsActPayPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsWbsActPayQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsWbsActPayVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsWbsActPayDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsWbsActPayDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsWbsActPayRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * wbs活动交付物
 *
 * @author carl
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class PmsWbsActPayDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsActPayRepo repo;
    private final QPmsWbsActPayDO qdo = QPmsWbsActPayDO.pmsWbsActPayDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsActPayVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsActPayVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // wbs主键(仅活动类型)
                qdo.wbsId,
                // wbs描述（名称）
                qdo.wbsName,
                // 交付物名称
                qdo.payName,
                qdo.payTypeName,
                // 交付物类型
                qdo.payType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsActPayVO> getJpaQueryWhere(PmsWbsActPayQuery query) {
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsWbsActPayQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsWbsActPayQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** wbs主键(仅活动类型) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsId())) {
            list.add(qdo.wbsId.eq(query.getWbsId()));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsName())));
        }
        /** 交付物名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getPayName())) {
            list.add(qdo.payName.like(SqlUtil.toSqlLikeString(query.getPayName())));
        }
        /** 交付物类型 精确 */
        if (!ObjectUtils.isEmpty(query.getPayType())) {
            list.add(qdo.payType.eq(query.getPayType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsActPayVO queryByKey(Long id) {
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据ProjectId查询
     *
     * @param ProjectId Project主键
     * @return 结果
     */
    public List<PmsWbsActPayVO> queryByProjectId(Long ProjectId) {
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(ProjectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据wbsId查询
     *
     * @param wbsId wbs主键
     * @return 结果
     */
    public List<PmsWbsActPayVO> queryByWbsId(Long wbsId) {
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.eq(wbsId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsActPayVO> queryListDynamic(PmsWbsActPayQuery query) {
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsWbsActPayVO> queryPaging(PmsWbsActPayQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsWbsActPayVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsWbsActPayVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsWbsActPayVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsWbsActPayDO save(PmsWbsActPayDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsActPayDO> saveAll(List<PmsWbsActPayDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsWbsActPayPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // wbs主键(仅活动类型)
        if (payload.getWbsId() != null) {
            update.set(qdo.wbsId, payload.getWbsId());
        }
        // wbs描述（名称）
        if (payload.getWbsName() != null) {
            update.set(qdo.wbsName, payload.getWbsName());
        }
        // 交付物名称
        if (payload.getPayName() != null) {
            update.set(qdo.payName, payload.getPayName());
        }
        // 交付物类型
        if (payload.getPayType() != null) {
            update.set(qdo.payType, payload.getPayType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // wbs主键(仅活动类型)
            if (nullFields.contains("wbsId")) {
                update.setNull(qdo.wbsId);
            }
            // wbs描述（名称）
            if (nullFields.contains("wbsName")) {
                update.setNull(qdo.wbsName);
            }
            // 交付物名称
            if (nullFields.contains("payName")) {
                update.setNull(qdo.payName);
            }
            // 交付物类型
            if (nullFields.contains("payType")) {
                update.setNull(qdo.payType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除根据wbs主键
     *
     * @param wbsIds
     * @return 删除的行数
     */
    public long deleteSoftByWbsIds(List<Long> wbsIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.wbsId.in(wbsIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

