package com.elitesland.tw.tw5.server.prd.pms.service;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectCalendarPayload;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectCalendarService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectCalendarVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.scheduling.TimeUtil;
import com.elitesland.tw.tw5.server.prd.pms.convert.PmsProjectCalendarConvert;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectCalendarDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectCalendarRepo;
import com.elitesland.tw.tw5.server.prd.provacation.model.query.ProVacationQuery;
import com.elitesland.tw.tw5.server.prd.provacation.model.vo.ProVacationVO;
import com.elitesland.tw.tw5.server.prd.provacation.service.ProVacationService;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

/**
 * 项目工作日历
 *
 * @date 2023-06-21
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsProjectCalendarServiceImpl implements PmsProjectCalendarService {

    private final PmsProjectCalendarRepo pmsProjectCalendarRepo;

    //    private final VacationService vacationService;
    private final ProVacationService proVacationService;

    /***
     *  工作日历的缓存
     */
    private static Map<String, List<Map<LocalDate, BigDecimal>>> calendarCache = new HashMap<>();

    /**
     * 数据校验
     *
     * @param payload
     * @return
     */
    Boolean check(PmsProjectCalendarPayload payload) {
        //非空验证
        if (ObjectUtils.isEmpty(payload.getSourceId())) {
            throw TwException.error("", "关联主键不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getSourceType())) {
            throw TwException.error("", "关联类型不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getEndDate()) && ObjectUtils.isEmpty(payload.getStartDate()) && ObjectUtils.isEmpty(payload.getDurationDays())) {
            throw TwException.error("", "开始日期、结束日期和持续工作日期不能同时为空，请核验！");
        }
        if (!ObjectUtils.isEmpty(payload.getDurationDays())) {
            //说明是工期为0的数据
            if (payload.getDurationDays() == 0) {
                LocalDate endDate = payload.getEndDate();
                LocalDate startDate = payload.getStartDate();
                payload.setStartDate(endDate);
                payload.setEndDate(startDate);
                payload.setDurationDays(2);

                return true;
            }
        }
        return false;
    }

    @Override
    public PmsProjectCalendarVO findBySource(PmsProjectCalendarPayload payload) {
        //非空验证
//        if (ObjectUtils.isEmpty(payload.getSourceId())) {
//            throw TwException.error("", "关联主键不存在，请核验！");
//        }
//        if (ObjectUtils.isEmpty(payload.getSourceType())) {
//            throw TwException.error("", "关联类型不存在，请核验！");
//        }

        //判断工期是否为0
        Boolean isSpecial = check(payload);

        LocalDate startDate = payload.getStartDate();
        LocalDate endDate = payload.getEndDate();

//        if (ObjectUtils.isEmpty(startDate) && ObjectUtils.isEmpty(payload.getStartDate()) && ObjectUtils.isEmpty(payload.getDurationDays())) {
//            throw TwException.error("", "开始日期、结束日期和持续工作日期不能同时为空，请核验！");
//        }

        if (ObjectUtils.isEmpty(startDate)) {
            if (ObjectUtils.isEmpty(endDate) || ObjectUtils.isEmpty(payload.getDurationDays())) {
                throw TwException.error("", "结束日期和持续工作日期不能同时为空，请核验！");
            }
//            if (0 == payload.getDurationDays().intValue()) {
//                startDate = endDate.minusDays(1);
//            } else {
//                startDate = endDate.minusDays(30);
//            }

            startDate = endDate.minusDays(30);
        } else {
            if (ObjectUtils.isEmpty(endDate)) {
                if (ObjectUtils.isEmpty(startDate) || ObjectUtils.isEmpty(payload.getDurationDays())) {
                    throw TwException.error("", "开始日期和持续工作日期不能同时为空，请核验！");
                }
//                if (0 == payload.getDurationDays().intValue()) {
//                    endDate = startDate.minusDays(1);
//                } else {
//                    endDate = startDate.plusDays(30);
//                }
                endDate = startDate.plusDays(30);

            } else {
                // 如果开始时间和结束日期是同一天
                if (!ObjectUtils.isEmpty(payload.getEndDate())) {
                    payload.setEndDate(payload.getStartDate().plusDays(1));
                }
            }
        }

        PmsProjectCalendarVO vo = new PmsProjectCalendarVO();
        vo.setSourceId(payload.getSourceId());
        vo.setSourceType(payload.getSourceType());
        if (!ObjectUtils.isEmpty(payload.getStartDate())) {
            vo.setStartDate(payload.getStartDate());
        }
        if (!ObjectUtils.isEmpty(payload.getEndDate())) {
            vo.setEndDate(endDate);
        }
        if (!ObjectUtils.isEmpty(payload.getDurationDays())) {
            vo.setDurationDays(payload.getDurationDays());
        }

        if (endDate.isBefore(startDate)) {
            vo.setEndDate(endDate);
            vo.setDurationDays(0);
        } else {
            if (payload.getFromCacheFlag()) {
                vo.setCalendarList(getDataByDate(calendarCache.get(payload.getSourceType() + payload.getSourceId()), startDate, endDate));
            } else {
                //项目日历
                List<ProVacationVO> vacationList = getVacationList(startDate, endDate, payload.getProjectId());
                // 先判断下 工作日历是否有数据，没有，则取企业日历
//                PmsProjectCalendarDO pmsProjectCalendar = pmsProjectCalendarRepo.findBySource(payload.getSourceId(), payload.getSourceType());
//                if (!ObjectUtils.isEmpty(pmsProjectCalendar) && !ObjectUtils.isEmpty(pmsProjectCalendar.getCalendarInfo())) {
//                    vo.setCalendarList(handleCalendar(vacationList, startDate, endDate, pmsProjectCalendar));
//                } else {
//                    vo.setCalendarList(vacationToList(vacationList));
//                }
                vo.setCalendarList(vacationToList(vacationList));
            }
            //获取预计持续天数
            List<Map<LocalDate, BigDecimal>> list = vo.getCalendarList();
            //获取预计持续天数
            if (!ObjectUtils.isEmpty(list)) {
                if (ObjectUtils.isEmpty(payload.getDurationDays())) {
                    countDurationDays(vo, list);
                } else {
                    if (!ObjectUtils.isEmpty(payload.getStartDate())) {
                        countEndDate(vo, list, payload.getDurationDays());
                    }
                    if (!ObjectUtils.isEmpty(payload.getEndDate())) {
                        countStartDate(vo, list, payload.getDurationDays());
                    }
                }
            }
        }
        if (isSpecial) {
            //工期为0
            LocalDate endDate0 = vo.getEndDate();
            LocalDate startDate0 = vo.getStartDate();
            vo.setStartDate(endDate0);
            vo.setEndDate(startDate0);
            vo.setDurationDays(0);

        }

        return vo;
    }


    @Override
    public void handleCalendarCache(List<Long> sourceIds, String sourceType, String handleType, Long projectId) {
        if ("save".equals(handleType)) {
            // 先判断下 工作日历是否有数据，没有，则取企业日历
//            List<PmsProjectCalendarDO> pmsProjectCalendars = pmsProjectCalendarRepo.findBySources(sourceIds, sourceType);
            //获取当前日期
            LocalDate today = LocalDate.now();
            // 获取一年前日期
            LocalDate startDate = today.minus(1, ChronoUnit.YEARS);
            // 获取一年后日期
            LocalDate endDate = today.plus(1, ChronoUnit.YEARS);
            //获取 企业日历
            List<ProVacationVO> vacationList = getVacationList(startDate, endDate, projectId);

            // 如果 工作日历 存在情况
//            if (!ObjectUtils.isEmpty(pmsProjectCalendars)) {
//                for (PmsProjectCalendarDO pmsProjectCalendar : pmsProjectCalendars) {
//                    List<Map<LocalDate, Float>> list = handleCalendar(vacationList, startDate, endDate, pmsProjectCalendar);
//                    calendarCache.put(sourceType + pmsProjectCalendar.getSourceId(), list);
//                }
//            }
            // 如果 工作日历 不存在情况 查找 企业日历
            List<Map<LocalDate, BigDecimal>> list = vacationToList(vacationList);
            for (Long sourceId : sourceIds) {
                String key = sourceType + sourceId;
                if (!calendarCache.containsKey(key)) {
                    calendarCache.put(key, list);
                }
            }

        } else if ("clean".equals(handleType)) {
            calendarCache.clear();
        }
    }


    /**
     * 从  企业日历  和 工作日历中 取数据
     *
     * @param vacationList
     * @param startDate
     * @param endDate
     * @param pmsProjectCalendar
     * @return
     */
    private List<Map<LocalDate, BigDecimal>> handleCalendar(List<ProVacationVO> vacationList, LocalDate startDate, LocalDate endDate, PmsProjectCalendarDO pmsProjectCalendar) {
        List<Map<LocalDate, BigDecimal>> list = new LinkedList<>();
        if (startDate.isBefore(pmsProjectCalendar.getStartDate())) {
            // 从企业日历获取
            List<ProVacationVO> vacation1 = vacationList.stream().filter(v -> v.getNaturalDate().isEqual(startDate) || (v.getNaturalDate().isAfter(startDate) && v.getNaturalDate().isBefore(pmsProjectCalendar.getStartDate()))).collect(toList());
            list.addAll(vacationToList(vacation1));
        }
        // 从工作日历获取
        list.addAll(calendarInfoToList(pmsProjectCalendar.getCalendarInfo(), startDate, endDate));
        if (endDate.isAfter(pmsProjectCalendar.getEndDate())) {
            // 从企业日历获取
            List<ProVacationVO> vacation2 = vacationList.stream().filter(v -> v.getNaturalDate().isAfter(pmsProjectCalendar.getEndDate()) || (v.getNaturalDate().isBefore(endDate) && v.getNaturalDate().isEqual(endDate))).collect(toList());
            list.addAll(vacationToList(vacation2));
        }
        return getDataByDate(list, startDate, endDate);
    }

    /**
     * 根据 开始时间结束时间  获取 日历
     *
     * @param startDate
     * @param endDate
     * @return
     */
    private List<Map<LocalDate, BigDecimal>> getDataByDate(List<Map<LocalDate, BigDecimal>> list, LocalDate startDate, LocalDate endDate) {
        List<Map<LocalDate, BigDecimal>> result = new LinkedList<>();
        for (Map<LocalDate, BigDecimal> map : list) {
            for (LocalDate date : map.keySet()) {
                if (date.isBefore(endDate) && date.isAfter(startDate)) {
                    Map<LocalDate, BigDecimal> m = new HashMap<>();
                    m.put(date, map.get(date));
                    result.add(m);
                }
                if (date.isEqual(endDate) || date.isEqual(startDate)) {
                    Map<LocalDate, BigDecimal> m = new HashMap<>();
                    m.put(date, map.get(date));
                    result.add(m);
                }
            }
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveData(PmsProjectCalendarPayload payload) {
        //非空验证
        nonEmptyValidation(payload);
        //设置固定班制
        setFixedShiftSystem(payload);

        // 先判断下 工作日历是否存在
        PmsProjectCalendarDO calendarDO = pmsProjectCalendarRepo.findBySource(payload.getSourceId(), payload.getSourceType());

        if (!ObjectUtils.isEmpty(calendarDO)) {
            calendarDO.setStartDate(payload.getStartDate());
            calendarDO.setEndDate(payload.getEndDate());
            calendarDO.setSetFixedShiftSystemFlag(payload.getSetFixedShiftSystemFlag());
            calendarDO.setHoliday(payload.getHoliday());
            Gson gson = new Gson();
            calendarDO.setCalendarInfo(gson.toJson(payload.getCalendarList()));
            pmsProjectCalendarRepo.save(calendarDO);
        } else {
            PmsProjectCalendarDO entityDo = PmsProjectCalendarConvert.INSTANCE.toDo(payload);
            Gson gson = new Gson();
            entityDo.setCalendarInfo(gson.toJson(payload.getCalendarList()));
            pmsProjectCalendarRepo.save(entityDo);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSave(List<PmsProjectCalendarPayload> list) {
        //判断主键是否存在
        List<Long> ids = list.stream().map(PmsProjectCalendarPayload::getId).collect(toList());
        if (!ObjectUtils.isEmpty(ids)) {
            long count = pmsProjectCalendarRepo.findCountByKey(ids);
            if (count != ids.size()) {
                throw TwException.error("", "存在工作日历主键不存在问题，请核验！");
            }
        }

        List<PmsProjectCalendarDO> doList = new ArrayList<>();
        for (PmsProjectCalendarPayload payload : list) {
            //非空验证
            nonEmptyValidation(payload);
            //设置固定班制
            setFixedShiftSystem(payload);

            PmsProjectCalendarDO entityDo = PmsProjectCalendarConvert.INSTANCE.toDo(payload);
            Gson gson = new Gson();
            entityDo.setCalendarInfo(gson.toJson(payload.getCalendarList()));
            doList.add(entityDo);
        }
        pmsProjectCalendarRepo.saveAll(doList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        pmsProjectCalendarRepo.deleteByKey(keys);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftBySource(List<Long> sourceIds, String sourceType) {
        pmsProjectCalendarRepo.deleteSoftBySource(sourceIds, sourceType);
    }


    /**
     * 非空验证
     *
     * @param payload
     */
    private void nonEmptyValidation(PmsProjectCalendarPayload payload) {
        if (ObjectUtils.isEmpty(payload.getSourceId())) {
            throw TwException.error("", "关联主键不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getSourceType())) {
            throw TwException.error("", "关联类型不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getStartDate())) {
            throw TwException.error("", "开始日期不存在，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getEndDate())) {
            throw TwException.error("", "结束日期存在，请核验！");
        }
//        if (!"WBS".equals(payload.getSourceType()) && !"project".equals(payload.getSourceType())) {
//            throw TwException.error("", "关联类型不匹配，请核验！");
//        }
    }

    /**
     * 设置固定班制
     *
     * @param payload
     */
    private void setFixedShiftSystem(PmsProjectCalendarPayload payload) {
        if (1 == payload.getSetFixedShiftSystemFlag()) {
            List<Map<String, Object>> info = new LinkedList<>();

            String holiday = payload.getHoliday();

            LocalDate startDate = payload.getStartDate();
            LocalDate endDate = payload.getEndDate();

            LocalDate currentDate = startDate;
            while (!currentDate.isAfter(endDate)) {
                //获取了当前日期对应的DayOfWeek枚举类型的实例
                DayOfWeek dayOfWeek = currentDate.getDayOfWeek();
                int day = dayOfWeek.getValue();

                Map<String, Object> m = new HashMap<>();
                if (!ObjectUtils.isEmpty(holiday) && holiday.contains(day + "")) {
                    m.put(TimeUtil.dateToYmd(currentDate), 0.0);
                } else {
                    m.put(TimeUtil.dateToYmd(currentDate), 8.0);
                }
                info.add(m);

                // 递增到下一个日期
                currentDate = currentDate.plusDays(1);
            }

            payload.setCalendarList(info);
        }
    }

    /**
     * 从企业日历获取数据
     *
     * @param startDate
     * @param endDate
     * @return
     */
    private List<ProVacationVO> getVacationList(LocalDate startDate, LocalDate endDate, Long projectId) {
//        List<VacationVO> list = vacationService.findListByDate(startDate, endDate);
        ProVacationQuery proVacationQuery = new ProVacationQuery();
        proVacationQuery.setNaturalDateStart(startDate);
        proVacationQuery.setNaturalDateEnd(endDate);
        proVacationQuery.setProjectId(projectId);
        List<ProVacationVO> list = proVacationService.getList(proVacationQuery);
        if (CollUtil.isEmpty(list)) {
            throw new BusinessException("该项目没有设置项目日历，请先设置");
        }
        //按时间正序排序
        return list.stream().sorted(Comparator.comparing(vo -> vo.getNaturalDate())).collect(Collectors.toList());
    }

    /**
     * 数据转换
     *
     * @param vacationList
     * @return
     */
    private List<Map<LocalDate, BigDecimal>> vacationToList(List<ProVacationVO> vacationList) {
        List<Map<LocalDate, BigDecimal>> info = new LinkedList<>();
        vacationList.forEach(twVacationView -> {
            Map<LocalDate, BigDecimal> map = new HashMap<>();
            map.put(twVacationView.getNaturalDate(), twVacationView.getWorkHours());
            info.add(map);
        });
        return info;
    }

    /**
     * 数据转换
     *
     * @return
     */
    private List<Map<LocalDate, BigDecimal>> calendarInfoToList(String calendarInfo, LocalDate startDate, LocalDate endDate) {
        List<Map<LocalDate, BigDecimal>> result = new LinkedList<>();
        Gson gson = new Gson();
        Type listType = new TypeToken<List<Map<String, Object>>>() {
        }.getType();
        List<Map<String, Object>> list = gson.fromJson(calendarInfo, listType);
        for (Map<String, Object> map : list) {
            for (String key : map.keySet()) {
                LocalDate date = LocalDate.parse(key);
                if (date.isBefore(endDate) && date.isAfter(startDate)) {
                    Map<LocalDate, BigDecimal> m = new HashMap<>();
                    m.put(date, new BigDecimal(map.get(key).toString()));
                    result.add(m);
                }
                if (date.isEqual(endDate) || date.isEqual(startDate)) {
                    Map<LocalDate, BigDecimal> m = new HashMap<>();
                    m.put(date, new BigDecimal(map.get(key).toString()));
                    result.add(m);
                }
            }
        }
        return result;
    }

    /**
     * 根据持续天数 计算结束时间
     *
     * @param vo
     * @param list
     * @param durationDays
     */
    private void countEndDate(PmsProjectCalendarVO vo, List<Map<LocalDate, BigDecimal>> list, int durationDays) {
        int count = 0;
        int holiDays = 0;
        LocalDate startDate = null;
        LocalDate endDate = null;
        List<Map<LocalDate, BigDecimal>> list2 = new LinkedList<>();
        for (Map<LocalDate, BigDecimal> map : list) {
            list2.add(map);
            BigDecimal value = map.values().iterator().next();
            if (value.compareTo(BigDecimal.ZERO) > 0) {
                if (ObjectUtils.isEmpty(startDate)) {
                    startDate = map.keySet().iterator().next();
                }
                count++;
                if (count == durationDays) {
                    endDate = map.keySet().iterator().next();
                    break;
                }
            } else {
                holiDays = holiDays + 1;
            }
        }
        vo.setStartDate(startDate);
        vo.setEndDate(endDate);
        vo.setHoliDays(holiDays);
        vo.setCalendarList(list2);
    }

    /**
     * 根据持续天数 计算结束时间
     *
     * @param vo
     * @param list
     * @param durationDays
     */
    private void countStartDate(PmsProjectCalendarVO vo, List<Map<LocalDate, BigDecimal>> list, int durationDays) {
        // Sort the list in descending order based on LocalDate key
        Collections.sort(list, new Comparator<Map<LocalDate, BigDecimal>>() {
            @Override
            public int compare(Map<LocalDate, BigDecimal> map1, Map<LocalDate, BigDecimal> map2) {
                LocalDate key1 = map1.keySet().iterator().next();
                LocalDate key2 = map2.keySet().iterator().next();
                return key2.compareTo(key1); // Compare in reverse order
            }
        });

        int count = 0;
        int holiDays = 0;
        LocalDate startDate = null;
        LocalDate endDate = null;
        List<Map<LocalDate, BigDecimal>> list2 = new LinkedList<>();
        for (Map<LocalDate, BigDecimal> map : list) {
            list2.add(map);
            BigDecimal value = map.values().iterator().next();
            if (value.compareTo(BigDecimal.ZERO) > 0) {
                if (ObjectUtils.isEmpty(endDate)) {
                    endDate = map.keySet().iterator().next();
                }
                count++;
                if (count == durationDays) {
                    startDate = map.keySet().iterator().next();
                    break;
                }
            } else {
                holiDays = holiDays + 1;
            }
        }
        vo.setStartDate(startDate);
        vo.setEndDate(endDate);
        vo.setHoliDays(holiDays);

        Collections.sort(list2, new Comparator<Map<LocalDate, BigDecimal>>() {
            @Override
            public int compare(Map<LocalDate, BigDecimal> map1, Map<LocalDate, BigDecimal> map2) {
                LocalDate key1 = map1.keySet().iterator().next();
                LocalDate key2 = map2.keySet().iterator().next();
                return key1.compareTo(key2); // Compare in reverse order
            }
        });
        vo.setCalendarList(list2);
    }


    /**
     * 计算持续天数
     *
     * @param vo
     * @param list
     */
    private void countDurationDays(PmsProjectCalendarVO vo, List<Map<LocalDate, BigDecimal>> list) {
        int durationDays = 0;
        int holiDays = 0;
        for (Map<LocalDate, BigDecimal> map : list) {
            boolean hasValueGreaterThanZero = false;
            for (BigDecimal value : map.values()) {
                if (value.doubleValue() > 0) {
                    hasValueGreaterThanZero = true;
                    break;
                } else {
                    holiDays = holiDays + 1;
                }
            }
            if (hasValueGreaterThanZero) {
                durationDays = durationDays + 1;
            }
        }
        vo.setHoliDays(holiDays);
        vo.setDurationDays(durationDays);
    }
}
