package com.elitesland.tw.tw5.server.prd.purchase.controller;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentAccountantApprovePayload;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentSlipPayload;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchasePaymentPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchasePaymentDefaultQuery;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchasePaymentQuery;
import com.elitesland.tw.tw5.api.prd.purchase.service.PurchasePaymentService;
import com.elitesland.tw.tw5.api.prd.purchase.vo.CostUndertakeDepartmentVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurConOrAgreementDetailsSimpleVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchasePaymentVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.permission.annotation.FunctionDetail;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionDomain;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionFunction;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionCodeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionTypeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionDomainEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionTypeEnum;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Objects;

/**
 * 付款申请单
 *
 * @folder 采购合同管理-付款申请单
 * @author likunpeng
 * @date 2023-11-07
 */
@Api(tags = "付款申请单")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm/purchasePayment")
@Slf4j
@PermissionDomain(domain = PermissionDomainEnum.PAYMENT_APPLY)
public class PurchasePaymentController {

private final PurchasePaymentService purchasePaymentService;

    /**
     * 付款申请单-新增/修改
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping
    @ApiOperation("付款申请单-新增/修改")
    @UdcNameClass
    public TwOutputUtil<PurchasePaymentVO> insertOrUpdate(@RequestBody PurchasePaymentPayload payload){
         return TwOutputUtil.ok(purchasePaymentService.insertOrUpdate(payload));
    }

    /**
     * 付款申请单-应付会计审批
     *
     * @param payload 应付会计审批
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/accountantApprove")
    @ApiOperation("付款申请单-应付会计审批")
    @UdcNameClass
    public TwOutputUtil<PurchasePaymentVO> accountantApprove(@RequestBody PaymentAccountantApprovePayload payload){
        return TwOutputUtil.ok(purchasePaymentService.accountantApprove(payload));
    }

    /**
     * 付款申请单-主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/{key}")
    @UdcNameClass
    @ApiOperation("付款申请单-主键查询")
    public TwOutputUtil<PurchasePaymentVO> queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(purchasePaymentService.queryByKey(key));
    }

    /**
     * 付款申请单-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/paging")
    @UdcNameClass
    @ApiOperation("付款申请单-分页")
    public TwOutputUtil<PagingVO<PurchasePaymentVO>> paging(PurchasePaymentQuery query) {
        return TwOutputUtil.ok(purchasePaymentService.queryPaging(query));
    }

    /**
     * 付款申请单-权限分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/permission/paging")
    @UdcNameClass
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.PA_PAYMENT_LIST_DATA_POWER, type = FunctionTypeEnum.TAB)
    }, fieldPermission = true, permissionType = PermissionTypeEnum.DATA_PERMISSION)
    public TwOutputUtil<PagingVO<PurchasePaymentVO>> permissionPaging(@RequestBody PurchasePaymentQuery query) {
        return TwOutputUtil.ok(purchasePaymentService.permissionPaging(query));
    }


    /**
     * 付款申请单-删除
     *
     * @param id 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    @ApiOperation("付款申请单-删除")
    public TwOutputUtil deleteSoft(Long id) {
         purchasePaymentService.deleteSoft(id, false);
         return TwOutputUtil.ok();
    }

    /**
     * 查询付款申请单操作日志
     * @param key 采购合同主键
     * @return
     */
    @GetMapping("/log/list/{key}")
    public TwOutputUtil<List<PrdSystemLogVO>> queryLogList(@PathVariable Long key) {
        return TwOutputUtil.ok(purchasePaymentService.queryLogList(key));
    }

    /**
     * 根据申请单类型以及单据编号查询默认信息
     * @param defaultQuery 查询条件
     */
    @GetMapping("/default")
    @UdcNameClass()
    public TwOutputUtil<PurchasePaymentVO> getDefaultByTypeAndDocNo(PurchasePaymentDefaultQuery defaultQuery) {
        return TwOutputUtil.ok(purchasePaymentService.getDefaultByTypeAndDocNo(defaultQuery));
    }

    /**
     * 导出付款单列表excel
     *
     * @param query 查询条件
     * @return result
     */
    @GetMapping("/exportPaymentApply")
    public void exportPaymentApply(HttpServletResponse response, PurchasePaymentQuery query) {
        purchasePaymentService.exportPaymentApply(response, query);
    }

    /**
     * 付款申请单-权限导出付款单列表excel
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/permission/exportPaymentApply")
    @UdcNameClass
    public void permissionExportPaymentApply(@RequestBody PurchasePaymentQuery query, HttpServletResponse response) {
        purchasePaymentService.permissionExportPaymentApply(response, query);
    }

    /**
     * 根据流程ID查询该流程是否符合紧急付款条件
     * @param procInstId 流程实例id
     * @return result
     */
    @GetMapping("/findUrgentPaymentByProcInstId")
    @UdcNameClass()
    public TwOutputUtil<PurchasePaymentVO> findUrgentPaymentByProcInstId(String procInstId) {
        return TwOutputUtil.ok(purchasePaymentService.findUrgentPaymentByProcInstId(procInstId));
    }

    /**
     * 根据采购合同IdOr采购协议Id查询采购明细
     * @param purConOrAgreementId 采购合同Id
     * @param docType 关联单据类型
     * @return 采购明细
     */
    @GetMapping("/queryByPurConOrAgreementId")
    @ApiOperation("根据采购合同IdOr采购协议Id查询采购明细")
    public TwOutputUtil<List<PurConOrAgreementDetailsSimpleVO>> queryByPurConOrAgreementId(Long purConOrAgreementId, String docType) {
        return TwOutputUtil.ok(purchasePaymentService.queryByPurConOrAgreementId(purConOrAgreementId, docType));
    }

    /**
     * 根据发票号查询已经传输JDE的费用部门承担明细
     * @param invoiceNo 发票号
     * @return 费用承担明细
     */
    @GetMapping("/queryByPurConOrInvoiceNo")
    @ApiOperation("根据发票号查询已经传输JDE的费用部门承担明细")
    @UdcNameClass
    public TwOutputUtil<List<CostUndertakeDepartmentVO>> queryByPurConOrInvoiceNo(String invoiceNo) {
        return TwOutputUtil.ok(purchasePaymentService.queryByPurConOrInvoiceNo(invoiceNo));
    }

    /**
     * 根据发票号查询已经传输JDE的费用部门承担明细
     * @param payload 发票号
     * @return 费用承担明细
     */
    @PutMapping("/queryByBatchPurConOrInvoiceNo")
    @ApiOperation("根据发票号查询已经传输JDE的费用部门承担明细")
    @UdcNameClass
    public TwOutputUtil<List<CostUndertakeDepartmentVO>> queryByBatchPurConOrInvoiceNo(@RequestBody PaymentSlipPayload payload) {

        if(Objects.isNull(payload) || CollectionUtils.isEmpty(payload.getInvoiceNos())){
            return TwOutputUtil.error("发票号数据传值不能为空");
        }
        return TwOutputUtil.ok(purchasePaymentService.queryByPurConOrInvoiceNos(payload));
    }
}

