package com.elitesland.tw.tw5.server.prd.salecon.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.org.query.PrdOrgOrganizationQuery;
import com.elitesland.tw.tw5.api.prd.org.service.PrdOrgOrganizationService;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConEpibolyCostConDPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConEpibolyCostConDQuery;
import com.elitesland.tw.tw5.api.prd.salecon.service.ConEpibolyCostConDService;
import com.elitesland.tw.tw5.api.prd.salecon.service.ConEpibolyCostConService;
import com.elitesland.tw.tw5.api.prd.salecon.service.SaleConContractService;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConEpibolyCostConDVO;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConEpibolyCostConVO;
import com.elitesland.tw.tw5.api.prd.salecon.vo.SaleConContractVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.prd.salecon.convert.ConEpibolyCostConDConvert;
import com.elitesland.tw.tw5.server.prd.salecon.dao.ConEpibolyCostConDDAO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConEpibolyCostConDDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConEpibolyCostConDRepo;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConEpibolyCostConRepo;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 外包费用确认明细
 *
 * @author likunpeng
 * @date 2023-04-17
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ConEpibolyCostConDServiceImpl extends BaseServiceImpl implements ConEpibolyCostConDService {

    private final ConEpibolyCostConDRepo conEpibolyCostConDRepo;
    private final ConEpibolyCostConRepo conEpibolyCostConRepo;
    private final ConEpibolyCostConDDAO conEpibolyCostConDDAO;
    @Autowired
    private SaleConContractService saleConContractService;
    @Autowired
    private final ConEpibolyCostConService costConService;
    private final WorkflowUtil workflowUtil;
    private final PrdOrgOrganizationService orgService;
    private final TransactionUtilService transactionUtilService;
    private final CacheUtil cacheUtil;

    @Value("${tw5.workflow.enabled}")
    private Boolean workflow_enabled;

    @Override
    public PagingVO<ConEpibolyCostConDVO> queryPaging(ConEpibolyCostConDQuery query) {
        //获取权限
        getPermissionParams(query);

        PagingVO<ConEpibolyCostConDVO> pagingVO = conEpibolyCostConDDAO.queryPaging(query);
        List<ConEpibolyCostConDVO> records = pagingVO.getRecords();
        List<Long> patentIds = records.stream().map(ConEpibolyCostConDVO::getId).collect(Collectors.toList());
        // 查询列表的子明细
        Map<Long, List<ConEpibolyCostConDVO>> childrenMap = this.queryByParentIdIn(patentIds)
                .stream().collect(Collectors.groupingBy(ConEpibolyCostConDVO::getParentId));
        records.forEach(conEpibolyCostConDVO -> {
            conEpibolyCostConDVO.setChildrenList(childrenMap.get(conEpibolyCostConDVO.getId()));
        });
        return pagingVO;
    }

    private List<ConEpibolyCostConDVO> queryByParentIdIn(List<Long> patentIds) {
        return conEpibolyCostConDDAO.queryByParentIdIn(patentIds);
    }

    private void getPermissionParams(ConEpibolyCostConDQuery query) {
        Long loginUserId = GlobalUtil.getLoginUserId();
        // 获取角色
        List<String> userSystemRoleCodes = cacheUtil.getSystemRoleCodes(loginUserId);
        // 销售副总裁(SALES_VP)、销售合同管理员(SALES_CONTRACT_ADMIN)、系统管理员(SYS)、财务负责人(FIN_PIC)、商务负责人(BUSINESS_MANAGER)、合同归档管理员(CONTRACT_FILING)、平台财务负责人(PLAT_FIN_PIC)、运营总裁(OPERATION_PRESIDENT)、财务核定(FIN_CHECK_PIC)可以看到全部
        List<String> roles = Arrays.asList("SYS", "PLAT_FIN_PIC", "OPERATION_PRESIDENT");
        if ((!CollectionUtils.isEmpty(userSystemRoleCodes)) && CollectionUtils.containsAny(userSystemRoleCodes, roles)) {
            query.setAuthAll(true);
            return;
        }
        // 查看当前登录人是哪个bu的leader
        PrdOrgOrganizationQuery orgQuery = new PrdOrgOrganizationQuery();
        orgQuery.setManageId(loginUserId);
        orgQuery.setSize(Integer.MAX_VALUE);
        PagingVO<PrdOrgOrganizationVO> paging = orgService.paging(orgQuery);
        List<PrdOrgOrganizationVO> records = paging.getRecords();
        if (!CollectionUtils.isEmpty(records)) {
            List<Long> orgIds = records.stream().map(PrdOrgOrganizationVO::getId).collect(Collectors.toList());
            //当前登录人作为部门负责人的所有bu
            query.setInchargeSignBus(orgIds);
        } else {
            //当前登录人作为签单bu负责人的所有bu
            query.setInchargeSignBus(new ArrayList<Long>());
        }
    }


    @Override
    public List<ConEpibolyCostConDVO> queryListDynamic(ConEpibolyCostConDQuery query) {
        List<ConEpibolyCostConDVO> conEpibolyCostConDVOS = conEpibolyCostConDDAO.queryListDynamic(query);
        for (ConEpibolyCostConDVO conEpibolyCostConDVO : conEpibolyCostConDVOS) {
            List<ConEpibolyCostConDVO> childrenList = this.queryByParentId(conEpibolyCostConDVO.getId());
            conEpibolyCostConDVO.setChildrenList(childrenList);
        }
        return conEpibolyCostConDVOS;
    }

    @Override
    public ConEpibolyCostConDVO queryByKey(Long key) {
        ConEpibolyCostConDVO epibolyCostConDVO = conEpibolyCostConDDAO.queryByKey(key);
        return epibolyCostConDVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ConEpibolyCostConDVO insert(ConEpibolyCostConDPayload payload) {
        ConEpibolyCostConDDO entityDo = ConEpibolyCostConDConvert.INSTANCE.toDo(payload);
        // 计算税费
        BigDecimal taxCost = payload.getAmt().multiply(payload.getTaxRate());
        entityDo.setTaxCost(taxCost);
        // 状态为新建CREATE
        entityDo.setDetailedStatus("CREATE");
        // 修改子合同有效合同金额
        if (payload.getParentId() == null) {
            ConEpibolyCostConVO conEpibolyCostConVO = costConService.queryByKey(payload.getEpibolyCostConId());
            SaleConContractVO saleConContractVO = saleConContractService.queryByKey(conEpibolyCostConVO.getContractId());
            BigDecimal effectiveAmt = saleConContractVO.getEffectiveAmt(); //子合同有效合同金额
            BigDecimal ExtraAmt = payload.getAmt();
            BigDecimal newEffectiveAmt = effectiveAmt.subtract(ExtraAmt);
            //修改子合同有效合同金额
            conEpibolyCostConDDAO.updateSubConEffectiveAmt(newEffectiveAmt, conEpibolyCostConVO.getContractId());
        }
        // 定时任务同步更新字段
        conEpibolyCostConRepo.updateRemark(entityDo.getEpibolyCostConId());
        return ConEpibolyCostConDConvert.INSTANCE.toVo(conEpibolyCostConDRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ConEpibolyCostConDVO update(ConEpibolyCostConDPayload payload) {
        ConEpibolyCostConDDO entity = conEpibolyCostConDRepo.findById(payload.getId()).orElseGet(ConEpibolyCostConDDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ConEpibolyCostConDDO entityDo = ConEpibolyCostConDConvert.INSTANCE.toDo(payload);
        // 计算税费
        BigDecimal taxCost = payload.getAmt().multiply(payload.getTaxRate());
        entityDo.setTaxCost(taxCost);
        // 修改子合同有效合同金额
        if (payload.getParentId() == null) {
            ConEpibolyCostConVO conEpibolyCostConVO = costConService.queryByKey(payload.getEpibolyCostConId());
            SaleConContractVO saleConContractVO = saleConContractService.queryByKey(conEpibolyCostConVO.getContractId());
            BigDecimal effectiveAmt = saleConContractVO.getEffectiveAmt(); //子合同有效合同金额
            BigDecimal newExtraAmt = payload.getAmt();
            BigDecimal oldExtraAmt = entity.getAmt();
            BigDecimal newEffectiveAmt = effectiveAmt.subtract(newExtraAmt).add(oldExtraAmt);
            //修改子合同有效合同金额
            conEpibolyCostConDDAO.updateSubConEffectiveAmt(newEffectiveAmt, conEpibolyCostConVO.getContractId());
        }
        entity.copy(entityDo);
        // 定时任务同步更新字段
        conEpibolyCostConRepo.updateRemark(entity.getEpibolyCostConId());
        return ConEpibolyCostConDConvert.INSTANCE.toVo(conEpibolyCostConDRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long updateByKeyDynamic(ConEpibolyCostConDPayload payload) {
        return conEpibolyCostConDDAO.updateByKeyDynamic(payload);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ConEpibolyCostConDDO> optional = conEpibolyCostConDRepo.findById(id);
                if (!optional.isEmpty()) {
                    ConEpibolyCostConDDO entity = optional.get();
                    // 修改子合同有效合同金额
                    if (entity.getParentId() == null) {
                        ConEpibolyCostConVO conEpibolyCostConVO = costConService.queryByKey(entity.getEpibolyCostConId());
                        SaleConContractVO saleConContractVO = saleConContractService.queryByKey(conEpibolyCostConVO.getContractId());
                        BigDecimal effectiveAmt = saleConContractVO.getEffectiveAmt(); //子合同有效合同金额
                        BigDecimal newEffectiveAmt = effectiveAmt.add(entity.getAmt());
                        //修改子合同有效合同金额
                        conEpibolyCostConDDAO.updateSubConEffectiveAmt(newEffectiveAmt, conEpibolyCostConVO.getContractId());
                    }
                }
            });
            conEpibolyCostConDDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional
    public void submission(Long contractId, Long dId) {
        SaleConContractVO saleConContractVO = saleConContractService.queryByKey(contractId);
        if (!"active".equalsIgnoreCase(saleConContractVO.getStatus())) {
            throw TwException.error("", "子合同尚未激活，不能提交！");
        }
        //查询要提交的明细Id
        ConEpibolyCostConDVO dvo = this.queryByKey(dId);
        if (!ObjectUtils.isEmpty(dvo.getParentId())) {
            throw TwException.error("", "只有外包主明细才可以提交！");
        }
        // 只有新建的可以发起
        if (!WorkFlowStatusEnum.CREATE_WORK.getCode().equals(dvo.getDetailedStatus())) {
            throw TwException.error("", "只有新建的外包明细才可以提交！");
        }
        //走新增审批流程
        submitProc(dvo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ConEpibolyCostConDVO splitMoney(BigDecimal amt, Long parentId) {
        //父明细
        ConEpibolyCostConDVO dvo = conEpibolyCostConDDAO.queryByKey(parentId);
        if (!StringUtils.hasText(dvo.getDetailedStatus()) || "CREATE".equalsIgnoreCase(dvo.getDetailedStatus())) {
            throw TwException.error("", "明细尚未提交，不能按金额拆");
        }
        //子明细集合
        List<ConEpibolyCostConDVO> subDVOS = conEpibolyCostConDDAO.queryByParentId(parentId);
        //子明细不含税金额总和
        BigDecimal subAmtSum = BigDecimal.ZERO;
        for (ConEpibolyCostConDVO subDVO : subDVOS) {
            if (subDVO.getAmt() != null) {
                subAmtSum = subAmtSum.add(subDVO.getAmt());
            }
        }
        //如果左边（剩余的可拆分金额）小于右边（要拆分的金额）
        if (dvo.getAmt().subtract(subAmtSum).compareTo(amt) == -1) {
            throw TwException.error("", "子明细总金额不能超过主明细金额！");
        }
        //拆分明细
        ConEpibolyCostConDPayload subPayload = ConEpibolyCostConDConvert.INSTANCE.toPayload(dvo);
        subPayload.setId(null);
        subPayload.setParentId(parentId);
        subPayload.setAmt(amt);

        // 计算税费
        BigDecimal subTaxCost = amt.multiply(subPayload.getTaxRate());
        subPayload.setTaxCost(subTaxCost);

        // 计算净支付额
        subPayload.setNetPay(amt.add(subTaxCost));

        ConEpibolyCostConDDO save = conEpibolyCostConDRepo.save(ConEpibolyCostConDConvert.INSTANCE.toDo(subPayload));
        // 定时任务同步更新字段
        conEpibolyCostConRepo.updateRemark(dvo.getEpibolyCostConId());
        return ConEpibolyCostConDConvert.INSTANCE.toVo(save);
    }

    @Override
    public List<ConEpibolyCostConDVO> queryByParentId(Long parentId) {
        return conEpibolyCostConDDAO.queryByParentId(parentId);
    }

    /**
     * 发起新增审批流程
     *
     * @param costConDVO
     */
    private void submitProc(ConEpibolyCostConDVO costConDVO) {
        ProcessInfo processInfo = new ProcessInfo();
        String status = WorkFlowStatusEnum.APPROVED_WORK.getCode();
        if (workflow_enabled) {
            status = WorkFlowStatusEnum.APPROVING_WORK.getCode();
            HashMap<String, Object> variables = new HashMap<>();
            //发起流程审批
            processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                    ProcDefKey.SALE_CON_EC.name(),
                    costConDVO.getSaleConName() + "-销售合同外包费用审批流程",
                    costConDVO.getId() + "",
                    variables)
            );
        }
        //流程启动成功后，回写业务表数据
        ConEpibolyCostConDPayload payload = new ConEpibolyCostConDPayload();
        payload.setProcInstId(processInfo.getProcInstId());
        payload.setId(costConDVO.getId());
        payload.setProcInstStatus(processInfo.getProcInstStatus());
        payload.setSubmitTime(LocalDateTime.now());
        payload.setDetailedStatus(status);
        //开启事务执行修改，主要是修改审批状态
        transactionUtilService.executeWithRunnable(() -> {
            conEpibolyCostConDDAO.updateByKeyDynamic(payload);
        });
    }
}
