package com.elitesland.tw.tw5.server.prd.task.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.task.payload.PmsWbsTaskPayload;
import com.elitesland.tw.tw5.api.prd.task.query.PmsWbsTaskQuery;
import com.elitesland.tw.tw5.api.prd.task.vo.PmsWbsTaskVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QPmsProjectRoleDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsDO;
import com.elitesland.tw.tw5.server.prd.task.entity.PmsWbsTaskDO;
import com.elitesland.tw.tw5.server.prd.task.entity.QPmsWbsTaskDO;
import com.elitesland.tw.tw5.server.prd.task.repo.PmsWbsTaskRepo;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 节点任务
 *
 * @author xxb
 * @date 2023-07-11
 */
@Repository
@RequiredArgsConstructor
public class PmsWbsTaskDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsTaskRepo repo;
    private final QPmsWbsTaskDO qdo = QPmsWbsTaskDO.pmsWbsTaskDO;

    private final QPmsProjectDO projectQdo = QPmsProjectDO.pmsProjectDO;

    private final QPmsProjectWbsDO wbsQdo = QPmsProjectWbsDO.pmsProjectWbsDO;

    private final QPmsProjectRoleDO roleQdo = QPmsProjectRoleDO.pmsProjectRoleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTaskVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsTaskVO.class,
                        qdo.id,
                        qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 项目主键
                        qdo.projectId,
                        // wbs主键
                        qdo.wbsId,
                        // 发包人id
                        qdo.disterUserId,
                        // 接包人id
                        qdo.receiverUserId,
                        // 任务编号
                        qdo.taskCode,
                        // 任务名称
                        qdo.taskName,
                        // 任务状态
                        qdo.taskStatus,
                        // 任务开始时间
                        qdo.startDate,
                        // 任务结束时间
                        qdo.endDate,
                        // 派发人天
                        qdo.days,
                        // 角色主键
                        qdo.projectRoleId,
                        // 进度
                        qdo.progress,
                        // 权重
                        qdo.weight,
                        // 任务说明
                        qdo.taskExplain,
                        // 资源类型(labour人工、material材料、device设备)
                        qdo.sourceType,
                        // 交付文档
                        qdo.deliveryDocuments,
                        // 权重设置方式
                        qdo.manualSettingWeight,
                        // 项目名称
                        projectQdo.projName.as("projectName"),
                        // 节点名称
                        wbsQdo.wbsName,
                        //节点负责人
                        wbsQdo.managerUserId,
                        // 角色名称
                        roleQdo.roleName
                )).from(qdo).leftJoin(projectQdo).on(qdo.projectId.eq(projectQdo.id))
                .leftJoin(wbsQdo).on(qdo.wbsId.eq(wbsQdo.id))
                .leftJoin(roleQdo).on(qdo.projectRoleId.eq(roleQdo.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsTaskVO> getJpaQueryWhere(PmsWbsTaskQuery query) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsWbsTaskQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsWbsTaskQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 项目名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjectName())) {
            list.add(projectQdo.projName.like(SqlUtil.toSqlLikeString(query.getProjectName())));
        }
        /** wbs主键 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsId())) {
            list.add(qdo.wbsId.eq(query.getWbsId()));
        }
        /** 节点负责人 */
        if (!ObjectUtils.isEmpty(query.getManagerUserId())) {
            list.add(wbsQdo.managerUserId.eq(query.getManagerUserId()));
        }
        // 资源类型
        if (!ObjectUtils.isEmpty(query.getSourceType())) {
            list.add(qdo.sourceType.eq(query.getSourceType()));
        }
        /** 发包人id 精确 */
        if (!ObjectUtils.isEmpty(query.getDisterUserId())) {
            list.add(qdo.disterUserId.eq(query.getDisterUserId()));
        }
        /** 接包人id 精确 */
        if (!ObjectUtils.isEmpty(query.getReceiverUserId())) {
            list.add(qdo.receiverUserId.eq(query.getReceiverUserId()));
        }
        /** 任务编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getTaskCode())) {
            list.add(qdo.taskCode.like(SqlUtil.toSqlLikeString(query.getTaskCode())));
        }
        /** 任务名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getTaskName())) {
            list.add(qdo.taskName.like(SqlUtil.toSqlLikeString(query.getTaskName())));
        }
        /** 任务状态 0是不相等，1是相等 */
        if (ObjectUtils.isEmpty(query.getTaskStatusType())) {
            query.setTaskStatusType("1");
        }
        if (!ObjectUtils.isEmpty(query.getTaskStatus())) {
            if ("0".equals(query.getTaskStatusType())) {
                list.add(qdo.taskStatus.ne(query.getTaskStatus()));
            } else {
                list.add(qdo.taskStatus.eq(query.getTaskStatus()));
            }
        }
        if (!ObjectUtils.isEmpty(query.getTaskStatusList())) {
            if ("0".equals(query.getTaskStatusType())) {
                list.add(qdo.taskStatus.notIn(query.getTaskStatusList()));
            } else {
                list.add(qdo.taskStatus.in(query.getTaskStatusList()));
            }
        }
        /** 任务开始时间 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 任务结束时间 精确 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.eq(query.getEndDate()));
        }
        /** 派发人天 精确 */
        if (!ObjectUtils.isEmpty(query.getDays())) {
            list.add(qdo.days.eq(query.getDays()));
        }
        /** 角色主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectRoleId())) {
            list.add(qdo.projectRoleId.eq(query.getProjectRoleId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsTaskVO queryByKey(Long id) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键集合查询
     *
     * @param keys 主键集合
     * @return 结果
     */
    public List<PmsWbsTaskVO> queryByKeys(List<Long> keys) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(keys));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据节点集合 查找任务
     * @param wbsIds
     * @return
     */
    public List<PmsWbsTaskVO> findByWbsIds(List<Long> wbsIds) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.in(wbsIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据 wbsId 查询
     *
     * @param wbsId wbs主键
     * @return 结果
     */
    public List<PmsWbsTaskVO> queryByWbsId(Long wbsId) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.in(wbsId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsTaskVO> queryListDynamic(PmsWbsTaskQuery query) {
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsWbsTaskVO> queryPaging(PmsWbsTaskQuery query) {
//        long total = count(query);
//        if (total == 0) {
//            return PagingVO.empty();
//        }
        JPAQuery<PmsWbsTaskVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PmsWbsTaskVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetchResults();
        return PagingVO.<PmsWbsTaskVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsWbsTaskDO save(PmsWbsTaskDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsTaskDO> saveAll(List<PmsWbsTaskDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsWbsTaskPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // wbs主键
        if (payload.getWbsId() != null) {
            update.set(qdo.wbsId, payload.getWbsId());
        }
        // 资源类型
        if (payload.getSourceType() != null) {
            update.set(qdo.sourceType, payload.getSourceType());
        }
        // 发包人id
        if (payload.getDisterUserId() != null) {
            update.set(qdo.disterUserId, payload.getDisterUserId());
        }
        // 接包人id
        if (payload.getReceiverUserId() != null) {
            update.set(qdo.receiverUserId, payload.getReceiverUserId());
        }
        // 任务编号
        if (payload.getTaskCode() != null) {
            update.set(qdo.taskCode, payload.getTaskCode());
        }
        // 任务名称
        if (payload.getTaskName() != null) {
            update.set(qdo.taskName, payload.getTaskName());
        }
        // 任务状态
        if (payload.getTaskStatus() != null) {
            update.set(qdo.taskStatus, payload.getTaskStatus());
        }
        // 任务开始时间
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 任务结束时间
        if (payload.getEndDate() != null) {
            update.set(qdo.endDate, payload.getEndDate());
        }
        // 派发人天
        if (payload.getDays() != null) {
            update.set(qdo.days, payload.getDays());
        }
        // 角色主键
        if (payload.getProjectRoleId() != null) {
            update.set(qdo.projectRoleId, payload.getProjectRoleId());
        }
        // 进度
        if (payload.getProgress() != null) {
            update.set(qdo.progress, payload.getProgress());
        }
        // 权重
        if (payload.getWeight() != null) {
            update.set(qdo.weight, payload.getWeight());
        }
        // 任务说明
        if (payload.getTaskExplain() != null) {
            update.set(qdo.taskExplain, payload.getTaskExplain());
        }
        // 交付文档
        if (payload.getDeliveryDocuments() != null) {
            update.set(qdo.deliveryDocuments, payload.getDeliveryDocuments());
        }
        // 备注
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // wbs主键
            if (nullFields.contains("wbsId")) {
                update.setNull(qdo.wbsId);
            }
            // 发包人id
            if (nullFields.contains("disterUserId")) {
                update.setNull(qdo.disterUserId);
            }
            // 接包人id
            if (nullFields.contains("receiverUserId")) {
                update.setNull(qdo.receiverUserId);
            }
            // 任务编号
            if (nullFields.contains("taskCode")) {
                update.setNull(qdo.taskCode);
            }
            // 任务名称
            if (nullFields.contains("taskName")) {
                update.setNull(qdo.taskName);
            }
            // 任务状态
            if (nullFields.contains("taskStatus")) {
                update.setNull(qdo.taskStatus);
            }
            // 任务开始时间
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 任务结束时间
            if (nullFields.contains("endDate")) {
                update.setNull(qdo.endDate);
            }
            // 派发人天
            if (nullFields.contains("days")) {
                update.setNull(qdo.days);
            }
            // 角色主键
            if (nullFields.contains("projectRoleId")) {
                update.setNull(qdo.projectRoleId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param wbsIds 节点集合
     * @return 删除的行数
     */
    public long deleteSoftByWbsIds(List<Long> wbsIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.wbsId.in(wbsIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更改任务状态
     *
     * @param keys
     * @param taskStatus
     */
    public long updateTaskStatus(List<Long> keys, String taskStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.taskStatus, taskStatus)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更改任务状态
     *
     * @param keys
     * @param taskStatus
     */
    public long updateTaskStatus(List<Long> keys, String taskStatus, BigDecimal progress) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.taskStatus, taskStatus).set(qdo.progress, progress)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 编辑权重
     *
     * @param key
     * @param weight
     */
    public long updateTaskWeight(Long key,BigDecimal weight) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.weight,weight).set(qdo.manualSettingWeight,1)
                .where(qdo.id.eq(key));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 指派任务
     *
     * @param keys
     * @param taskStatus
     */
    public long updateTaskStatus(List<Long> keys, String taskStatus, long disterUserId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.taskStatus, taskStatus).set(qdo.disterUserId, disterUserId)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

