package com.elitesland.tw.tw5.server.common.funConfig.dao;

import com.elitesland.tw.tw5.api.common.funConfig.payload.BusinessDocTypeRefPayload;
import com.elitesland.tw.tw5.api.common.funConfig.query.BusinessDocTypeRefQuery;
import com.elitesland.tw.tw5.api.common.funConfig.vo.BusinessDocTypeRefVO;
import com.elitesland.tw.tw5.server.common.funConfig.entity.BusinessDocTypeRefDO;
import com.elitesland.tw.tw5.server.common.funConfig.entity.QBusinessDocTypeRefDO;
import com.elitesland.tw.tw5.server.common.funConfig.repo.BusinessDocTypeRefRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 易稻壳yeedoc通用文档类型配置与业务关系表
 *
 * @author duwh
 * @date 2023-12-08
 */
@Repository
@RequiredArgsConstructor
public class BusinessDocTypeRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BusinessDocTypeRefRepo repo;
    private final QBusinessDocTypeRefDO qdo = QBusinessDocTypeRefDO.businessDocTypeRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessDocTypeRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BusinessDocTypeRefVO.class,
            qdo.id,
            qdo.remark,
            qdo.createUserId,
            qdo.creator,
            qdo.createTime,
            qdo.modifyUserId,
            qdo.updater,
            qdo.modifyTime,
            qdo.libraryId,
            // 文件夹id -yeedoc返回
            qdo.folderId,
            // 文件夹名称 LeafName -yeedoc返回
            qdo.folderName,
            // 单据主键
            qdo.docId,
            // 单据类型
            qdo.docType,
            // 文档类型配置主键 prd_business_doc_type_config.id
            qdo.configId,
            // 文档类型名称（冗余）prd_business_doc_type_config.name
            qdo.configName,
            // prd_business_doc_type_config_rule.id
            qdo.configRuleId,
            // 拓展字段1
            qdo.ext1,
            // 拓展字段2
            qdo.ext2,
            // 拓展字段3
            qdo.ext3
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BusinessDocTypeRefVO> getJpaQueryWhere(BusinessDocTypeRefQuery query) {
        JPAQuery<BusinessDocTypeRefVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BusinessDocTypeRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BusinessDocTypeRefQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 易稻壳libraryId -yeedoc返回 精确 */
        if (!ObjectUtils.isEmpty(query.getLibraryId())) {
            list.add(qdo.libraryId.eq(query.getLibraryId()));
        }
        /** 文件夹id -yeedoc返回 精确 */
        if (!ObjectUtils.isEmpty(query.getFolderId())) {
            list.add(qdo.folderId.eq(query.getFolderId()));
        }
        /** 文件夹名称 LeafName -yeedoc返回 精确 */
        if (!ObjectUtils.isEmpty(query.getFolderName())) {
            list.add(qdo.folderName.eq(query.getFolderName()));
        }
        /** 单据主键 精确 */
        if (!ObjectUtils.isEmpty(query.getDocId())) {
            list.add(qdo.docId.eq(query.getDocId()));
        }
        /** 单据类型 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdo.docType.eq(query.getDocType()));
        }
        /** 文档类型配置主键 prd_business_doc_type_config.id 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigId())) {
            list.add(qdo.configId.eq(query.getConfigId()));
        }
        /** 文档类型名称（冗余）prd_business_doc_type_config.name 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigName())) {
            list.add(qdo.configName.eq(query.getConfigName()));
        }
        /** prd_business_doc_type_config_rule.id 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigRuleId())) {
            list.add(qdo.configRuleId.eq(query.getConfigRuleId()));
        }
        /** 拓展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 拓展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 拓展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BusinessDocTypeRefVO queryByKey(Long id) {
        JPAQuery<BusinessDocTypeRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BusinessDocTypeRefVO> queryListDynamic(BusinessDocTypeRefQuery query) {
        JPAQuery<BusinessDocTypeRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BusinessDocTypeRefVO> queryPaging(BusinessDocTypeRefQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BusinessDocTypeRefVO> jpaQuery = getJpaQueryWhere(query);
        List<BusinessDocTypeRefVO> result = jpaQuery
            .offset(query.getPageRequest().getOffset())
            .limit(query.getPageRequest().getPageSize())
            .fetch();
        return PagingVO.<BusinessDocTypeRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BusinessDocTypeRefDO save(BusinessDocTypeRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BusinessDocTypeRefDO> saveAll(List<BusinessDocTypeRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BusinessDocTypeRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 文件夹id -yeedoc返回
        if (payload.getFolderId() != null) {
            update.set(qdo.folderId, payload.getFolderId());
        }
        // 文件夹名称 LeafName -yeedoc返回
        if (payload.getFolderName() != null) {
            update.set(qdo.folderName, payload.getFolderName());
        }
        // 单据主键
        if (payload.getDocId() != null) {
            update.set(qdo.docId, payload.getDocId());
        }
        // 单据类型
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 文档类型配置主键 prd_business_doc_type_config.id
        if (payload.getConfigId() != null) {
            update.set(qdo.configId, payload.getConfigId());
        }
        // 文档类型名称（冗余）prd_business_doc_type_config.name
        if (payload.getConfigName() != null) {
            update.set(qdo.configName, payload.getConfigName());
        }
        // prd_business_doc_type_config_rule.id
        if (payload.getConfigRuleId() != null) {
            update.set(qdo.configRuleId, payload.getConfigRuleId());
        }
        // 拓展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 拓展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 拓展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 文件夹id -yeedoc返回
            if (nullFields.contains("folderId")) {
                update.setNull(qdo.folderId);
            }
            // 文件夹名称 LeafName -yeedoc返回
            if (nullFields.contains("folderName")) {
                update.setNull(qdo.folderName);
            }
            // 单据主键
            if (nullFields.contains("docId")) {
                update.setNull(qdo.docId);
            }
            // 单据类型
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 文档类型配置主键 prd_business_doc_type_config.id
            if (nullFields.contains("configId")) {
                update.setNull(qdo.configId);
            }
            // 文档类型名称（冗余）prd_business_doc_type_config.name
            if (nullFields.contains("configName")) {
                update.setNull(qdo.configName);
            }
            // prd_business_doc_type_config_rule.id
            if (nullFields.contains("configRuleId")) {
                update.setNull(qdo.configRuleId);
            }
            // 拓展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 拓展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 拓展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 按单据类型、单据id和规则id获取
     *
     * @param docType      单据类型
     * @param docId        文档id
     * @param configRuleId 配置规则id
     * @return {@link BusinessDocTypeRefVO}
     */
    public BusinessDocTypeRefVO getByDocTypeAndDocIdAndRuleId(String docType, Long docId, Long configRuleId) {
        return getJpaQuerySelect().where(qdo.docType.eq(docType), qdo.docId.eq(docId), qdo.configRuleId.eq(configRuleId)).fetchOne();
    }
}

