package com.elitesland.tw.tw5.server.common.permission;

import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemBusinessObjectPayload;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemNewFunctionPayload;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemPermissionFieldPayload;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemPermissionFunctionObjectPayload;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemBusinessObjectService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemNewFunctionService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemPermissionFunctionObjectService;
import com.elitesland.tw.tw5.server.common.permission.annotation.FunctionDetail;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionDomain;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionFunction;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionDomainEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionFieldType;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionMappingFieldEnum;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.annotation.Order;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RestController;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.time.temporal.Temporal;
import java.util.*;

/**
 * @author : JS
 * @date 2023/09/21
 */
@Component
@Order
@Slf4j
@RequiredArgsConstructor
public class PermissionApplicationAware implements ApplicationContextAware, ApplicationRunner {

    private ApplicationContext applicationContext;

    private final PrdSystemBusinessObjectService businessObjectService;

    private final PrdSystemNewFunctionService prdSystemNewFunctionService;

    private final PrdSystemPermissionFunctionObjectService prdSystemPermissionFunctionObjectService;

    private final StringRedisTemplate stringRedisTemplate;

    private static final String DEL_KEYS = "*sys:permission*";

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public void parsePermissionDomain() {

        // 清除缓存
        clearCache();

        Map<String, Object> permissionBeanMap = applicationContext.getBeansWithAnnotation(PermissionDomain.class);

        Map<String, List<PrdSystemNewFunctionPayload>> functionMap = new HashMap<>();
        Map<String, PrdSystemBusinessObjectPayload> businessObjectMap = new HashMap<>();
        Map<String, List<PrdSystemPermissionFunctionObjectPayload>> functionObjectMap = new HashMap<>();

        permissionBeanMap.forEach((key, value) -> {
            Class<?> clazz;
            try {
                clazz = Class.forName(key);
            } catch (ClassNotFoundException e) {
                log.error("获取类对象【" + key + "】失败", e);
                return;
            }
            // 解析接口类
            if (!clazz.isAnnotationPresent(RestController.class)) {
                return;
            }
            PermissionDomain permissionDomain = clazz.getAnnotation(PermissionDomain.class);
            PermissionDomainEnum domainEnum = permissionDomain.domain();
            if (!businessObjectMap.containsKey(domainEnum.name())) {
                PrdSystemBusinessObjectPayload businessObject = new PrdSystemBusinessObjectPayload();
                businessObject.setObjectCode(domainEnum.name());
                businessObject.setObjectName(domainEnum.getDesc());
                businessObjectMap.put(domainEnum.name(), businessObject);
            }

            // 解析方法
            Method[] declaredMethods = clazz.getDeclaredMethods();

            for (Method method : declaredMethods) {
                List<PrdSystemPermissionFunctionObjectPayload> functionObjectPayloadList = new ArrayList<>();
                List<PrdSystemNewFunctionPayload> functionPayload = parseFunctionInfo(clazz.getSimpleName(), method, functionObjectPayloadList);
//                if (functionPayload.isEmpty()) continue;

                functionMap.computeIfAbsent(domainEnum.name(), t -> new ArrayList<>()).addAll(functionPayload);
                functionObjectMap.computeIfAbsent(domainEnum.name(), t -> new ArrayList<>()).addAll(functionObjectPayloadList);
            }

        });

        // 持久化
        if (!CollectionUtils.isEmpty(businessObjectMap)) {
            businessObjectService.autoCreate(businessObjectMap);
        }
        if (!CollectionUtils.isEmpty(functionMap)) {
            prdSystemNewFunctionService.autoCreate(functionMap);
        }
        if (!CollectionUtils.isEmpty(functionObjectMap)) {
            prdSystemPermissionFunctionObjectService.autoCreate(functionObjectMap);
        }
    }

    private List<PrdSystemNewFunctionPayload> parseFunctionInfo(String className, Method method, List<PrdSystemPermissionFunctionObjectPayload> functionObjectPayloadList) {
        // 是否功能权限控制
        if (!method.isAnnotationPresent(PermissionFunction.class)) {
            return new ArrayList<>();
        }
        PermissionFunction permissionFunction = method.getAnnotation(PermissionFunction.class);

        FunctionDetail[] functions = permissionFunction.functions();

        List<String> functionCodes = new ArrayList<>();
        List<PrdSystemNewFunctionPayload> funcPayloads = new ArrayList<>();
        for (int i = 0; i < functions.length; i++) {

            PrdSystemNewFunctionPayload funcPayload = new PrdSystemNewFunctionPayload();

            funcPayload.setFunctionCode(functions[i].functionCode().name());
            funcPayload.setFunctionName(!"".equals(functions[i].name()) ? functions[i].name() : functions[i].functionCode().getDesc());
            funcPayload.setFunctionType(functions[i].type().name());
            funcPayload.setFunctionStatus(0);
            funcPayload.setClientType(functions[i].clientType().name());
            if(functions[i].referDomain() != null && functions[i].referDomain().length != 0){
                // 外部引用功能
                List<String> referDomainCodes = Arrays.stream(functions[i].referDomain()).map(Enum::name).toList();
                funcPayload.setReferDomainCodes(referDomainCodes);

                String functionCode = functions[i].functionCode().name();
                List<String> referFunctionCodes = referDomainCodes.stream().map(item -> item + "/" + functionCode).toList();
                functionCodes.addAll(referFunctionCodes);
            }
            funcPayloads.add(funcPayload);

            functionCodes.add(functions[i].functionCode().name());
        }
        // 字段权限
        if (permissionFunction.fieldPermission()) {
            Type genericReturnType = method.getGenericReturnType();
            Class<?> baseClass = getBaseClass(genericReturnType);

            PrdSystemPermissionFunctionObjectPayload functionObjectPayload = new PrdSystemPermissionFunctionObjectPayload();

            functionObjectPayload.setClassName(baseClass.getName());
            functionObjectPayload.setFunctionCodes(functionCodes);
            functionObjectPayload.setPermissionType(permissionFunction.permissionType().getName());

            List<PrdSystemPermissionFieldPayload> fieldPayloadList = new ArrayList<>();
            Field[] declaredFields = baseClass.getDeclaredFields();
            for (Field field : declaredFields) {
                if (field.isAnnotationPresent(ApiModelProperty.class)) {
                    PrdSystemPermissionFieldPayload fieldPayload = new PrdSystemPermissionFieldPayload();
                    String fieldDesc = field.getAnnotation(ApiModelProperty.class).value();

                    fieldPayload.setField(field.getName());
                    fieldPayload.setFieldName(fieldDesc);

                    String fieldType = PermissionFieldType.NORMAL.getName();
                    if (field.getName().toLowerCase().contains("userid") || field.getName().toLowerCase().contains("resid")) {
                        fieldType = PermissionFieldType.BUSINESS_USER.getName();
                    } else if (field.getName().toLowerCase().contains("orgid") || field.getName().toLowerCase().contains("buid")) {
                        fieldType = PermissionFieldType.BUSINESS_DEPT.getName();
                    }

                    getMappingField(fieldPayload, field);

//                    目前只区分普通属性 用户属性 部门属性
//                    else if (Temporal.class.isAssignableFrom(field.getType()) || Date.class.isAssignableFrom(field.getType())) {
//                        fieldType = PermissionFieldType.DATE.getName();
//                    }
                    fieldPayload.setFieldType(fieldType);

                    fieldPayloadList.add(fieldPayload);
                }
            }
            functionObjectPayload.setFields(fieldPayloadList);
            functionObjectPayloadList.add(functionObjectPayload);
        }

        return funcPayloads;
    }

    /**
     * 获取映射数据库字段
     *
     * @param fieldPayload
     * @param field
     */
    private void getMappingField(PrdSystemPermissionFieldPayload fieldPayload, Field field) {
        Class<?> type = field.getType();
        String mappingField = "";
        if (String.class.isAssignableFrom(type)) {
            mappingField = PermissionMappingFieldEnum.VARCAHR.name();
        }else if (Integer.class.isAssignableFrom(type) || int.class.isAssignableFrom(type) || Boolean.class.isAssignableFrom(type) || boolean.class.isAssignableFrom(type)) {
            mappingField = PermissionMappingFieldEnum.INT.name();
        }else if (Long.class.isAssignableFrom(type) || long.class.isAssignableFrom(type)) {
            mappingField = PermissionMappingFieldEnum.BIGINT.name();
        }else if (BigDecimal.class.isAssignableFrom(type)) {
            mappingField = PermissionMappingFieldEnum.DECIMAL.name();
        }else if (Temporal.class.isAssignableFrom(field.getType()) || Date.class.isAssignableFrom(field.getType())) {
            mappingField = PermissionMappingFieldEnum.DATETIME.name();
        }
        fieldPayload.setMappingField(mappingField);
    }

    private Class<?> getBaseClass(Type type) {
        if (type instanceof ParameterizedType parameterizedType) {
            return getBaseClass(parameterizedType.getActualTypeArguments()[0]);
        }
        return (Class<?>) type;
    }

    @Override
    public void run(ApplicationArguments args) {
        this.parsePermissionDomain();
    }

    /**
     * 清理缓存
     *
     */
    private void clearCache() {
        Set<String> keys = stringRedisTemplate.keys(DEL_KEYS);
        if (!CollectionUtils.isEmpty(keys)) {
            stringRedisTemplate.delete(keys);
        }
    }

}
