package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimTaxPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimTaxQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimExportVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimTaxVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.*;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimTaxRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 费用报销税额管理
 *
 * @author sunxw
 * @date 2023-12-05
 */
@Repository
@RequiredArgsConstructor
public class AccReimTaxDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimTaxRepo repo;
    private final QAccReimTaxDO qdo = QAccReimTaxDO.accReimTaxDO;
    private final QAccReimDO qMasdo = QAccReimDO.accReimDO;
    private final QAccReimDetailDO qDtldo = QAccReimDetailDO.accReimDetailDO;
    private final QAccFinancialSubjectDO qFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;
    private final QAccBusinessItemDO qBusinessItemDO = QAccBusinessItemDO.accBusinessItemDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimTaxVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimTaxVO.class,
                qdo.id,
                qMasdo.id.as("reimId"),
                qMasdo.remark,
                qMasdo.createUserId,
                qMasdo.creator,
                qMasdo.createTime,
                qMasdo.modifyUserId,
                qMasdo.updater,
                qMasdo.modifyTime,
                // 报销单号
                qMasdo.reimNo,
                // 报销人RES_ID
                qMasdo.reimUserId,
                // 报销人BU_ID
                qMasdo.reimOrgId,
                // 报销人职级
                qMasdo.reimResGrade,
                qMasdo.reasonType,
                // 单据类型
                qMasdo.reimDocType,
                // 相关申请单
                qMasdo.relatedDocId,
                qMasdo.relatedDocName,
                // 相关预算
                qMasdo.relatedBudgetId,
                // 费用承担项目
                qMasdo.expenseProjectId,
                // 费用承担部门
                qMasdo.expenseOrgId,
                // 费用承担公司
                qMasdo.expenseCompany,
                qMasdo.expenseByType,
                // 记账日期
                qMasdo.accountingDate,
                // 报销单状态
                qMasdo.reimStatus,
                qMasdo.reimTotalAmt,
                qMasdo.adjustAmt,
                // 事由号
                qMasdo.reasonId,
                // 事由名称
                qMasdo.reasonName,
                // 付款批次号
                qMasdo.batchNo,
                // 付款方式
                qMasdo.payMode,
                qMasdo.finChargeUpTime,
                qMasdo.fileCode,
                qMasdo.applyDate,
                qMasdo.apprProcName,
                qMasdo.finPicApprTime,
                qMasdo.bankTime,
                qMasdo.apprStatus,
                qMasdo.contractId,
                qMasdo.procInstId,
                qMasdo.procInstName,
                qMasdo.finPeriodId,
                qDtldo.id.as("reimDtlId"),
                qDtldo.expenseDate,
                qDtldo.reimAmt,
                qDtldo.taxAmt,
                qDtldo.taxRate,
                qDtldo.reimRemark,
                qdo.finAccSubjId,
                qFinancialSubjectDO.accName.as("finAccSubjName")
        )).from(qdo).leftJoin(qMasdo).on(qdo.reimId.eq(qMasdo.id))
                .leftJoin(qDtldo).on(qDtldo.id.eq(qdo.reimDtlId).and(qDtldo.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qdo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimTaxVO> getJpaQueryWhere(AccReimTaxQuery query) {
        JPAQuery<AccReimTaxVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimTaxQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .leftJoin(qMasdo).on(qdo.reimId.eq(qMasdo.id))
                .leftJoin(qDtldo).on(qDtldo.id.eq(qdo.reimDtlId).and(qDtldo.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qdo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimTaxQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** ID集合 精确 */
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 报销单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimId())) {
            list.add(qdo.reimId.eq(query.getReimId()));
        }
        /** 报销单明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimDtlId())) {
            list.add(qdo.reimDtlId.eq(query.getReimDtlId()));
        }
        /** 会计科目 精确 */
        if (!ObjectUtils.isEmpty(query.getFinAccSubjId())) {
            list.add(qdo.finAccSubjId.eq(query.getFinAccSubjId()));
        }
        /** 税率 精确 */

        list.add(qdo.taxRate.gt(0));
//        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
//            list.add(qdo.taxRate.eq(query.getTaxRate()));
//        }
        /** 税额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAmt())) {
            list.add(qdo.taxAmt.eq(query.getTaxAmt()));
        }
        if (!ObjectUtils.isEmpty(query.getReimIdList())) {
            list.add(qdo.reimId.in(query.getReimIdList()));
        }
        //报销单号
        if (!ObjectUtils.isEmpty(query.getReimNo())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getReimNo());
            list.add(qMasdo.reimNo.like(likeStr));
        }
        //报销申请人
        if (!ObjectUtils.isEmpty(query.getReimUserId())) {
            list.add(qMasdo.reimUserId.eq(query.getReimUserId()));
        }
        //报销单据类型
        if (!ObjectUtils.isEmpty(query.getReimDocType())) {
            list.add(qMasdo.reimDocType.eq(query.getReimDocType()));
        }
        //财务记账日期
        if (!ObjectUtils.isEmpty(query.getAccountingDateStart()) && !ObjectUtils.isEmpty(query.getAccountingDateEnd())) {
            list.add(qMasdo.accountingDate.goe(query.getAccountingDateStart()).and(qMasdo.accountingDate.loe(query.getAccountingDateEnd())));
        }
        //记账批次号
        if (!ObjectUtils.isEmpty(query.getBatchNo())) {
            String likeStr = SqlUtil.toSqlLikeString(query.getBatchNo());
            list.add(qMasdo.batchNo.like(likeStr));
        }
        //费用所属公司
        if (!ObjectUtils.isEmpty(query.getExpenseCompany())) {
            list.add(qMasdo.expenseCompany.eq(query.getExpenseCompany()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimTaxVO queryByKey(Long id) {
        JPAQuery<AccReimTaxVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimTaxVO> queryListDynamic(AccReimTaxQuery query) {
        JPAQuery<AccReimTaxVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimTaxVO> queryPaging(AccReimTaxQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimTaxVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimTaxVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimTaxVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimTaxDO save(AccReimTaxDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimTaxDO> saveAll(List<AccReimTaxDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimTaxPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销单ID
        if (payload.getReimId() != null) {
            update.set(qdo.reimId, payload.getReimId());
        }
        // 报销单明细ID
        if (payload.getReimDtlId() != null) {
            update.set(qdo.reimDtlId, payload.getReimDtlId());
        }
        // 会计科目
        if (payload.getFinAccSubjId() != null) {
            update.set(qdo.finAccSubjId, payload.getFinAccSubjId());
        }
        // 税率
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 税额
        if (payload.getTaxAmt() != null) {
            update.set(qdo.taxAmt, payload.getTaxAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销单ID
            if (nullFields.contains("reimId")) {
                update.setNull(qdo.reimId);
            }
            // 报销单明细ID
            if (nullFields.contains("reimDtlId")) {
                update.setNull(qdo.reimDtlId);
            }
            // 会计科目
            if (nullFields.contains("finAccSubjId")) {
                update.setNull(qdo.finAccSubjId);
            }
            // 税率
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 税额
            if (nullFields.contains("taxAmt")) {
                update.setNull(qdo.taxAmt);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据报销单id删除
     *
     * @param reimId
     */
    public long delByReimId(Long reimId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.reimId.eq(reimId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 获取税额导出数据
     *
     * @param idList
     * @return
     */
    public List<AccReimExportVO> getExportTaxData(Set<Long> idList) {
        JPAQuery<AccReimExportVO> query = jpaQueryFactory.selectDistinct(Projections.bean(AccReimExportVO.class,
                qMasdo.id,
                qMasdo.expenseCompany,
                qMasdo.reasonType,
                qMasdo.reasonId,
                qMasdo.reasonName,
                qMasdo.reimNo,
                qMasdo.reimUserId,
                qMasdo.reimDocType,
                qMasdo.reimOrgId,
                qMasdo.expenseOrgId,
                qMasdo.expenseProjectId,
                qdo.taxAmt.as("reimAmt"),
                qdo.finAccSubjId,
                qdo.id.as("detailId"),
                 //核算科目
                qDtldo.busAccItemId,
                qBusinessItemDO.businessName.as("busAccItemName"),
                qDtldo.expenseDate,
                qFinancialSubjectDO.accCode.as("finAccSubjCode"),
                qFinancialSubjectDO.accName,
                qDtldo.reimRemark.as("dtlReimRemark"),
                qDtldo.invoiceNum
        )).from(qdo).leftJoin(qMasdo).on(qdo.reimId.eq(qMasdo.id))
                .leftJoin(qDtldo).on(qDtldo.id.eq(qdo.reimDtlId).and(qDtldo.deleteFlag.eq(0)))
                .leftJoin(qFinancialSubjectDO).on(qdo.finAccSubjId.eq(qFinancialSubjectDO.id).and(qFinancialSubjectDO.deleteFlag.eq(0)))
                .leftJoin(qBusinessItemDO).on(qDtldo.busAccItemId.eq(qBusinessItemDO.id).and(qBusinessItemDO.deleteFlag.eq(0)))
                .where(qdo.reimId.in(idList));
        return query.fetch();
    }
}

