package com.elitesland.tw.tw5.server.prd.crm.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOpportunityMeasureDPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmOpportunityMeasurePayload;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOpportunityMeasureDService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmOpportunityMeasureService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOpportunityMeasureDVO;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmOpportunityMeasureVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmOpportunityMeasureConvert;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmOpportunityMeasureDConvert;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmOpportunityMeasureDAO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmOpportunityMeasureDDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmOpportunityMeasureDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmOpportunityMeasureDRepo;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmOpportunityMeasureRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 商机成单率计算指标
 *
 * @author likunpeng
 * @date 2023-11-03
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmOpportunityMeasureServiceImpl extends BaseServiceImpl implements CrmOpportunityMeasureService {

    private final CrmOpportunityMeasureDService crmOpportunityMeasureDService;
    private final CrmOpportunityMeasureDAO crmOpportunityMeasureDAO;
    private final CrmOpportunityMeasureRepo crmOpportunityMeasureRepo;
    private final CrmOpportunityMeasureDRepo crmOpportunityMeasureDRepo;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<CrmOpportunityMeasureVO> insertOrUpdate(List<CrmOpportunityMeasurePayload> payloads) {
        // 计算指标明细
        List<CrmOpportunityMeasureDDO> crmOpportunityMeasureDDOList = new ArrayList<>();
        // 总权重
        final BigDecimal[] totalWeight = {BigDecimal.ZERO};
        // 数据源字段
        Set<Long> dataSourceFieldsIdsSet = new HashSet<>(payloads.size());
        payloads.forEach(crmOpportunityMeasurePayload -> {
            // 校验字段是否唯一
            if (!dataSourceFieldsIdsSet.add(crmOpportunityMeasurePayload.getDataSourceFieldsId())) {
                throw TwException.error("", "计算指标-数据源字段不唯一，请检查");
            }
            // 校验权重
            BigDecimal weights = crmOpportunityMeasurePayload.getWeights() == null
                    ? BigDecimal.ZERO : crmOpportunityMeasurePayload.getWeights();
            totalWeight[0] = totalWeight[0].add(weights);
            CrmOpportunityMeasureDO crmOpportunityMeasureDO = CrmOpportunityMeasureConvert.INSTANCE.toDo(crmOpportunityMeasurePayload);
            CrmOpportunityMeasureDO save = crmOpportunityMeasureRepo.save(crmOpportunityMeasureDO);
            // 详情
            List<CrmOpportunityMeasureDPayload> crmOpportunityMeasureDPayloads = crmOpportunityMeasurePayload.getCrmOpportunityMeasureDPayloads();
            // 数值区间
            List<BigDecimal[]> numericalValueInterval = new ArrayList<>(crmOpportunityMeasureDPayloads.size());
            crmOpportunityMeasureDPayloads.forEach(crmOpportunityMeasureDPayload -> {
                CrmOpportunityMeasureDDO crmOpportunityMeasureDDO = CrmOpportunityMeasureDConvert.INSTANCE.toDo(crmOpportunityMeasureDPayload);
                if ("数值类型".equals(crmOpportunityMeasureDDO.getIndicatorType())) {
                    BigDecimal numericalValueMin = crmOpportunityMeasureDDO.getNumericalValueMin();
                    BigDecimal numericalValueMax = crmOpportunityMeasureDDO.getNumericalValueMax();
                    // 校验数值类型的数值区间合法性
                    if (numericalValueMin == null || numericalValueMax == null) {
                        throw TwException.error("", "数据源字段【" + crmOpportunityMeasurePayload.getOppoTableFieldsName() + "】存在数值区间为空，请检查");
                    } else if (numericalValueMin.compareTo(numericalValueMax) >= 0) {
                        throw TwException.error("", "数据源字段【" + crmOpportunityMeasurePayload.getOppoTableFieldsName() + "】存在数值区间不合法，请检查");
                    }
                    numericalValueInterval.add(new BigDecimal[]{numericalValueMin, numericalValueMax});
                }
                BigDecimal score = crmOpportunityMeasureDDO.getScore() == null
                        ? BigDecimal.ZERO : crmOpportunityMeasureDDO.getScore();
                // 校验分值
                if (score.compareTo(BigDecimal.ZERO) < 0 || score.compareTo(new BigDecimal(100)) > 0) {
                    throw TwException.error("", "数据源字段【" + crmOpportunityMeasurePayload.getOppoTableFieldsName() + "】配置的选项分值应为0~100，请检查");
                }
                crmOpportunityMeasureDDO.setMeasureId(save.getId());
                crmOpportunityMeasureDDOList.add(crmOpportunityMeasureDDO);
            });
            // 校验数值类型的数值区间是否有交叉
            numericalValueInterval.sort(Comparator.comparing(a -> a[0])); // 对列表排序
            for (int i = 0; i < numericalValueInterval.size() - 1; i++) {
                BigDecimal[] interval1 = numericalValueInterval.get(i);
                BigDecimal[] interval2 = numericalValueInterval.get(i + 1);
                if (interval1[0].compareTo(interval2[1]) < 0 && interval2[0].compareTo(interval1[1]) < 0) {
                    throw TwException.error("", "数据源字段【" + crmOpportunityMeasurePayload.getOppoTableFieldsName() + "】存在交叉的数值区间，请检查");
                }
            }
        });
        // 校验总权重
        if (totalWeight[0].compareTo(new BigDecimal(100)) != 0) {
            throw TwException.error("", "计算指标的总权重应为100，请检查");
        }
        // 删除计算指标明细
        crmOpportunityMeasureDRepo.deleteAll();
        // 保存计算指标明细
        crmOpportunityMeasureDRepo.saveAll(crmOpportunityMeasureDDOList);

        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long deleteSoft(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        // 删除明细
        crmOpportunityMeasureDService.deleteSoftByMeasureId(ids);
        return crmOpportunityMeasureDAO.deleteSoft(ids);
    }

    @Override
    public List<CrmOpportunityMeasureVO> queryAll() {
        // 查询全部
        List<CrmOpportunityMeasureVO> crmOpportunityMeasureVOS = crmOpportunityMeasureDAO.queryAll();
        // 查询详情的全部
        List<CrmOpportunityMeasureDVO> crmOpportunityMeasureDVOS = crmOpportunityMeasureDService.queryAll();
        // key 成单率指标id value 详情list
        Map<Long, List<CrmOpportunityMeasureDVO>> map = crmOpportunityMeasureDVOS.stream().collect(Collectors.groupingBy(CrmOpportunityMeasureDVO::getMeasureId));
        crmOpportunityMeasureVOS.forEach(crmOpportunityMeasureVO -> {
            crmOpportunityMeasureVO.setCrmOpportunityMeasureDVOS(map.get(crmOpportunityMeasureVO.getId()));
        });
        return crmOpportunityMeasureVOS;
    }
}
