package com.elitesland.tw.tw5.server.prd.personcare.service;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.tw.tw5.server.prd.personcare.dto.PersonCareDTO;
import com.elitesland.tw.tw5.server.prd.personcare.repo.PersonCareRepo;
import com.xxl.job.core.log.XxlJobLogger;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.mail.MailProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ResourceUtils;

import javax.imageio.ImageIO;
import javax.mail.MessagingException;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 员工关怀
 *
 * @author duwh
 * @date 2021/10/25
 */

@Service
@RequiredArgsConstructor
@Slf4j
// 配置文件动态刷新
@RefreshScope
public class PersonCareServiceImpl implements PersonCareService {

    private final MailProperties mailProperties;
    private final JavaMailSender javaMailSender;
    private final PersonCareRepo personCareRepo;

    @Value("${tw5.mail.enabled:false}")
    private Boolean enabled;

    /**
     * 发送生日祝福邮件
     */
    @Override
    public int taskSendBirthEmailMessage() {
        // 拉取过生日的人员
        List<PersonCareDTO> prdOrgPersonDOS = personCareRepo.listPersonBirthday();
        AtomicInteger result = new AtomicInteger();
        if (!CollectionUtils.isEmpty(prdOrgPersonDOS)) {
            prdOrgPersonDOS.forEach(personCareView -> {
                try {
                    File temp = MarkImageUtils.markMain(personCareView.getPersonName());
                    log.info("【MESSAGE-EMAIL】 开始发送邮件： {}...");
                    MimeMessageHelper helper = messageHelper(true);
                    List<String> receiveEmails = new ArrayList<>();
                    String email = personCareView.getEmail();
                    if (ObjectUtil.isNotEmpty(email)) {
                        receiveEmails.add(email);
                    }
                    if (receiveEmails.size() < 1) {
                        XxlJobLogger.log("[员工关怀邮件发送]  邮件发送失败，收件人邮箱不能为空!");
                        throw new RuntimeException("邮件发送失败，收件人邮箱不能为空!");
                    }
                    String[] toArray = new String[receiveEmails.size()];
                    helper.setTo(receiveEmails.toArray(toArray));
                    helper.setSubject("亲爱的" + personCareView.getPersonName() + ",祝你生日快乐！");
                    helper.setText("<html><head></head><body>" + "<img src=\"cid:image\"/></body></html>", true);
                    helper.addInline("image", temp);
                    // 固定抄送人事？
                    // helper.setCc("ccArray");
                    sendMessage(helper);
                    log.info("【MESSAGE-EMAIL】 邮件发送成功");
                    log.debug("[EDP-SNS] EMAIL message SENT.");
                    temp.delete();
                    result.getAndIncrement();
                } catch (Throwable e) {
                    log.error("[EDP-SNS] EMAIL message sent FAILED.", e);
                }
            });
        }
        return result.get();
    }

    public void sendMessage(MimeMessageHelper messageHelper) {
        if (enabled) {
            javaMailSender.send(messageHelper.getMimeMessage());
        } else {
            log.warn("邮件未发送，原因：邮件服务已关闭。请设置'el.mail.enabled: true' 打开");
        }

    }

    /**
     * 获取MimeMessageHelper
     *
     * @param withAttachments 是否有附件
     * @return MimeMessageHelper
     * @throws MessagingException 邮件异常
     */
    private MimeMessageHelper messageHelper(boolean withAttachments) throws MessagingException {
        MimeMessageHelper msg = new MimeMessageHelper(javaMailSender.createMimeMessage(), withAttachments, "UTF-8");
        msg.setSentDate(Calendar.getInstance().getTime());
        String senderMail = mailProperties.getUsername();
        msg.setFrom(senderMail);
        msg.setReplyTo(senderMail);
        return msg;
    }
}

@Slf4j
class MarkImageUtils {
    private MarkImageUtils() {
    }

    /**
     * @param username 过生日人的名字
     */
    public static File markMain(String username) {
        // 图片源路径
        String[] source = new String[10];
        for (int i = 0; i < source.length; i++) {
            source[i] = File.separator + "src" + File.separator + "birthday_background_images" + File.separator + "template" + i + ".jpg";
        }
        // 图片保存名称
        String imageName = username + "_image";
        // 图片保存格式
        //图片类型jpg,jpeg,png,gif
        String imageType = "jpg";
        // 生日祝福语
        String[] waterMarkWord = new String[12];
        waterMarkWord[0] = "   愿你去往之地皆为热土，愿你所遇之人皆为挚友。生日快乐！";
        waterMarkWord[1] = "   愿以后有人陪你烧烤火锅焖大虾，也有人陪你素琴调弦品香茶。生日快乐！";
        waterMarkWord[2] = "   愿时光能缓，愿故人不散；愿你惦念的人能和你道晚安，愿你独闯的日子里不觉得孤单。生日快乐！";
        waterMarkWord[3] = "   愿你三冬暖，愿你春不寒，愿你天黑有灯，下雨有伞，愿你一路上，有良人相伴。生日快乐！";
        waterMarkWord[4] = "   相见亦无事，别后常忆君。生日快乐！";
        waterMarkWord[5] = "   莫失莫忘，仙寿恒昌；不离不弃，芳龄永继。生日快乐！";
        waterMarkWord[6] = "   祝你吃饭大鱼大肉；唱歌帅哥美女伴奏;日进斗金不够；敢与乌龟比长寿。生日快乐！";
        waterMarkWord[7] = "   彩虹是太阳的花边，装饰着晴朗的每一天，星星是月亮的蕾丝，点缀着浪漫的每一晚，快乐是生日的祝福，希望你拥有美妙的每一刻，亲爱的，生日快乐！";
        waterMarkWord[8] = "   愿你此后每一天，眼里是阳光，笑里是坦荡。生日快乐！";
        waterMarkWord[9] = "   愿你成为自己喜欢的样子：不抱怨，不将就，有自由，有光芒。生日快乐！";
        waterMarkWord[10] = "   愿你有前进一寸的勇气，亦有后退一尺的从容。生日快乐！";
        waterMarkWord[11] = "   友谊其实是一种感觉：一种突然就会想起的感觉，一种经常牵挂的感觉，一种失落可以依靠的感觉，一种快乐可以分享的感觉。朋友，祝福你生日快乐！";
        //水印旋转角度-45，null表示不旋转
        Integer degree = 0;
        // 给图片加单个文字水印
        File result = MarkImageUtils.markImageBySingleText(username, source, imageName, imageType, waterMarkWord, new Color(0, 0, 0, (float) 0.8), 20, degree);
        return result;
    }

    /**
     * 给图片添加单个文字水印、可设置水印文字旋转角度
     *
     * @param source        需要添加水印的图片路径（如：F:/images/6.jpg）
     * @param imageName     图片名称（如：11111）
     * @param imageType     图片类型（如：jpg）
     * @param waterMarkWord 水印文字
     * @param color         水印文字的颜色
     * @param fontSize      水印字体的大小
     * @param degree        水印文字旋转角度，为null表示不旋转
     */
    private static File markImageBySingleText(String username, String[] source, String imageName, String imageType, String[] waterMarkWord, Color color, int fontSize, Integer degree) {

        try {
            // 获取随机祝福语
            int wordIndex = (int) (Math.random() * waterMarkWord.length);
            // 获取随机图片
            int imgIndex = (int) (Math.random() * source.length);

            //读取原图片信息
            File file = ResourceUtils.getFile("classpath:static" + source[imgIndex]);
            if (!file.isFile()) {
                return null;
            }
            Image img = ImageIO.read(file);
            int width = img.getWidth(null);
            int height = img.getHeight(null);
            //初始化
            BufferedImage bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = bi.createGraphics();
            g.drawImage(img, 0, 0, width, height, null);
            // 设置字体路径
            String typefaceRoute = "classpath:static" + File.separator + "ttf" + File.separator + "WenQuanYi Micro Hei Mono.ttf";
            // 加载字体文件
            File fontFile = ResourceUtils.getFile(typefaceRoute);
            Font customFont = Font.createFont(Font.TRUETYPE_FONT, fontFile);
            // 设置字体样式和大小
            Font font = customFont.deriveFont(Font.PLAIN, fontSize);
            //根据图片的背景设置水印颜色
            g.setColor(color);
            // 设置字体
            g.setFont(font);
            // 消除文字锯齿
            g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            if (null != degree) {
                //设置水印旋转
                g.rotate(Math.toRadians(degree), (double) bi.getWidth() / 2, (double) bi.getHeight() / 2);
            }
            String text = "亲爱的";
            text += username + "：";
            drawStringWithFontStyleLineFeed(g, text, 150, height / 2 - 50, width - 280);
            drawStringWithFontStyleLineFeed(g, waterMarkWord[wordIndex], 150, height / 2, width - 280);
            g.dispose();
            // 创建临时文件
            File temp = File.createTempFile(imageName, "." + imageType, null);
            // 将图片写入到临时文件内
            ImageIO.write(bi, imageType, temp);
            return temp;

        } catch (Exception e) {
            log.error("生日祝福模板添加文字失败", e);
        }
        return null;
    }

    /**
     * @param g             画笔
     * @param waterMarkWord 字符串
     * @return 字符串总宽度
     * @Description: 水印文字总宽度（包括中英文）
     */
    private static int getStringLength(Graphics g, String waterMarkWord) {
        char[] strcha = waterMarkWord.toCharArray();
        int strWidth = g.getFontMetrics().charsWidth(strcha, 0, waterMarkWord.length());
        log.debug("水印文字总宽度:" + strWidth);
        return strWidth;
    }

    private static int getCharLen(char c, Graphics g) {
        return g.getFontMetrics(g.getFont()).charWidth(c);
    }

    /**
     * @param g
     * @return
     * @Description: 字符高度
     */
    private static int getStringHeight(Graphics g) {
        int height = g.getFontMetrics().getHeight();
        log.debug("字符高度:" + height);
        return height;
    }

    private static void drawStringWithFontStyleLineFeed(Graphics g, String waterMarkWord, int locX, int locY, int width) {
        //获取水印文字总宽度
        int strContentWidth = getStringLength(g, waterMarkWord);
        //图片宽度
        int imgWidth = width;
        log.debug("每行文字|图片宽度:" + imgWidth);
        //获取字符高度
        int strHeight = getStringHeight(g);
        //字符串总个数（中英文标点都算一个字符）
        int strLength = waterMarkWord.length();
        log.debug("字符串总个数:" + strLength);

        int tempX = locX;
        int tempY = locY;
        if (strContentWidth > imgWidth) {
            //文字叠加,自动换行叠加
            int tempCharLen = 0;//单字符长度
            int tempLineLen = 0;//单行字符总长度临时计算
            int basicCharLen = getCharLen('一', g);
            StringBuilder stringBuffer = new StringBuilder();
            for (int i = 0; i < waterMarkWord.length(); i++) {
                char tempChar = waterMarkWord.charAt(i);
                tempCharLen = getCharLen(tempChar, g);
                if (tempLineLen + 2 * basicCharLen >= imgWidth) {
                    // 绘制前一行
                    g.drawString(stringBuffer.toString(), tempX, tempY);
                    //清空内容,重新追加
                    stringBuffer.delete(0, stringBuffer.length());
                    //文字长度已经满一行,Y的位置加1字符高度
                    tempY = tempY + strHeight + 30;
                    tempLineLen = 0;
                }
                //追加字符
                tempLineLen += tempCharLen;
                stringBuffer.append(tempChar);
            }
            //最后叠加余下的文字
            g.drawString(stringBuffer.toString(), tempX, tempY);
        } else {
            //直接绘制
            g.drawString(waterMarkWord, tempX, tempY);
        }
    }

}
