package com.elitesland.tw.tw5.server.prd.personplan.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.personplan.payload.PmsProjectRolePayload;
import com.elitesland.tw.tw5.api.prd.personplan.query.PmsProjectRoleQuery;
import com.elitesland.tw.tw5.api.prd.personplan.vo.PmsProjectRoleVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.personplan.entity.PmsProjectRoleDO;
import com.elitesland.tw.tw5.server.prd.personplan.entity.QPmsProjectRoleDO;
import com.elitesland.tw.tw5.server.prd.personplan.repo.PmsProjectRoleRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目角色
 *
 * @author xxb
 * @date 2023-07-07
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectRoleDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectRoleRepo repo;
    private final QPmsProjectRoleDO qdo = QPmsProjectRoleDO.pmsProjectRoleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectRoleVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectRoleVO.class,
                qdo.id,
                qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 角色名称
                qdo.roleName,
                // 角色编号
                qdo.roleCode,
                // 平均单价
                qdo.unitPrice
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectRoleVO> getJpaQueryWhere(PmsProjectRoleQuery query) {
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectRoleQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectRoleQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 记录唯一ID集合 精确 */
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 角色名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getRoleName())) {
             list.add(qdo.roleName.like(SqlUtil.toSqlLikeString(query.getRoleName())));
        }
        /** 角色编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getRoleCode())) {
             list.add(qdo.roleCode.like(SqlUtil.toSqlLikeString(query.getRoleCode())));
        }
        /** 平均单价 精确 */
        if (!ObjectUtils.isEmpty(query.getUnitPrice())) {
            list.add(qdo.unitPrice.eq(query.getUnitPrice()));
        }

        /** 角色名称、编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getKeyWord())) {
            list.add(qdo.roleName.like(SqlUtil.toSqlLikeString(query.getKeyWord()))
                    .or(qdo.roleCode.like(SqlUtil.toSqlLikeString(query.getKeyWord()))));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectRoleVO queryByKey(Long id) {
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据角色编号查询
     *
     * @param roleCode 角色编号
     * @return 结果
     */
    public PmsProjectRoleVO queryByRoleCode(String roleCode) {
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.roleCode.eq(roleCode));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据角色名称查询
     *
     * @param roleName 角色名称
     * @return 结果
     */
    public PmsProjectRoleVO queryByRoleName(String roleName) {
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.roleName.eq(roleName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectRoleVO> queryListDynamic(PmsProjectRoleQuery query) {
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectRoleVO> queryPaging(PmsProjectRoleQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsProjectRoleVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectRoleVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectRoleVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectRoleDO save(PmsProjectRoleDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectRoleDO> saveAll(List<PmsProjectRoleDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectRolePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 角色名称
        if (payload.getRoleName() != null) {
            update.set(qdo.roleName, payload.getRoleName());
        }
        // 角色编号
        if (payload.getRoleCode() != null) {
            update.set(qdo.roleCode, payload.getRoleCode());
        }
        // 平均单价
        if (payload.getUnitPrice() != null) {
            update.set(qdo.unitPrice, payload.getUnitPrice());
        }
        // 备注
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 角色名称
            if (nullFields.contains("roleName")) {
                update.setNull(qdo.roleName);
            }
            // 角色编号
            if (nullFields.contains("roleCode")) {
                update.setNull(qdo.roleCode);
            }
            // 平均单价
            if (nullFields.contains("unitPrice")) {
                update.setNull(qdo.unitPrice);
            }
            // 备注
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

