package com.elitesland.tw.tw5.server.prd.personplan.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.db.DbUtil;
import cn.hutool.db.sql.SqlExecutor;
import cn.hutool.json.JSONUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.common.util.DataSourceUtil;
import com.elitesland.tw.tw5.api.prd.personplan.payload.PersonPlanPayload;
import com.elitesland.tw.tw5.api.prd.personplan.payload.PersonPlanVersionPayload;
import com.elitesland.tw.tw5.api.prd.personplan.query.*;
import com.elitesland.tw.tw5.api.prd.personplan.service.*;
import com.elitesland.tw.tw5.api.prd.personplan.vo.*;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsResourcePlanDaysService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsResourcePlanRoleService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsResourcePlanService;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.personplan.constants.PersonPlanTypeEnum;
import com.elitesland.tw.tw5.server.prd.personplan.convert.PersonPlanConvert;
import com.elitesland.tw.tw5.server.prd.personplan.dao.PersonPlanDao;
import com.elitesland.tw.tw5.server.prd.personplan.dao.PersonPlanDtlDao;
import com.elitesland.tw.tw5.server.prd.personplan.dao.ProRelatedPartiesDao;
import com.elitesland.tw.tw5.server.prd.personplan.entity.PersonPlanDO;
import com.elitesland.tw.tw5.server.prd.personplan.entity.PersonPlanDtlDO;
import com.elitesland.tw.tw5.server.prd.personplan.repo.PersonPlanDtlRepo;
import com.elitesland.tw.tw5.server.prd.personplan.repo.PersonPlanRepo;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * @author : WWW
 * @date : 2024-2-21
 * @desc : 人员规划Service
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class PersonPlanServiceImpl implements PersonPlanService {


    private final DataSourceUtil dataSource;
    private final PersonPlanDao personPlanDao;

    private final PersonPlanDtlDao personPlanDtlDao;

    private final PersonPlanRepo personPlanRepo;

    private final PersonPlanDtlService personPlanDtlService;
    private final PersonPlanDtlRepo personPlanDtlRepo;

    private final FileUtil fileUtil;
    private final CacheUtil cacheUtil;
    private final UdcUtil udcUtil;

    private final PersonPlanVersionService personPlanVersionService;
    //    private final CacheUtil cacheUtil;
    private final PmsResourcePlanService pmsResourcePlanService;
    private final PmsResourcePlanRoleService pmsResourcePlanRoleService;
    private final PmsResourcePlanDaysService pmsResourcePlanDaysService;
    private static final BigDecimal hours = new BigDecimal(8);
    private static final BigDecimal jsonDay = BigDecimal.ONE;

    private final PmsProjectRoleService pmsProjectRoleService;
    private final ProRelatedPartiesService proRelatedPartiesService;
    private final ProRelatedPartiesDao proRelatedPartiesDao;
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PersonPlanVO save(PersonPlanPayload personPlanPayload) {

        checkData(personPlanPayload);
        PersonPlanDO personPlanDO = new PersonPlanDO();
        personPlanDO = PersonPlanConvert.INSTANCE.p2d(personPlanPayload);
        personPlanDO.setModifyTime(LocalDateTime.now());
        PersonPlanDO res = personPlanRepo.save(personPlanDO);
        PersonPlanVO personPlanVO = PersonPlanConvert.INSTANCE.d2v(res);

        // 先删除明细
        personPlanDtlService.del(personPlanPayload.getDelIds());

        // 人员规划明细
        if (!CollectionUtils.isEmpty(personPlanPayload.getPersonPlanDtlPayloadList())) {
            personPlanPayload.getPersonPlanDtlPayloadList().stream().forEach(x -> {
                x.setPlanId(res.getId());
            });
            List<PersonPlanDtlVO> personPlanDtlVOS = personPlanDtlService.saveAll(personPlanPayload.getPersonPlanDtlPayloadList(), res.getId());
            personPlanVO.setPersonPlanDtlVOList(personPlanDtlVOS);
        }
        return personPlanVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PersonPlanVO updateAll(PersonPlanPayload personPlanPayload) {

        Assert.notNull(personPlanPayload.getId(), "id is null");
        PersonPlanVO res = save(personPlanPayload);
        return res;

    }


    @Override
    public PersonPlanVO get(Long id) {

        if (null == id) {
            return null;
        }
        PersonPlanVO res = personPlanDao.get(id);
        if (null != res) {
            PersonPlanDtlQuery personPlanDtlQuery = new PersonPlanDtlQuery();
            personPlanDtlQuery.setPlanId(res.getId());
            List<PersonPlanDtlVO> dtlList = personPlanDtlDao.getList(personPlanDtlQuery);
            if (CollUtil.isNotEmpty(dtlList)) {
                this.translateDtl(dtlList);
            }
            res.setPersonPlanDtlVOList(dtlList);
            if (StringUtils.hasText(res.getFileCodes())) {
                res.setFileDatas(fileUtil.getFileDatas(res.getFileCodes()));
            }
        }
        return res;
    }


    @Override
    public PagingVO<PersonPlanVO> page(PersonPlanQuery personPlanQuery) {

        PagingVO<PersonPlanVO> res = personPlanDao.page(personPlanQuery);

        if (!CollectionUtils.isEmpty(res.getRecords())) {
            res.stream().forEach(p -> translate(p));
        }
        return res;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long del(List<Long> ids) {

        if (CollectionUtil.isEmpty(ids)) {
            return 0L;
        }
        Long res = personPlanDao.del(ids);
        if (res != null && res > 0) {
            personPlanDtlService.deleteByPlanIds(ids);
        }
        return res;

    }

    @Override
    public List<PersonPlanVO> getList(PersonPlanQuery personPlanQuery) {

        List<PersonPlanVO> res = personPlanDao.getList(personPlanQuery);
        if (!CollectionUtils.isEmpty(res)) {
            res.stream().forEach(p -> translate(p));
        }
        return res;

    }


    /**
     * 数据校验
     *
     * @param personPlanPayload
     */
    private void checkData(PersonPlanPayload personPlanPayload) {

        Assert.notEmpty(personPlanPayload.getUom(), "周期单位不能为空");
        Assert.notNull(personPlanPayload.getStartDate(), "开始时间不能为空");
        Assert.notNull(personPlanPayload.getEndDate(), "结束时间不能为空");
        Assert.notNull(personPlanPayload.getObjId(), "关联资源id不能为空");

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long update(PersonPlanPayload personPlanPayload) {
        Assert.notNull(personPlanPayload.getId(), "id不能为空");
        Long res = personPlanDao.update(personPlanPayload);
        return res;
    }

    @Override
    public PersonPlanVO getByObjIdAndPlanType(Long objId, List<String> planTypeList) {
        PersonPlanVO res = null;
        // 商机
        if (planTypeList.contains(PersonPlanTypeEnum.PRE_SALES.getCode()) || planTypeList.contains(PersonPlanTypeEnum.DELIVERY.getCode())) {
            res = personPlanDao.queryOppo(objId, planTypeList);
        }
        // 合同预算
        if (planTypeList.contains(PersonPlanTypeEnum.BUDGET.getCode())) {
            res = personPlanDao.queryBudget(objId, planTypeList);
        }
        //项目
        if (planTypeList.contains(PersonPlanTypeEnum.PROJECT.getCode())) {
            res = personPlanDao.queryProject(objId, planTypeList);
        }
        // 内部项目
        if (planTypeList.contains(PersonPlanTypeEnum.INNER_PROJECT.getCode())) {
            res = personPlanDao.queryInnerProject(objId, planTypeList);
        }

        if (null != res) {
            PersonPlanDtlQuery personPlanDtlQuery = new PersonPlanDtlQuery();
            personPlanDtlQuery.setPlanId(res.getId());
            List<PersonPlanDtlVO> dtlList = personPlanDtlService.getList(personPlanDtlQuery);
            if (CollUtil.isNotEmpty(dtlList)) {
                this.translateDtl(dtlList);
            }
            res.setPersonPlanDtlVOList(dtlList);
            if (StringUtils.hasText(res.getFileCodes())) {
                res.setFileDatas(fileUtil.getFileDatas(res.getFileCodes()));
            }
        }
        return res;
    }

    @Override
    @Transactional
    public void saveNewVersion(PersonPlanPayload personPlanPayload) {
        // 先保存一下当前数据
        PersonPlanVO personPlanVO = save(personPlanPayload);
        List<PersonPlanDtlVO> personPlanDtlVOList = personPlanVO.getPersonPlanDtlVOList();
        personPlanVO = (PersonPlanVO) udcUtil.translate(personPlanVO);
        if (!CollectionUtils.isEmpty(personPlanDtlVOList)) {
            personPlanVO.setPersonPlanDtlVOList(udcUtil.translateList(personPlanDtlVOList));
        }
        //保存最新的版本
        String content = JSONUtil.toJsonStr(personPlanVO);
        String versionName = personPlanVO.getObjName();
        if (PersonPlanTypeEnum.PRE_SALES.getCode().equals(personPlanPayload.getPlanType()) || PersonPlanTypeEnum.DELIVERY.getCode().equals(personPlanPayload.getPlanType())) {
            String format = LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd"));
            versionName = versionName + "-" + personPlanPayload.getOppoSalePhaseName() + "-" + format;
            PersonPlanVersionQuery personPlanVersionQuery = new PersonPlanVersionQuery();
            personPlanVersionQuery.setPlanId(personPlanVO.getId());
            personPlanVersionQuery.setVersionName(versionName);
            List<PersonPlanVersionVO> personPlanVersionVOS = personPlanVersionService.queryListDynamic(personPlanVersionQuery);
            if (CollectionUtils.isEmpty(personPlanVersionVOS)) {
                versionName = versionName + "01";
            } else {
                PersonPlanVersionVO personPlanVersionVO = personPlanVersionVOS.get(0);
                personPlanVersionVO.getVersionName();
                versionName = personPlanVersionVO.getVersionName().substring(0, personPlanVersionVO.getVersionName().length() - 2) + String.format("%02d", Integer.parseInt(personPlanVersionVO.getVersionName().substring(personPlanVersionVO.getVersionName().length() - 2)) + 1);
            }
        }
        PersonPlanVersionPayload personPlanVersionPayload = PersonPlanVersionPayload
                .builder()
                .versionName(versionName)
                .planId(personPlanVO.getId())
                .planType(personPlanVO.getPlanType())
                .objId(personPlanVO.getObjId())
                .changeContent(content).build();
        personPlanVersionService.save(personPlanVersionPayload);
    }

    @Override
    public void transferData() {
//        PmsResourcePlanQuery pmsResourcePlanQuery = new PmsResourcePlanQuery();
//        // 查询老的资源规划主表
//        List<PmsResourcePlanVO> pmsResourcePlanVOS = pmsResourcePlanService.queryListDynamic(pmsResourcePlanQuery);
//        if (!CollectionUtils.isEmpty(pmsResourcePlanVOS)) {
//            pmsResourcePlanVOS = pmsResourcePlanVOS.stream().filter(e -> e.getStartDate() != null && e.getEndDate() != null).collect(Collectors.toList());
//            List<Long> planIdList = pmsResourcePlanVOS.stream().map(e -> e.getId()).collect(Collectors.toList());
//            // 查询老的资源规划明细表
//            PmsResourcePlanDaysQuery pmsResourcePlanDaysQuery = new PmsResourcePlanDaysQuery();
//            pmsResourcePlanDaysQuery.setLargeDays(BigDecimal.ZERO);
//            pmsResourcePlanDaysQuery.setNotNullPlanIdFlag(true);
//            List<PmsResourcePlanDaysVO> pmsResourcePlanDaysVOS = pmsResourcePlanDaysService.queryListDynamic(pmsResourcePlanDaysQuery);
//            Map<Long, List<PmsResourcePlanDaysVO>> pmsResourcePlanDaysMap = pmsResourcePlanDaysVOS.stream().collect(Collectors.groupingBy(PmsResourcePlanDaysVO::getPlanId));
//            // 查询老的资源规划对应人员表
//            PmsResourcePlanRoleQuery pmsResourcePlanRoleQuery = new PmsResourcePlanRoleQuery();
//            pmsResourcePlanRoleQuery.setPlanIdList(planIdList);
//            List<PmsResourcePlanRoleVO> pmsResourcePlanRoleVOS = pmsResourcePlanRoleService.queryListByPlanIdList(pmsResourcePlanRoleQuery);
//            if (CollectionUtils.isEmpty(pmsResourcePlanRoleVOS)) {
//                log.error("老的资源规划对应人员表数据为空");
//                return;
//            }
//            final Map<Long, List<PmsResourcePlanRoleVO>> pmsResourcePlanRoleMap = pmsResourcePlanRoleVOS.stream().collect(Collectors.groupingBy(PmsResourcePlanRoleVO::getPlanId));
//            // 遍历老的资源规划主表
//            List<PersonPlanDO> personPlanDOList = new ArrayList<>();
//            List<PersonPlanDtlDO> personPlanDtlDOList = new ArrayList<>();
//            PrdSystemSelectionVO roleView = cacheUtil.getSystemSelection(FunctionSelectionEnum.PMS_RESOURCE_PLAN_ROLE.getCode());
//            Map<String, List<PrdSystemSelectionVO>> childrenMap = new HashMap<>();
//            if (roleView != null) {
//                List<PrdSystemSelectionVO> children = roleView.getChildren();
//                if (!CollectionUtils.isEmpty(children)) {
//                    childrenMap = children.stream().collect(Collectors.groupingBy(PrdSystemSelectionVO::getSelectionName));
//                }
//            }
//            final Map<String, List<PrdSystemSelectionVO>> map = childrenMap;
//            pmsResourcePlanVOS.stream().forEach(p -> {
//                // 先处理主表数据
//                PersonPlanDO personPlanDO = new PersonPlanDO();
//                personPlanDO.setCreator("data_initial");
//                personPlanDO.setId(p.getId());
//                personPlanDO.setPlanType(PersonPlanTypeEnum.PROJECT.getCode());
//                personPlanDO.setObjId(p.getObjId());
//                personPlanDO.setObjName(p.getObjName());
//                personPlanDO.setStartDate(p.getStartDate());
//                personPlanDO.setEndDate(p.getEndDate());
//                personPlanDO.setUom(PersonPlanUomEnum.month.getCode());
//                LocalDate startDate = p.getStartDate().with(TemporalAdjusters.firstDayOfMonth());
//                LocalDate endDate = p.getEndDate().with(TemporalAdjusters.firstDayOfMonth());
//                long monthsBetween = getMonthsBetween(startDate, endDate);
//                personPlanDO.setDuration(new BigDecimal(monthsBetween).add(BigDecimal.ONE).toString());
//                personPlanDO.setVersion("0");
//                personPlanDO.setCreateTime(p.getCreateTime());
//                personPlanDO.setCreateUserId(p.getCreateUserId());
//                personPlanDO.setDeleteFlag(p.getDeleteFlag());
//                personPlanDO.setModifyTime(p.getModifyTime());
//                personPlanDOList.add(personPlanDO);
//                List<PmsResourcePlanDaysVO> planDaysList = pmsResourcePlanDaysMap.get(p.getId());
//                if (pmsResourcePlanRoleMap.containsKey(p.getId())) {
//                    List<PmsResourcePlanRoleVO> planRoleList = pmsResourcePlanRoleMap.get(p.getId());
//                    planRoleList.stream().forEach(role -> {
//                        PersonPlanDtlDO personPlanDtlDO = new PersonPlanDtlDO();
//                        if (!CollectionUtils.isEmpty(map)) {
//                            List<PrdSystemSelectionVO> prdSystemSelectionVOS = map.get(role.getRole());
//                            if (!CollectionUtils.isEmpty(prdSystemSelectionVOS)) {
//                                personPlanDtlDO.setRoleCode(prdSystemSelectionVOS.get(0).getSelectionValue());
//                            }
//                        }
//                        personPlanDtlDO.setRoleName(role.getRole());
//                        personPlanDtlDO.setResId(role.getResId());
//                        personPlanDtlDO.setDistributeRate(role.getDistributeRate());
//                        personPlanDtlDO.setPlanId(role.getPlanId());
//                        personPlanDtlDO.setId(role.getId());
//                        personPlanDtlDO.setCreator("data_initial");
//                        personPlanDtlDO.setDays(role.getTotalDays());
//                        personPlanDtlDO.setTotalEqva(role.getTotalEqva());
//                        personPlanDtlDO.setCapasetLevelId(role.getCapasetLevelId());
//                        personPlanDtlDO.setCreateTime(role.getCreateTime());
//                        personPlanDtlDO.setCreateUserId(role.getCreateUserId());
//                        personPlanDtlDO.setDeleteFlag(role.getDeleteFlag());
//                        personPlanDtlDO.setModifyTime(role.getModifyTime());
//                        List<DayJsonVO> dayJsonVOList = new ArrayList<>();
//                        // 某个人的规划天数明细
//                        if (!CollectionUtils.isEmpty(planDaysList)) {
//                            List<PmsResourcePlanDaysVO> planDaysRoleList = planDaysList.stream().filter(x -> x.getRoleDetailId().equals(role.getId())).collect(Collectors.toList());
//                            if (!CollectionUtils.isEmpty(planDaysRoleList)) {
//                                planDaysRoleList.forEach(dayRole -> {
//                                    BigDecimal day = dayRole.getDays();
//                                    BigDecimal upper = day.setScale(0, RoundingMode.UP);
//                                    int j = 0;
//                                    for (BigDecimal i = BigDecimal.ZERO; upper.compareTo(i) > 0;i = i.add(BigDecimal.ONE))  {
//                                        DayJsonVO dayJsonVO = new DayJsonVO();
//                                        if(day.compareTo(jsonDay)>0){
//                                            dayJsonVO.setHour(hours);
//                                            dayJsonVO.setDay(jsonDay);
//                                            day = day.subtract(BigDecimal.ONE);
//                                        }else{
//                                            dayJsonVO.setHour(day.multiply(hours));
//                                            dayJsonVO.setDay(day);
//                                        }
//                                        dayJsonVO.setDate(dayRole.getStartDate().plusDays(j).format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
//                                        dayJsonVOList.add(dayJsonVO);
//                                        j++;
//                                    }
//                                });
//                                personPlanDtlDO.setDaysJson(JSONUtil.toJsonStr(dayJsonVOList));
//                            }
//                        }
//                        personPlanDtlDOList.add(personPlanDtlDO);
//                    });
//                }
//
//            });
////            // 保存主表
////            if (!CollectionUtils.isEmpty(personPlanDOList)) {
////                personPlanRepo.saveAll(personPlanDOList);
////            }
////            // 保存明细表
////            if (!CollectionUtils.isEmpty(personPlanDtlDOList)) {
////                personPlanDtlRepo.saveAll(personPlanDtlDOList);
////            }
//            saveDate(personPlanDOList, personPlanDtlDOList);
//        }
    }


    public void saveDate(List<PersonPlanDO> personPlanDOList, List<PersonPlanDtlDO> personPlanDtlDOList) {
        long startTime = System.currentTimeMillis();
        if (CollectionUtils.isEmpty(personPlanDOList)) {
            return;
        }
        DataSource ds = dataSource.getDs();
        try {
            Connection conn = ds.getConnection();
            try {
                // 批量插入
                conn.setAutoCommit(false);
                // 执行非查询语句，返回影响的行数
                String sql = "INSERT INTO person_plan(id, start_date, end_date, duration, uom, version," +
                        " tenant_id,  remark, create_user_id, creator, create_time, modify_user_id, " +
                        "updater, modify_time, delete_flag,obj_id, obj_name, plan_type, file_codes) \n" +
                        "VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
                int num = 0;
                // 创建参数列表
                List<Object[]> paramsList = new ArrayList<>();
                for (PersonPlanDO personPlanDO : personPlanDOList) {
                    num++;
                    Object[] params = {
                            personPlanDO.getId(),
                            personPlanDO.getStartDate(),
                            personPlanDO.getEndDate(),
                            personPlanDO.getDuration(),
                            personPlanDO.getUom(),
                            personPlanDO.getVersion(),
                            "-1",
                            personPlanDO.getRemark(),
                            personPlanDO.getCreateUserId(),
                            personPlanDO.getCreator(),
                            personPlanDO.getCreateTime(),
                            personPlanDO.getModifyUserId(),
                            personPlanDO.getUpdater(),
                            personPlanDO.getModifyTime(),
                            0,
                            personPlanDO.getObjId(),
                            personPlanDO.getObjName(),
                            personPlanDO.getPlanType(),
                            personPlanDO.getFileCodes()
                    };
                    paramsList.add(params);
                    if (num > 1000) {
                        SqlExecutor.executeBatch(conn, sql, paramsList);
                        conn.commit();
                        paramsList = new ArrayList<>();
                        num = 0;
                    }
                }
                SqlExecutor.executeBatch(conn, sql, paramsList);
                conn.commit();

                log.info("资源规划主表保存完毕，开始保存资源规划明细表");
                if (!CollectionUtils.isEmpty(personPlanDtlDOList)) {
                    // 执行非查询语句，返回影响的行数
                    String sql2 = "INSERT INTO person_plan_dtl(id, plan_id, price, days, amt, days_json, tenant_id," +
                            "  remark, create_user_id, creator, create_time, modify_user_id, updater, " +
                            "modify_time, delete_flag, role_name, capaset_level_id, res_id, distribute_rate," +
                            " total_eqva, hidden_flag, role_code) \n" +
                            "VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
                    int num2 = 0;
                    // 创建参数列表
                    List<Object[]> paramsList2 = new ArrayList<>();
                    for (PersonPlanDtlDO personPlanDtlDO : personPlanDtlDOList) {
                        num++;
                        Object[] params = {
                                personPlanDtlDO.getId(),
                                personPlanDtlDO.getPlanId(),
                                personPlanDtlDO.getPrice(),
                                personPlanDtlDO.getDays(),
                                personPlanDtlDO.getAmt(),
                                personPlanDtlDO.getDaysJson(),
                                "-1",
                                personPlanDtlDO.getRemark(),
                                personPlanDtlDO.getCreateUserId(),
                                personPlanDtlDO.getCreator(),
                                personPlanDtlDO.getCreateTime(),
                                personPlanDtlDO.getModifyUserId(),
                                personPlanDtlDO.getUpdater(),
                                personPlanDtlDO.getModifyTime(),
                                0,
                                personPlanDtlDO.getRoleName(),
                                personPlanDtlDO.getCapasetLevelId(),
                                personPlanDtlDO.getResId(),
                                personPlanDtlDO.getDistributeRate(),
                                personPlanDtlDO.getTotalEqva(),
                                personPlanDtlDO.getHiddenFlag(),
                                personPlanDtlDO.getRoleCode()
                        };
                        paramsList2.add(params);
                        if (num2 > 1000) {
                            SqlExecutor.executeBatch(conn, sql2, paramsList2);
                            conn.commit();
                            paramsList2 = new ArrayList<>();
                            num2 = 0;
                        }
                    }
                    SqlExecutor.executeBatch(conn, sql2, paramsList2);
                    conn.commit();
                    log.info("资源规划明细表保存完毕");
                }
            } catch (SQLException e) {
                log.error("SQL error!", e);
            } finally {
                DbUtil.close(conn);
            }
        } catch (Exception e) {
            log.error("sql error", e);
            throw TwException.error("", "数据导入错误，请联系管理员");
        } finally {
            DbUtil.close(dataSource);
            long endTime = System.currentTimeMillis();
            log.info("数据导入完毕，耗时：" + (endTime - startTime) + "ms");
        }
    }

    private void translateDtl(List<PersonPlanDtlVO> dtlList) {
        dtlList.forEach(dtl -> {
            BigDecimal days = dtl.getDays();
            BigDecimal distributeRate = dtl.getDistributeRate();
            if (days != null && distributeRate != null) {
                dtl.setTotalEqva(distributeRate.multiply(days));
            } else {
                dtl.setTotalEqva(BigDecimal.ZERO);
            }
        });

//        Map<Long, String> proRoleMap = new HashMap<>();
//        Map<Long, String> proRelateMap = new HashMap<>();
//
//        List<Long> proRoleIds = dtlList.stream().map(x -> x.getRoleId()).collect(Collectors.toList());
//        PmsProjectRoleQuery pmsProjectRoleQuery = new PmsProjectRoleQuery();
//        pmsProjectRoleQuery.setIds(proRoleIds);
//        List<PmsProjectRoleVO> pmsProjectRoleVOS = pmsProjectRoleService.queryListDynamic(pmsProjectRoleQuery);
//        pmsProjectRoleVOS.stream().forEach(p -> {
//            proRoleMap.put(p.getId(), p.getRoleName());
//        });
//
//        List<Long> proRelateIds = dtlList.stream().map(x -> x.getRelatePartiesId()).collect(Collectors.toList());
//
//        ProRelatedPartiesQuery proRelatedPartiesQuery = new ProRelatedPartiesQuery();
//        proRelatedPartiesQuery.setIdList(proRelateIds);
//        List<ProRelatedPartiesVO> relatedPartiesVOList = proRelatedPartiesService.getList(proRelatedPartiesQuery);
//        relatedPartiesVOList.stream().forEach(x -> {
//            proRelateMap.put(x.getId(), x.getName());
//        });
//
//        dtlList.forEach(x -> {
//            // 项目角色
//            x.setRoleName(proRoleMap.get(x.getRoleId()));
//            // 相关方名称
//            x.setRelatePartiesName(proRelateMap.get(x.getRelatePartiesId()));
//        });
    }

    private void translate(PersonPlanVO res) {
        if (StringUtils.hasText(res.getFileCodes())) {
            fileUtil.getFileDatas(res.getFileCodes());
        }
    }

    public static long getMonthsBetween(LocalDate startDate, LocalDate endDate) {
        return Math.abs(ChronoUnit.MONTHS.between(startDate, endDate));
    }

    @Override
    public PersonPlanVO getByProId(Long proId) {

        PersonPlanVO res = personPlanDao.getByProId(proId);
        if (null != res) {
            PersonPlanDtlQuery personPlanDtlQuery = new PersonPlanDtlQuery();
            personPlanDtlQuery.setPlanId(res.getId());
            List<PersonPlanDtlVO> dtlList = personPlanDtlDao.getList(personPlanDtlQuery);
            if (CollUtil.isNotEmpty(dtlList)) {
                this.translate(dtlList,res);
            }
            res.setPersonPlanDtlVOList(dtlList);
        }

        return res;
    }
    private void translate(List<PersonPlanDtlVO> dtlList,PersonPlanVO res) {

        Map<Long, String> proRoleMap = new HashMap<>();
        Map<Long, String> proRelateMap = new HashMap<>();

        List<Long> proRoleIds = dtlList.stream().map(x -> x.getRoleId()).collect(Collectors.toList());
        PmsProjectRoleQuery pmsProjectRoleQuery = new PmsProjectRoleQuery();
        pmsProjectRoleQuery.setIds(proRoleIds);
        List<PmsProjectRoleVO> pmsProjectRoleVOS = pmsProjectRoleService.queryListDynamic(pmsProjectRoleQuery);
        pmsProjectRoleVOS.stream().forEach(p -> {
            proRoleMap.put(p.getId(), p.getRoleName());
        });

        List<Long> proRelateIds = dtlList.stream().map(x -> x.getRelatePartiesId()).collect(Collectors.toList());

        List<Long> userIdList = new ArrayList<>();
        Map<Long,Long> relatedPartiesAndUserMap = new HashMap<>();
        ProRelatedPartiesQuery proRelatedPartiesQuery = new ProRelatedPartiesQuery();
        proRelatedPartiesQuery.setIdList(proRelateIds);
        List<ProRelatedPartiesVO> relatedPartiesVOList = proRelatedPartiesService.getList(proRelatedPartiesQuery);


        relatedPartiesVOList.stream().forEach(x -> {
            proRelateMap.put(x.getId(), x.getName());
            relatedPartiesAndUserMap.put(x.getId(), x.getUserId());
            userIdList.add(x.getUserId());
        });

        // 查询资源所有的资源规划
        List<PersonPlanDtlAndProjectVO> list = new ArrayList<>();
        if (!CollectionUtil.isEmpty(userIdList)) {
            ProRelatedPartiesQuery tempQuery = new ProRelatedPartiesQuery();
            tempQuery.setUserIdList(userIdList);
            List<ProRelatedPartiesVO> tempPartiesList = proRelatedPartiesDao.getList(tempQuery);
            Map<Long, Long> map = tempPartiesList.stream().collect(Collectors.toMap(ProRelatedPartiesVO::getId, ProRelatedPartiesVO::getUserId));
            List<Long> idList = tempPartiesList.stream().map(p -> p.getId()).collect(Collectors.toList());
            PersonPlanDtlQuery personPlanDtlQuery = new PersonPlanDtlQuery();
            personPlanDtlQuery.setRelatePartiesIdList(idList);
            List<PersonPlanDtlAndProjectVO> personPlanDtlAndProjectVOS = personPlanDtlService.queryByRelatePartiesId(personPlanDtlQuery);
            LocalDate startDate = res.getStartDate();
            LocalDate endDate = res.getEndDate();
            if (!CollectionUtil.isEmpty(personPlanDtlAndProjectVOS)) {
                personPlanDtlAndProjectVOS.stream().forEach(p -> {
                    if(p.getProId().equals(res.getProId())){
                        return;
                    }
                    boolean overlapping = isOverlapping(p.getStartDate(), p.getEndDate(), startDate, endDate);
                    if (overlapping) {
                        p.setUserId(map.get(p.getRelatePartiesId()));
                        list.add(p);
                    }
                });
            }
        }
        Map<Long, List<PersonPlanDtlAndProjectVO>> listMap = list.stream().collect(Collectors.groupingBy(o -> o.getUserId()));
        dtlList.forEach(x -> {
            // 项目角色
            x.setRoleName(proRoleMap.get(x.getRoleId()));
            // 相关方名称
            x.setRelatePartiesName(proRelateMap.get(x.getRelatePartiesId()));
            // 资源的相关项目的资源规划天数
            if(relatedPartiesAndUserMap.containsKey(x.getRelatePartiesId())){
                Long userId = relatedPartiesAndUserMap.get(x.getRelatePartiesId());
                if(listMap.containsKey(userId)){
                    List<PersonPlanDtlAndProjectVO> personPlanDtlAndProjectVOS = listMap.get(userId);
                    x.setPersonPlanDtlAndProjectVOList(personPlanDtlAndProjectVOS);
                }
            }
        });
    }
    private boolean isOverlapping(LocalDate date1, LocalDate date2, LocalDate startDate, LocalDate endDate) {
        // 确保 date1 <= date2 并且 startDate <= endDate
        if (date1.isAfter(date2) || startDate.isAfter(endDate)) {
            return false;
        }
        // 检查是否有重叠
        return !(date1.isAfter(endDate) || date2.isBefore(startDate));
    }
}
