package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.BudgetStatusEnum;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetRepo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 预算基本信息
 *
 * @author xiexb
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetRepo repo;
    private final QPmsBudgetDO qdo = QPmsBudgetDO.pmsBudgetDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 预算名称
                qdo.budgetName,
                // 预算编号
                qdo.budgetCode,
                // 预算模板id
                qdo.templateId,
                // 开始时间
                qdo.startTime,
                // 结束时间
                qdo.endTime,
                // 预算状态：新建、审批中、激活（已审批）、关闭
                qdo.budgetStatus,
                // 费用总预算
                qdo.totalMoney,
                // 预算附件
                qdo.budgetFiles,
                // 项目id
                qdo.projectId,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime,
                // 拨付状态：审批中、驳回、已审批
                qdo.procInstStatus,
                //币种
                qdo.currCode,
                // 预算控制：刚性控制、柔性控制、不控制
                qdo.budgetControl,
                // 已拨付金额
                qdo.paidMoney,
                //已使用预算金额
                qdo.usedMoney,
                //使用占比
                qdo.proportion,
                //拨付占比
                qdo.appropriationProportion,
                //可使用金额
                qdo.residueMoney
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetVO> getJpaQueryWhere(PmsBudgetQuery query) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
//            // 常用基础查询条件拼装
//            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetQuery query) {
        List<Predicate> list = new ArrayList<>();
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetVO queryByKey(Long id) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public PmsBudgetVO queryByWBS(Long wbsId) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.eq(wbsId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public List<PmsBudgetVO> queryByWbsList(List<Long> wbsIds) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.in(wbsIds));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public List<PmsBudgetVO> queryListByProjectId(Long projectId) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetVO> queryListDynamic(PmsBudgetQuery query) {
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetVO> queryPaging(PmsBudgetQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetDO save(PmsBudgetDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetDO> saveAll(List<PmsBudgetDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 变更工作流相关数据
     *
     * @param payload
     * @return
     */

    public long updateWorkFlow(PmsBudgetPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        // 流程实例id
        if (!ObjectUtils.isEmpty(payload.getProcInstId())) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(payload.getProcInstStatus())) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 状态
        if (!ObjectUtils.isEmpty(payload.getBudgetStatus())) {
            update.set(qdo.budgetStatus, payload.getBudgetStatus());
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(payload.getSubmitTime())) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(payload.getApprovedTime())) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (!ObjectUtils.isEmpty(payload.getDeleteFlag())) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * @param projectId           项目id
     * @param appropriationAmount 审批通过的拨付金额
     * @return
     */
    public long updateAppropriationAmount(Long projectId, BigDecimal appropriationAmount) {
//        PmsBudgetVO vo = queryByProjectId(projectId);
//
//        //已拨付预算金额
//        BigDecimal paid = vo.getPaidMoney().add(appropriationAmount);
//
//        //累计拨付占比 = 已拨付金额 / 预算总金额
//        BigDecimal appropriationProportion = BigDecimal.ZERO;
//        if (vo.getTotalMoney().doubleValue() > 0) {
//            appropriationProportion = paid.divide(vo.getTotalMoney(), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
//        }
//
//        //可使用金额 = 已拨付预算 - 已使用预算
//        BigDecimal residue = paid.subtract(vo.getUsedMoney());
//
//        JPAUpdateClause update = jpaQueryFactory.update(qdo)
//                .where(qdo.id.eq(vo.getId()));
//        update.set(qdo.paidMoney, paid);
//        update.set(qdo.appropriationProportion, appropriationProportion);
//        update.set(qdo.residueMoney, residue);
//        update.set(qdo.budgetStatus, BudgetStatusEnum.APPROVED.getCode());
//        //拼装更新
//        SqlUtil.updateCommonJpaQuery(update, qdo._super);
//        // 执行修改
//        return update.execute();
        return 1;
    }


    /**
     * 更新已使用金额
     *
     * @param projectId
     * @param usedMoney 已使用金额
     * @return
     */
    public long updateUsedMoney(Long projectId, BigDecimal usedMoney) {
//        PmsBudgetVO vo = queryByProjectId(projectId);
//
//        //已拨付预算金额=已完成拨付的金额
//        BigDecimal paid = BigDecimal.ZERO;
//        if (!ObjectUtils.isEmpty(vo.getPaidMoney())) {
//            paid = vo.getPaidMoney();
//        }
//
//        //已使用预算金额=已经申请报销并且已完成审批的预算金额+已结算当量预算金额
//        BigDecimal used = BigDecimal.ZERO;
//        if (ObjectUtils.isEmpty(vo.getUsedMoney())) {
//            used = usedMoney;
//        } else {
//            used = vo.getUsedMoney().add(usedMoney);
//        }
//
//        //使用占比=已使用预算金额/预算总金额
//        BigDecimal proportion = BigDecimal.ZERO;
//        if (vo.getTotalMoney().doubleValue() > 0) {
//            proportion = used.divide(vo.getTotalMoney(), 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
//        }
//
//        //可使用金额=已拨付预算-已使用预算
//        BigDecimal residue = paid.subtract(used);
//
//        JPAUpdateClause update = jpaQueryFactory.update(qdo)
//                .where(qdo.id.eq(vo.getId()));
//        update.set(qdo.usedMoney, used);
//        update.set(qdo.proportion, proportion);
//        update.set(qdo.residueMoney, residue);
//
//        //拼装更新
//        SqlUtil.updateCommonJpaQuery(update, qdo._super);
//        // 执行修改
//        return update.execute();
        return 1;
    }

    /**
     * 预算变更 审批通过
     * @param vo 旧的预算记录
     * @param totalMoney 预算总额
     * @param payload  新的预算记录
     * @return
     */
    public long updateMoney(PmsBudgetVO vo, BigDecimal totalMoney, PmsBudgetPayload payload) {
        //已拨付预算金额=已完成拨付的金额
        BigDecimal paid = BigDecimal.ZERO;
        if (!ObjectUtils.isEmpty(vo.getPaidMoney())) {
            paid = vo.getPaidMoney();
        }

        //已使用预算金额
        BigDecimal used = vo.getUsedMoney();

        //使用占比=已使用预算金额/预算总金额
        BigDecimal proportion = BigDecimal.ZERO;
        //累计拨付占比==已拨付金额总和/预算总金额
        BigDecimal appropriationProportion = BigDecimal.ZERO;
        if (totalMoney.doubleValue() > 0) {
            proportion = used.divide(totalMoney, 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
            appropriationProportion = paid.divide(totalMoney, 4, RoundingMode.HALF_UP).multiply(new BigDecimal(100));
        }
        //可使用金额=已拨付预算-已使用预算
        BigDecimal residue = paid.subtract(used);

        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(vo.getId()));

        //金额相关
        update.set(qdo.proportion, proportion);
        update.set(qdo.appropriationProportion, appropriationProportion);
        update.set(qdo.residueMoney, residue);
        update.set(qdo.totalMoney, totalMoney);

        //审批状态、时间
        update.set(qdo.budgetStatus, BudgetStatusEnum.APPROVED.getCode());
        update.set(qdo.approvedTime, LocalDateTime.now());

        //预算名称、起止日期、附件、基本信息-备注
        update.set(qdo.budgetName, payload.getBudgetName());
        update.set(qdo.startTime, payload.getStartTime());
        update.set(qdo.endTime, payload.getEndTime());
        update.set(qdo.budgetFiles, payload.getBudgetFiles());
        update.set(qdo.remark, payload.getRemark());

        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }


    /**
     * 更新金额
     * @param id
     * @param totalMoney
     * @param usedMoney
     * @param proportion
     * @return
     */
    public long updateMoney(Long id,BigDecimal totalMoney,BigDecimal usedMoney,BigDecimal proportion) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(id));
        update.set(qdo.totalMoney, totalMoney);
        update.set(qdo.usedMoney, usedMoney);
        update.set(qdo.proportion, proportion);
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }



    /**
     * 变更工作流相关数据
     * @return
     */

    public long updateWorkFlow(List<Long> keys, LocalDateTime submitTime,
                               String procInstId, String budgetStatus,
                               LocalDateTime approvedTime, ProcInstStatus procInstStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.in(keys));

        // 流程实例id
        if (!ObjectUtils.isEmpty(procInstId)) {
            update.set(qdo.procInstId, procInstId);
        }
        // 审批状态
        if (!ObjectUtils.isEmpty(procInstStatus)) {
            update.set(qdo.procInstStatus, procInstStatus);
        }
        // 状态
        if (!ObjectUtils.isEmpty(budgetStatus)) {
            update.set(qdo.budgetStatus, budgetStatus);
        }
        // 提审时间
        if (!ObjectUtils.isEmpty(submitTime)) {
            update.set(qdo.submitTime, submitTime);
        }
        // 审批结束时间
        if (!ObjectUtils.isEmpty(approvedTime)) {
            update.set(qdo.approvedTime, approvedTime);
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

}

