package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetDetailSubjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetDetailSubjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetDetailSubjectVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetDetailSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetDetailSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetDetailSubjectRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算详细明细-科目
 *
 * @author xxb
 * @date 2023-04-24
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetDetailSubjectDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetDetailSubjectRepo repo;
    private final QPmsBudgetDetailSubjectDO qdo = QPmsBudgetDetailSubjectDO.pmsBudgetDetailSubjectDO;

    private final QPmsBudgetSubjectDO qSdo = QPmsBudgetSubjectDO.pmsBudgetSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDetailSubjectVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetDetailSubjectVO.class,
                qdo.id,
                // 项目id
                qdo.projectId,
                // 科目id
                qdo.subjectId,
                /*
                科目明细字段
                 */
                qSdo.subjectCode,
                qSdo.subjectName,
                qSdo.parentName,
                qSdo.parentId.as("subjectParentId"),
                qSdo.subjectStatus,
                qSdo.subjectLevel,
                qSdo.remark
        )).from(qdo).leftJoin(qSdo).on(qSdo.id.eq(qdo.subjectId));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetDetailSubjectVO> getJpaQueryWhere(PmsBudgetDetailSubjectQuery query) {
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        jpaQuery.where(qSdo.deleteFlag.eq(0));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetDetailSubjectQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetDetailSubjectQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 科目id 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectId())) {
            list.add(qdo.subjectId.eq(query.getSubjectId()));
        }
        /** 科目层级 */
        if (!ObjectUtils.isEmpty(query.getSubjectLevel())) {
            list.add(qSdo.subjectLevel.eq(query.getSubjectLevel()));
        }
        /** 父级科目  */
        if (!ObjectUtils.isEmpty(query.getParentId())) {
            list.add(qSdo.parentId.eq(query.getParentId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetDetailSubjectVO queryByKey(Long id) {
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetDetailSubjectVO> queryListDynamic(PmsBudgetDetailSubjectQuery query) {
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetDetailSubjectVO> queryPaging(PmsBudgetDetailSubjectQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetDetailSubjectVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetDetailSubjectVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetDetailSubjectDO save(PmsBudgetDetailSubjectDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetDetailSubjectDO> saveAll(List<PmsBudgetDetailSubjectDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetDetailSubjectPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    public List<PmsBudgetDetailSubjectVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public long getCountByProjectAndParent(Long projectId, Long subjectParentId) {
        JPAQuery<PmsBudgetDetailSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qSdo.parentId.eq(subjectParentId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.stream().count();
    }

    public long deleteByProjectId(Long projectId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.projectId.eq(projectId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

