package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsBudgetSubjectPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsBudgetSubjectQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsBudgetSubjectVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsBudgetSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsBudgetSubjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsBudgetSubjectRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算科目
 *
 * @author xxb
 * @date 2023-04-23
 */
@Repository
@RequiredArgsConstructor
public class PmsBudgetSubjectDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsBudgetSubjectRepo repo;
    private final QPmsBudgetSubjectDO qdo = QPmsBudgetSubjectDO.pmsBudgetSubjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetSubjectVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsBudgetSubjectVO.class,
                qdo.id,
                qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 科目编号
                qdo.subjectCode,
                // 科目名称
                qdo.subjectName,
                // 上一级科目即父主键
                qdo.parentId,
                qdo.parentName,
                // 科目层级
                qdo.subjectLevel,
                // 科目状态
                qdo.subjectStatus
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsBudgetSubjectVO> getJpaQueryWhere(PmsBudgetSubjectQuery query) {
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        //   SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsBudgetSubjectQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsBudgetSubjectQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 科目编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getSubjectCode())) {
            list.add(qdo.subjectCode.like(SqlUtil.toSqlLikeString(query.getSubjectCode())));
        }
        /** 科目名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSubjectName())) {
            list.add(qdo.subjectName.like(SqlUtil.toSqlLikeString(query.getSubjectName())));
        }
        /** 科目层级 */
        if (!ObjectUtils.isEmpty(query.getSubjectLevel())) {
            list.add(qdo.subjectLevel.eq(query.getSubjectLevel()));
        }
        /** 父级科目  */
        if (!ObjectUtils.isEmpty(query.getParentId())) {
            list.add(qdo.parentId.eq(query.getParentId()));
        }
        /** 科目ID */
        if (!ObjectUtils.isEmpty(query.getSubjectIds())) {
            list.add(qdo.id.in(query.getSubjectIds()));
        }
        /** 科目状态 */
        if (!ObjectUtils.isEmpty(query.getSubjectStatus())) {
            list.add(qdo.subjectStatus.eq(query.getSubjectStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsBudgetSubjectVO queryByKey(Long id) {
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据科目编号查询
     *
     * @param subjectCode 科目编号
     * @return 结果
     */
    public PmsBudgetSubjectVO queryByCode(String subjectCode) {
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.subjectCode.eq(subjectCode));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据科目编号查询
     *
     * @param subjectName 科目名称
     * @return 结果
     */
    public PmsBudgetSubjectVO queryByName(String subjectName) {
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.subjectName.eq(subjectName));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsBudgetSubjectVO> queryListDynamic(PmsBudgetSubjectQuery query) {
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsBudgetSubjectVO> queryPaging(PmsBudgetSubjectQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsBudgetSubjectVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsBudgetSubjectVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsBudgetSubjectVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsBudgetSubjectDO save(PmsBudgetSubjectDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsBudgetSubjectDO> saveAll(List<PmsBudgetSubjectDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsBudgetSubjectPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 科目编号
        if (payload.getSubjectCode() != null) {
            update.set(qdo.subjectCode, payload.getSubjectCode());
        }
        // 科目名称
        if (payload.getSubjectName() != null) {
            update.set(qdo.subjectName, payload.getSubjectName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 科目编号
            if (nullFields.contains("subjectCode")) {
                update.setNull(qdo.subjectCode);
            }
            // 科目名称
            if (nullFields.contains("subjectName")) {
                update.setNull(qdo.subjectName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys).or(qdo.parentId.in(keys)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 更改科目状态 0正常  1弃用
     *
     * @param keys 主键集合
     */
    public long updateSubjectStatus(Long[] keys,Integer subjectStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.subjectStatus, subjectStatus)
                .where(qdo.id.in(keys).or(qdo.parentId.in(keys)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

