package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectWbsPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectWbsQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectWbsVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectWbsDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectWbsDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectWbsRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 项目wbs表
 *
 * @author carl
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectWbsDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectWbsRepo repo;
    private final QPmsProjectWbsDO qdo = QPmsProjectWbsDO.pmsProjectWbsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectWbsVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // 父主键
                qdo.parentId,
                // wbs描述（名称）
                qdo.wbsName,
                // wbs类型(WBS,NET,ACT,MS)
                qdo.wbsType,
                // wbs编码
                qdo.wbsCode,
                // node码
                qdo.nodeCode,
                // 父节点Wbs码
                qdo.parentWbsCode,
                qdo.wbsStatus,
                qdo.cancelShowFlag,
                qdo.managerUserId,
                // 成本对象，勾选为1
                qdo.subjectDist,
                // 实际开始时间
                qdo.startDate,
                // 实际结束时间
                qdo.endDate,
                // 预计开始时间
                qdo.preStartDate,
                // 预计结束时间
                qdo.preEndDate,
                // 预计持续时间（天）
                qdo.preDurationDay,
                // 进度(%)
                qdo.wbsProgress,
                // 实际持续时间（天）
                qdo.durationDay,
                // 最早开始时间
                qdo.earlyStartDate,
                // 最早结束时间
                qdo.earlyEndDate,
                // 最晚开始时间
                qdo.lateStartDate,
                // 最晚结束日期
                qdo.lateEndDate,
                // 总浮时
                qdo.totalFloat,
                // 自由浮时
                qdo.freeFloat,
                // 延时
                qdo.delayLag,
                //关键节点
                qdo.isKeyNode,
                //自动排期表示
                qdo.autoScheduling,
                // 成本类型
                qdo.costType,
                // 开票属性，勾选为1
                qdo.invoiceAttr,
                // 成本计划，勾选为1
                qdo.costPlan,
                // wbs权重
                qdo.wbsWeight,
                // 权重设置方式
                qdo.manualSettingWeight,
                qdo.versionId,
                qdo.versionNo,
                qdo.state,
                qdo.effRelateId,
                qdo.releaseFlag,
                qdo.cancelShowFlag,

                // 收款计划ID
                qdo.receivePlanId,
                // 拨付状态
                qdo.allocateStatus,
                // 拨付费用
                qdo.allocateCost,
                // 拨付当量
                qdo.allocateEqva,
                // 已派发当量
                qdo.distedEqva,
                // 规划当量
                qdo.planEqva,
                // 原始规划当量
                qdo.originalPlanEqva,
                // 明细控制标志
                qdo.detailControlFlag,
                // 占用当量数
                qdo.occupyEqva,
                // 已使用当量数
                qdo.usedEqva,
                qdo.usedEqvaProportion,
                qdo.usedAmtProportion

        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectWbsVO> getJpaQueryWhere(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectWbsQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsSearch())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsSearch())).or(qdo.wbsCode.like(SqlUtil.toSqlLikeString(query.getWbsSearch()))));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsName())));
        }
        /** wbs类型(WBS,NET,ACT,MS) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsType())) {
            list.add(qdo.wbsType.eq(query.getWbsType()));
        }
        /** wbs编码 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsCode())) {
            list.add(qdo.wbsCode.like(SqlUtil.toSqlLikeString(query.getWbsCode())));
        }
        /** node码 精确 */
        if (!ObjectUtils.isEmpty(query.getNodeCode())) {
            list.add(qdo.nodeCode.eq(query.getNodeCode()));
        }
        /** 父节点Wbs码 精确 */
        if (!ObjectUtils.isEmpty(query.getParentWbsCode())) {
            list.add(qdo.parentWbsCode.eq(query.getParentWbsCode()));
        }
        if (!ObjectUtils.isEmpty(query.getParentWbsCodes())) {
            list.add(qdo.parentWbsCode.in(query.getParentWbsCodes()));
        }
        /** 成本计划，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getCostPlan())) {
            list.add(qdo.costPlan.eq(query.getCostPlan()));
        }
        /** 科目分配，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getSubjectDist())) {
            list.add(qdo.subjectDist.eq(query.getSubjectDist()));
        }
        /** 开票属性，勾选为1 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAttr())) {
            list.add(qdo.invoiceAttr.eq(query.getInvoiceAttr()));
        }
        /** 持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getDurationDay())) {
            list.add(qdo.durationDay.eq(query.getDurationDay()));
        }

        /** 成本类型 精确 */
        if (!ObjectUtils.isEmpty(query.getCostType())) {
            list.add(qdo.costType.eq(query.getCostType()));
        }
        /** 实际开始时间 精确 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.eq(query.getStartDate()));
        }
        /** 实际结束时间 精确 */
        if (!ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.endDate.eq(query.getEndDate()));
        }
        /** 预计开始时间 大于等于 */
        if (!ObjectUtils.isEmpty(query.getPreStartDate())) {
            list.add(qdo.preStartDate.goe(query.getPreStartDate()));
        }
        /** 预计结束时间 小于等于 */
        if (!ObjectUtils.isEmpty(query.getPreEndDate())) {
            list.add(qdo.preEndDate.loe(query.getPreEndDate()));
        }
        /** 预计持续时间（天） 精确 */
        if (!ObjectUtils.isEmpty(query.getPreDurationDay())) {
            list.add(qdo.preDurationDay.eq(query.getPreDurationDay()));
        }
        /** 进度(%) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsProgress())) {
            list.add(qdo.wbsProgress.eq(query.getWbsProgress()));
        }

        /** 负责人列表 精确 */
        if (!ObjectUtils.isEmpty(query.getManagerUserIds())) {
            list.add(qdo.managerUserId.in(query.getManagerUserIds()));
        }
        /** wbs状态 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsStatus())) {
            list.add(qdo.wbsStatus.eq(query.getWbsStatus()));
        }

        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getState())) {
            list.add(qdo.state.eq(query.getState()));
        }
        /** 版本号 精确 */
        if (!ObjectUtils.isEmpty(query.getVersionNo())) {
            list.add(qdo.versionNo.eq(query.getVersionNo()));
        }
        /** 版本id 精确 */
        if (!ObjectUtils.isEmpty(query.getVersionId())) {
            list.add(qdo.versionId.eq(query.getVersionId()));
        }
        /** 收款计划ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReceivePlanId())) {
            list.add(qdo.receivePlanId.eq(query.getReceivePlanId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectWbsVO queryByKey(Long id) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param keys 主键
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryByKeys(List<Long> keys) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(keys));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据父节点查询
     *
     * @param parentWbsCode 父节点
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryByParentWbsCode(Long projectId, String parentWbsCode) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.parentWbsCode.eq(parentWbsCode));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据父节点查询
     *
     * @param parentWbsCodes 父节点
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryByParentWbsCodes(Long projectId, List<String> parentWbsCodes) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.parentWbsCode.in(parentWbsCodes));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryByProjectId(Long projectId) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public List<PmsProjectWbsVO> queryByProjectIdAndVersion(Long projectId, Integer versionNo) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.where(qdo.versionNo.eq(versionNo));
        return jpaQuery.fetch();
    }

    /**
     * 根据主键查询
     *
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryListByIds(List<Long> keys) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(keys));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectWbsVO> queryListDynamic(PmsProjectWbsQuery query) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }


    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectWbsDO save(PmsProjectWbsDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectWbsDO> saveAll(List<PmsProjectWbsDO> dos) {
        return repo.saveAll(dos);
    }


    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据project主键查询
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectWbsVO> querySimpleByProjectId(Long projectId) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PmsProjectWbsVO.class,
                qdo.id,
                // 计划描述（名称）
                qdo.wbsName,
                // 预计开始时间
                qdo.preStartDate,
                // 预计结束时间
                qdo.preEndDate,
                // 持续时间（天）
                qdo.durationDay
        )).from(qdo);
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 修改节点负责人
     *
     * @param projectId      主集合
     * @param managerUserIds 主集合
     * @return 删除的行数
     */
    public long upadateManagerUserId(Long projectId, List<Long> managerUserIds, Long managerUserId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.managerUserId, managerUserId)
                .where(qdo.projectId.eq(projectId).and(qdo.managerUserId.in(managerUserIds)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新活动进度 状态
     *
     * @param id
     * @param wbsProgress
     * @return
     */
    public long updateWbsProgress(Long id, BigDecimal wbsProgress, String wbsStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.wbsProgress, wbsProgress).set(qdo.wbsStatus, wbsStatus)
                .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 更新活动进度 状态
     *
     * @param id
     * @param wbsProgress
     * @return
     */
    public long updateWbsProgress(Long id, BigDecimal wbsProgress) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.wbsProgress, wbsProgress)
                .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 根据 项目  和 wbs 查找
     *
     * @param projectId
     * @param wbsCode
     * @return
     */
    public PmsProjectWbsVO queryByProjectAndWbs(Long projectId, String wbsCode) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.wbsCode.eq(wbsCode));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }


    /**
     * 根据project主键查询  父节点是null的数据
     *
     * @param projectId project主键
     * @return 结果
     */
    public List<PmsProjectWbsVO> queryParentWbsCodeIsNull(Long projectId) {
        JPAQuery<PmsProjectWbsVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PmsProjectWbsVO.class,
                qdo.id,
                qdo.wbsWeight,
                qdo.wbsProgress
        )).from(qdo);
        jpaQuery.where(qdo.projectId.eq(projectId));
        jpaQuery.where(qdo.parentWbsCode.eq(""));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 更改关键节点
     *
     * @param projectId
     * @return
     */
    public long updateKeyNode(Long projectId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.isKeyNode, 0)
                .where(qdo.projectId.eq(projectId));
        return update.execute();
    }

    public long updateKeyNodeAndVersionNo(Long projectId, Integer versionNo) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.isKeyNode, 0)
                .where(qdo.projectId.eq(projectId))
                .where(qdo.versionNo.eq(versionNo));
        return update.execute();
    }

    /**
     * 更改排期
     *
     * @param projectId
     * @return
     */
    public long updateScheduling(Long projectId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.autoScheduling, 2)
                .where(qdo.projectId.eq(projectId).and(qdo.autoScheduling.eq(1)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public void submit(List<Long> wbsId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.state, "1")
                .where(qdo.id.in(wbsId))
                .where(qdo.state.eq("0"));
        update.execute();

    }


    public Long queryBykeyAndVersion(Long proid, String code, int versionNo) {

        Long res = jpaQueryFactory.select(qdo.id)
                .from(qdo)
                .where(qdo.projectId.eq(proid))
                .where(qdo.wbsCode.eq(code))
                .where(qdo.versionNo.eq(versionNo))
                .fetchFirst();
        return res;
    }

    public Long updateByKeyNoVersion(PmsProjectWbsPayload pmsProjectWbsPayload) {
        JPAUpdateClause update = getJpaUpdateClause(pmsProjectWbsPayload);
        long res = update.where(qdo.id.eq(pmsProjectWbsPayload.getId()))
                .execute();
        return res;

    }

    public Long updateByKeyNoVersionByEffId(PmsProjectWbsPayload pmsProjectWbsPayload) {
        JPAUpdateClause update = getJpaUpdateClause(pmsProjectWbsPayload);
        long res = update.where(qdo.effRelateId.eq(pmsProjectWbsPayload.getEffRelateId()))
            .execute();
        return res;

    }

    private JPAUpdateClause getJpaUpdateClause(PmsProjectWbsPayload pmsProjectWbsPayload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);

        // 项目主键
        if (null != pmsProjectWbsPayload.getProjectId()) {
            update.set(qdo.projectId, pmsProjectWbsPayload.getProjectId());
        }
        // 父主键
        if (null != pmsProjectWbsPayload.getParentId()) {
            update.set(qdo.parentId, pmsProjectWbsPayload.getParentId());
        }
        // wbs描述（名称）
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getWbsName())) {
            update.set(qdo.wbsName, pmsProjectWbsPayload.getWbsName());
        }
        // wbs类型(WBS,NET,ACT,MS)
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getWbsType())) {
            update.set(qdo.wbsType, pmsProjectWbsPayload.getWbsType());
        }
        // wbs编码
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getWbsCode())) {
            update.set(qdo.wbsCode, pmsProjectWbsPayload.getWbsCode());
        }
        // 父节点
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getParentWbsCode())) {
            update.set(qdo.parentWbsCode, pmsProjectWbsPayload.getParentWbsCode());
        }
        // node码
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getNodeCode())) {
            update.set(qdo.nodeCode, pmsProjectWbsPayload.getNodeCode());
        }
        // 节点负责人
        if (null != pmsProjectWbsPayload.getManagerUserId()) {
            update.set(qdo.managerUserId, pmsProjectWbsPayload.getManagerUserId());
        }
        // 成本对象，勾选为1
        if (null != pmsProjectWbsPayload.getSubjectDist()) {
            update.set(qdo.subjectDist, pmsProjectWbsPayload.getSubjectDist());
        }
        // 成本类型
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getCostType())) {
            update.set(qdo.costType, pmsProjectWbsPayload.getCostType());
        }
        // 预计开始时间
        if (null != pmsProjectWbsPayload.getPreStartDate()) {
            update.set(qdo.preStartDate, pmsProjectWbsPayload.getPreStartDate());
        }
        // 预计结束时间
        if (null != pmsProjectWbsPayload.getPreEndDate()) {
            update.set(qdo.preEndDate, pmsProjectWbsPayload.getPreEndDate());
        }
        // 预计持续时间（天）
        if (null != pmsProjectWbsPayload.getPreDurationDay()) {
            update.set(qdo.preDurationDay, pmsProjectWbsPayload.getPreDurationDay());
        }
        // 进度(%)
        if (null != pmsProjectWbsPayload.getWbsProgress()) {
            update.set(qdo.wbsProgress, pmsProjectWbsPayload.getWbsProgress());
        }
        // 实际开始时间
        if (null != pmsProjectWbsPayload.getStartDate()) {
            update.set(qdo.startDate, pmsProjectWbsPayload.getStartDate());
        }
        // 实际结束时间
        if (null != pmsProjectWbsPayload.getEndDate()) {
            update.set(qdo.endDate, pmsProjectWbsPayload.getEndDate());
        }
        if (null != pmsProjectWbsPayload.getEndDateIsNull()) {
            update.setNull(qdo.endDate);
        }
        // 实际持续时间（天）
        if (null != pmsProjectWbsPayload.getDurationDay()) {
            update.set(qdo.durationDay, pmsProjectWbsPayload.getDurationDay());
        }
        // 最早开始时间
        if (null != pmsProjectWbsPayload.getEarlyStartDate()) {
            update.set(qdo.earlyStartDate, pmsProjectWbsPayload.getEarlyStartDate());
        }
        // 最晚开始时间
        if (null != pmsProjectWbsPayload.getLateStartDate()) {
            update.set(qdo.lateStartDate, pmsProjectWbsPayload.getLateStartDate());
        }
        // 最晚结束日期
        if (null != pmsProjectWbsPayload.getLateEndDate()) {
            update.set(qdo.lateEndDate, pmsProjectWbsPayload.getLateEndDate());
        }
        // 总浮时
        if (null != pmsProjectWbsPayload.getTotalFloat()) {
            update.set(qdo.totalFloat, pmsProjectWbsPayload.getTotalFloat());
        }
        // 自由浮时
        if (null != pmsProjectWbsPayload.getFreeFloat()) {
            update.set(qdo.freeFloat, pmsProjectWbsPayload.getFreeFloat());
        }
        // 延时
        if (null != pmsProjectWbsPayload.getDelayLag()) {
            update.set(qdo.delayLag, pmsProjectWbsPayload.getDelayLag());
        }
        // 是否是关键节点0:否，1：是
        if (null != pmsProjectWbsPayload.getIsKeyNode()) {
            update.set(qdo.isKeyNode, pmsProjectWbsPayload.getIsKeyNode());
        }
        // 是否自动排期0：未进行，1：已排期，2：已过期
        if (null != pmsProjectWbsPayload.getAutoScheduling()) {
            update.set(qdo.autoScheduling, pmsProjectWbsPayload.getAutoScheduling());
        }
        // 状态
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getWbsStatus())) {
            update.set(qdo.wbsStatus, pmsProjectWbsPayload.getWbsStatus());
        }
        // 最早结束时间
        if (null != pmsProjectWbsPayload.getEarlyEndDate()) {
            update.set(qdo.earlyEndDate, pmsProjectWbsPayload.getEarlyEndDate());
        }
        // 开票属性，勾选为1
        if (null != pmsProjectWbsPayload.getInvoiceAttr()) {
            update.set(qdo.invoiceAttr, pmsProjectWbsPayload.getInvoiceAttr());
        }
        // wbs权重
        if (null != pmsProjectWbsPayload.getWbsWeight()) {
            update.set(qdo.wbsWeight, pmsProjectWbsPayload.getWbsWeight());
        }
        // wbs权重设置方式：1为手动设置  0为系统设置
        if (null != pmsProjectWbsPayload.getManualSettingWeight()) {
            update.set(qdo.manualSettingWeight, pmsProjectWbsPayload.getManualSettingWeight());
        }
        // 成本计划，勾选为1
        if (null != pmsProjectWbsPayload.getCostPlan()) {
            update.set(qdo.costPlan, pmsProjectWbsPayload.getCostPlan());
        }
        // 状态 0草稿 1提交
        if (StringUtils.isNotEmpty(pmsProjectWbsPayload.getState())) {
            update.set(qdo.state, pmsProjectWbsPayload.getState());
        }
        if (pmsProjectWbsPayload.getEffRelateId() != null) {
            update.set(qdo.effRelateId, pmsProjectWbsPayload.getEffRelateId());
        }
        if (pmsProjectWbsPayload.getReleaseFlag() != null) {
            update.set(qdo.releaseFlag, pmsProjectWbsPayload.getReleaseFlag());
        }
        if (pmsProjectWbsPayload.getCancelShowFlag() != null) {
            update.set(qdo.cancelShowFlag, pmsProjectWbsPayload.getCancelShowFlag());
        }
        // 收款计划ID
        if (pmsProjectWbsPayload.getReceivePlanId() != null) {
            update.set(qdo.receivePlanId, pmsProjectWbsPayload.getReceivePlanId());
        }
        // 拨付状态
        if (pmsProjectWbsPayload.getAllocateStatus() != null) {
            update.set(qdo.allocateStatus, pmsProjectWbsPayload.getAllocateStatus());
        }
        // 拨付当量
        if (pmsProjectWbsPayload.getAllocateEqva() != null) {
            update.set(qdo.allocateEqva, pmsProjectWbsPayload.getAllocateEqva());
        }
        // 拨款金额
        if (pmsProjectWbsPayload.getAllocateCost() != null) {
            update.set(qdo.allocateCost, pmsProjectWbsPayload.getAllocateCost());
        }
        // 原始当量
        if (pmsProjectWbsPayload.getOriginalPlanEqva() != null) {
            update.set(qdo.originalPlanEqva, pmsProjectWbsPayload.getOriginalPlanEqva());
        }
        // 规划当量
        if (pmsProjectWbsPayload.getPlanEqva() != null) {
            update.set(qdo.planEqva, pmsProjectWbsPayload.getPlanEqva());
        }
        // 备注
        if (pmsProjectWbsPayload.getRemark() != null) {
            update.set(qdo.remark, pmsProjectWbsPayload.getRemark());
        }
        // 明细控制标志
        if (pmsProjectWbsPayload.getDetailControlFlag() != null) {
            update.set(qdo.detailControlFlag, pmsProjectWbsPayload.getDetailControlFlag());
        }
        // 占用当量数
        if (pmsProjectWbsPayload.getOccupyEqva() != null) {
            update.set(qdo.occupyEqva, pmsProjectWbsPayload.getOccupyEqva());
        }
        // 已使用当量数
        if (pmsProjectWbsPayload.getUsedEqva() != null) {
            update.set(qdo.usedEqva, pmsProjectWbsPayload.getUsedEqva());
        }
        if (pmsProjectWbsPayload.getUsedEqvaProportion() != null) {
            update.set(qdo.usedEqvaProportion, pmsProjectWbsPayload.getUsedEqvaProportion());
        }
        if (pmsProjectWbsPayload.getUsedAmtProportion() != null) {
            update.set(qdo.usedAmtProportion, pmsProjectWbsPayload.getUsedAmtProportion());
        }
        return update;
    }

    /**
     * 更新wbs状态
     *
     * @param id             主键
     * @param wbsStatus      wbs状态
     * @param cancelShowFlag
     * @return {@link Long}
     */
    public Long updateWbsStatus(Long id, String wbsStatus, String state, Boolean cancelShowFlag) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.wbsStatus, wbsStatus);
        if (StringUtils.isNotBlank(state)) {
            update.set(qdo.state, state);
        }
        if (!ObjectUtils.isEmpty(cancelShowFlag)) {
            update.set(qdo.cancelShowFlag, cancelShowFlag);
        }
        update.where(qdo.id.eq(id));
        return update.execute();
    }

    public Long updateWbsStatusAndEndDateIsNull(Long id, String wbsStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.wbsStatus, wbsStatus)
            .setNull(qdo.endDate)
            .where(qdo.id.eq(id));
        return update.execute();
    }

    public void delByEffIds(List<Long> effIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(effIds));
        update.execute();
    }

    /***
     * 批量更改拨付状态
     * @param keys
     * @param status
     * @return
     */
    public long updateAllocateStatus(List<Long> keys, String allocateStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.allocateStatus, allocateStatus)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long updateAllocateStatusByEffId(List<Long> wbsIdList, String allocateStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.allocateStatus, allocateStatus)
            .where(qdo.effRelateId.in(wbsIdList));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

