package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsWbsActCostPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsWbsActCostQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsWbsActCostVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsWbsActCostDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsWbsActCostDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsWbsActCostRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * wbs活动成预估
 *
 * @author carl
 * @date 2023-04-06
 */
@Repository
@RequiredArgsConstructor
public class PmsWbsActCostDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsWbsActCostRepo repo;
    private final QPmsWbsActCostDO qdo = QPmsWbsActCostDO.pmsWbsActCostDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsActCostVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsWbsActCostVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                // wbs主键(仅活动类型)
                qdo.wbsId,
                // wbs描述（名称）
                qdo.wbsName,
                // 成本要素
                qdo.costName,
                // 成本预估
                qdo.costEstimate
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsWbsActCostVO> getJpaQueryWhere(PmsWbsActCostQuery query) {
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsWbsActCostQuery query) {
        long total = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .where(where(query))
                .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsWbsActCostQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** wbs主键(仅活动类型) 精确 */
        if (!ObjectUtils.isEmpty(query.getWbsId())) {
            list.add(qdo.wbsId.eq(query.getWbsId()));
        }
        /** wbs描述（名称） 模糊 */
        if (!ObjectUtils.isEmpty(query.getWbsName())) {
            list.add(qdo.wbsName.like(SqlUtil.toSqlLikeString(query.getWbsName())));
        }
        /** 成本要素 模糊 */
        if (!ObjectUtils.isEmpty(query.getCostName())) {
            list.add(qdo.costName.like(SqlUtil.toSqlLikeString(query.getCostName())));
        }
        /** 成本预估 模糊 */
        if (!ObjectUtils.isEmpty(query.getCostEstimate())) {
            list.add(qdo.costEstimate.eq(query.getCostEstimate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsWbsActCostVO queryByKey(Long id) {
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据ProjectId查询
     *
     * @param ProjectId Project主键
     * @return 结果
     */
    public List<PmsWbsActCostVO> queryByProjectId(Long ProjectId) {
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.projectId.eq(ProjectId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 根据wbsId查询
     *
     * @param wbsId wbs主键
     * @return 结果
     */
    public List<PmsWbsActCostVO> queryByWbsId(Long wbsId) {
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.wbsId.eq(wbsId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsWbsActCostVO> queryListDynamic(PmsWbsActCostQuery query) {
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsWbsActCostVO> queryPaging(PmsWbsActCostQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsWbsActCostVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsWbsActCostVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsWbsActCostVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsWbsActCostDO save(PmsWbsActCostDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsWbsActCostDO> saveAll(List<PmsWbsActCostDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsWbsActCostPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // wbs主键(仅活动类型)
        if (payload.getWbsId() != null) {
            update.set(qdo.wbsId, payload.getWbsId());
        }
        // wbs描述（名称）
        if (payload.getWbsName() != null) {
            update.set(qdo.wbsName, payload.getWbsName());
        }
        // 成本要素
        if (payload.getCostName() != null) {
            update.set(qdo.costName, payload.getCostName());
        }
        // 成本预估
        if (payload.getCostEstimate() != null) {
            update.set(qdo.costEstimate, payload.getCostEstimate());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // wbs主键(仅活动类型)
            if (nullFields.contains("wbsId")) {
                update.setNull(qdo.wbsId);
            }
            // wbs描述（名称）
            if (nullFields.contains("wbsName")) {
                update.setNull(qdo.wbsName);
            }
            // 成本要素
            if (nullFields.contains("costName")) {
                update.setNull(qdo.costName);
            }
            // 成本预估
            if (nullFields.contains("costEstimate")) {
                update.setNull(qdo.costEstimate);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除根据wbs主键
     *
     * @param wbsIds
     * @return 删除的行数
     */
    public long deleteSoftByWbsIds(List<Long> wbsIds) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.wbsId.in(wbsIds));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

