package com.elitesland.tw.tw5.server.prd.pms.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsWbsTemplateNodePayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsWbsTemplateNodeQuery;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsWbsTemplateNodeService;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsWbsTemplateNodeVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;

import com.elitesland.tw.tw5.server.prd.pms.convert.PmsWbsTemplateNodeConvert;
import com.elitesland.tw.tw5.server.prd.pms.dao.PmsWbsTemplateNodeDAO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsWbsTemplateNodeDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsWbsTemplateNodeRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * wbs模板节点
 *
 * @author Echo
 * @date 2023-03-29
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsWbsTemplateNodeServiceImpl extends BaseServiceImpl implements PmsWbsTemplateNodeService {

    private final PmsWbsTemplateNodeRepo pmsWbsTemplateNodeRepo;
    private final PmsWbsTemplateNodeDAO pmsWbsTemplateNodeDAO;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchInsert(List<PmsWbsTemplateNodePayload> nodePayloads, Long templateId) {
        //根据模板id删除节点
        pmsWbsTemplateNodeDAO.deleteSoftByTemplateId(templateId);
        //修改节点编码
        setNodeCode(nodePayloads,templateId);
    }

    @Override
    public List<PmsWbsTemplateNodeVO> queryByTemplateId(Long templateId) {
        return pmsWbsTemplateNodeDAO.queryByTemplateIds(List.of(templateId));
    }

    @Override
    public PagingVO<PmsWbsTemplateNodeVO> paging(PmsWbsTemplateNodeQuery query){
        Page<PmsWbsTemplateNodeDO> page = pmsWbsTemplateNodeRepo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root,query,criteriaBuilder),query.getPageRequest());
        return PageUtil.toPageVo(page.map(PmsWbsTemplateNodeConvert.INSTANCE::toVo));
    }

    @Override
    public PagingVO<PmsWbsTemplateNodeVO> queryPaging(PmsWbsTemplateNodeQuery query){
        return pmsWbsTemplateNodeDAO.queryPaging(query);
    }

    @Override
    public List<PmsWbsTemplateNodeVO> queryList(PmsWbsTemplateNodeQuery query){
        return PmsWbsTemplateNodeConvert.INSTANCE.toVoList(
                pmsWbsTemplateNodeRepo.findAll(
                (root, criteriaQuery, criteriaBuilder)
                    -> QueryHelp.getPredicate(root, query, criteriaBuilder)
                , query.getPageRequest().getSort()
             )
        );
    }

    @Override
    public List<PmsWbsTemplateNodeVO> queryListDynamic(PmsWbsTemplateNodeQuery query){
        return pmsWbsTemplateNodeDAO.queryListDynamic(query);
    }

    @Override
    public PmsWbsTemplateNodeVO queryByKey(Long key) {
        PmsWbsTemplateNodeDO entity = pmsWbsTemplateNodeRepo.findById(key).orElseGet(PmsWbsTemplateNodeDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsWbsTemplateNodeVO vo = PmsWbsTemplateNodeConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTemplateNodeVO insert(PmsWbsTemplateNodePayload payload) {
        PmsWbsTemplateNodeDO entityDo = PmsWbsTemplateNodeConvert.INSTANCE.toDo(payload);
        return PmsWbsTemplateNodeConvert.INSTANCE.toVo(pmsWbsTemplateNodeRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTemplateNodeVO update(PmsWbsTemplateNodePayload payload) {
        PmsWbsTemplateNodeDO entity = pmsWbsTemplateNodeRepo.findById(payload.getId()).orElseGet(PmsWbsTemplateNodeDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PmsWbsTemplateNodeDO entityDo = PmsWbsTemplateNodeConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return PmsWbsTemplateNodeConvert.INSTANCE.toVo(pmsWbsTemplateNodeRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<PmsWbsTemplateNodeDO> optional = pmsWbsTemplateNodeRepo.findById(id);
                if (!optional.isEmpty()) {
                    PmsWbsTemplateNodeDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    pmsWbsTemplateNodeRepo.save(entity);
                }
             });
        }
    }



    /**
     * 修改节点编码
     *
     * @param nodes 节点集合
     * @param templateId      wbs模板id
     */
    private void setNodeCode(List<PmsWbsTemplateNodePayload> nodes,Long templateId) {
        List<PmsWbsTemplateNodeDO> dNodes = new ArrayList<>();
        //遍历节点集合(该数据为必填，无需判断是否为空）
        nodes.forEach(node -> {
            //绑定模板id
            node.setTemplateId(templateId);
            //拼接nodeCode字符串并设置
            node.setNodeCode("TEMP-[" + templateId + "]-" + node.getWbsType() + "-" + node.getWbsCode());
            dNodes.add(PmsWbsTemplateNodeConvert.INSTANCE.toDo(node));
        });
        List<PmsWbsTemplateNodeVO> templateNodeVOS = pmsWbsTemplateNodeDAO.queryByTemplateIds(Arrays.asList(templateId));
        checkTemplateNode(nodes,templateNodeVOS);
        pmsWbsTemplateNodeDAO.saveAll(dNodes);
    }

    /**
     * 核验节点数据
     *
     * @param templateNodePayload
     * @param templateNodeVOS
     */
    void checkTemplateNode(List<PmsWbsTemplateNodePayload> templateNodePayload,List<PmsWbsTemplateNodeVO> templateNodeVOS) {
        List<String> collect = templateNodePayload.stream().map(PmsWbsTemplateNodePayload::getWbsName).distinct().collect(Collectors.toList());
        //获取WbsName判断长度是否相等，如果不等则说明有重复元素，抛异常
        if (templateNodePayload.size() != collect.size()) {
            throw TwException.error("", "节点描述不可重复，请核验！");
        }
        List<String> collect0 = templateNodePayload.stream().map(PmsWbsTemplateNodePayload::getWbsCode).distinct().collect(Collectors.toList());
        //WbsCode判断长度是否相等，如果不等则说明有重复元素，抛异常
        if (templateNodePayload.size() != collect0.size()) {
            throw TwException.error("", "wbs编码不可重复，请核验！");
        }
        //遍历模板节点集合
        templateNodePayload.forEach(entityDo -> {
            //判断wbs编码是否为空
            if (ObjectUtils.isEmpty(entityDo.getWbsCode())) {
                throw TwException.error("", "wbs编码不存在，请核验！");
            }
            //判断wbs类型是否为空
            if (ObjectUtils.isEmpty(entityDo.getWbsType())) {
                throw TwException.error("", "wbs类型不存在，请核验！");
            }
            //判断父节点编码是否为空
            if (ObjectUtils.isEmpty(entityDo.getParentWbsCode())) {
                //如果为空，说明是一级节点，判断一级节点只能是WBS和网络
                if (!entityDo.getWbsType().equals("WBS") && !entityDo.getWbsType().equals("NET")) {
                    throw TwException.error("", "一级节点只可新建“WBS元素”和“网络”，请核验！");
                }
                //如果父节点不为空，则说明是子节点
            } else {
                Optional<PmsWbsTemplateNodePayload> first = templateNodePayload.stream().filter(wbsPayload -> wbsPayload.getWbsCode().equals(entityDo.getParentWbsCode())).findFirst();
                if (first.isPresent()) {
                    //获得父节点的编码
                    PmsWbsTemplateNodePayload parentWbsPayload = first.get();
                    if (entityDo.getWbsType().equals("WBS")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS")) {
                            throw TwException.error("", "WBS元素上级只能是WBS元素，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("NET")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS")) {
                            throw TwException.error("", "网络上级只能是WBS元素，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("ACT")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS") && !parentWbsPayload.getWbsType().equals("NET")) {
                            throw TwException.error("", "活动上级只能是WBS元素或网络，请核验！");
                        }
                    }
                    if (entityDo.getWbsType().equals("MS")) {
                        if (!parentWbsPayload.getWbsType().equals("WBS") && !parentWbsPayload.getWbsType().equals("NET")) {
                            throw TwException.error("", "里程碑上级只能是WBS元素或网络，请核验！");
                        }
                    }
                } else {
                    throw TwException.error("", entityDo.getParentWbsCode() + "编码数据不存在，请核验！");
                }
            }

            if (!ObjectUtils.isEmpty(templateNodeVOS)) {

                PmsWbsTemplateNodeVO first = null;
                PmsWbsTemplateNodeVO first0 = null;
                for (PmsWbsTemplateNodeVO wbsVO : templateNodeVOS) {
                    Optional<PmsWbsTemplateNodePayload> firstPayload = templateNodePayload.stream().filter(wbsPayload -> wbsPayload.getId().equals(wbsVO.getId())).findFirst();
                    if (firstPayload.isPresent()) {
                        wbsVO.setWbsCode(firstPayload.get().getWbsCode());
                        wbsVO.setWbsName(firstPayload.get().getWbsName());
                    }
                    if (wbsVO.getWbsCode().equals(entityDo.getWbsCode())) {
                        first = wbsVO;
                    }
                    if (wbsVO.getWbsName().equals(entityDo.getWbsName())) {
                        first0 = wbsVO;
                    }
                }
                if (first != null) {
                    if (entityDo.getId() == null || !first.getId().equals(entityDo.getId())) {
                        throw TwException.error("", "【" + entityDo.getWbsName() + "】wbs编码已存在，请核验！");
                    }
                }
                if (first0 != null) {
                    if (entityDo.getId() == null || !first0.getId().equals(entityDo.getId())) {
                        throw TwException.error("", "【" + entityDo.getWbsName() + "】节点描述已存在，请核验！");
                    }
                }
            }

        });
    }
}
