package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitesland.tw.tw5.api.prd.salecon.payload.ConEpibolyCostConPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConEpibolyCostConQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConEpibolyCostConVO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmOpportunityDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.QPrjProjectDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConEpibolyCostConDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConEpibolyCostConDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConEpibolyCostConRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 外包费用确认单
 *
 * @author likunpeng
 * @date 2023-04-17
 */
@Repository
@RequiredArgsConstructor
public class ConEpibolyCostConDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConEpibolyCostConRepo repo;
    private final QConEpibolyCostConDO qdo = QConEpibolyCostConDO.conEpibolyCostConDO;
    private final QSaleConContractDO qdoCon = QSaleConContractDO.saleConContractDO;
    private final QCrmOpportunityDO qdoOppo = QCrmOpportunityDO.crmOpportunityDO;
    private final QPrjProjectDO qdoProj = QPrjProjectDO.prjProjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConEpibolyCostConVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConEpibolyCostConVO.class,
                qdo.id,
                qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 合同ID
                qdo.contractId,
                // 合同编号
                qdoCon.code.as("contractCode"),
                // 合同名称
                qdoCon.name.as("contractName"),
                // 外包费用单号
                qdo.epibolyCostNo,
                // 申请人
                qdo.applyUserId,
                // 申请日期
                qdo.applyDate,
                qdo.costConIdV4,
                // 申请BU
                qdo.applyBuId,
                // 商机id
                qdoCon.oppoId,
                // 商机名称
                qdoProj.projectName.as("oppoName"),
                // 商机编号
                qdoProj.projectNo.as("oppoNo")
        )).from(qdo)
                .leftJoin(qdoCon).on(qdo.contractId.longValue().eq(qdoCon.id.longValue()))
                .leftJoin(qdoOppo).on(qdoCon.oppoId.longValue().eq(qdoOppo.id.longValue()))
                .leftJoin(qdoProj).on(qdoOppo.projectId.longValue().eq(qdoProj.id.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConEpibolyCostConVO> getJpaQueryWhere(ConEpibolyCostConQuery query) {
        JPAQuery<ConEpibolyCostConVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConEpibolyCostConQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConEpibolyCostConQuery query){
        List<Predicate> list = new ArrayList<>();
        list.add(qdo.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 合同ID 精确 */
        if (!ObjectUtils.isEmpty(query.getContractId())) {
            list.add(qdo.contractId.eq(query.getContractId()));
        }
        /** 外包费用单号 模糊 */
        if (!ObjectUtils.isEmpty(query.getEpibolyCostNo())) {
             list.add(qdo.epibolyCostNo.like(SqlUtil.toSqlLikeString(query.getEpibolyCostNo())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConEpibolyCostConVO queryByKey(Long id) {
        JPAQuery<ConEpibolyCostConVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConEpibolyCostConVO> queryListDynamic(ConEpibolyCostConQuery query) {
        JPAQuery<ConEpibolyCostConVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConEpibolyCostConVO> queryPaging(ConEpibolyCostConQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConEpibolyCostConVO> jpaQuery = getJpaQueryWhere(query);
        List<ConEpibolyCostConVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConEpibolyCostConVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConEpibolyCostConDO save(ConEpibolyCostConDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConEpibolyCostConDO> saveAll(List<ConEpibolyCostConDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConEpibolyCostConPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 合同ID
        if (payload.getContractId() != null) {
            update.set(qdo.contractId, payload.getContractId());
        }
        // 外包费用单号
        if (payload.getEpibolyCostNo() != null) {
            update.set(qdo.epibolyCostNo, payload.getEpibolyCostNo());
        }
        // 渠道费用备注
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 申请日期
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 申请Bu_Id
        if (payload.getApplyBuId() != null) {
            update.set(qdo.applyBuId, payload.getApplyBuId());
        }
        // 申请Bu_Id
        if (payload.getCostConIdV4() != null) {
            update.set(qdo.costConIdV4, payload.getCostConIdV4());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 合同ID
            if (nullFields.contains("contractId")) {
                update.setNull(qdo.contractId);
            }
            // 外包费用单号
            if (nullFields.contains("epibolyCostNo")) {
                update.setNull(qdo.epibolyCostNo);
            }
            // 渠道费用备注
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
            // 申请日期
            if (nullFields.contains("applyDate")) {
                update.setNull(qdo.applyDate);
            }
            // 申请Bu_Id
            if (nullFields.contains("applyBuId")) {
                update.setNull(qdo.applyBuId);
            }
            // 申请Bu_Id
            if (nullFields.contains("costConIdV4")) {
                update.setNull(qdo.costConIdV4);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据合同ID查询外包费用单
     * @param saleConId
     * @return
     */
    public ConEpibolyCostConVO queryByContractId(Long saleConId) {
        JPAQuery<ConEpibolyCostConVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.contractId.eq(saleConId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        jpaQuery.groupBy(qdo.id);
        return jpaQuery.fetchOne();
    }

    /**
     * 修改子合同有效合同金额
     * @param newEffectiveAmt
     * @param contractId
     */
    public void updateSubConEffectiveAmt(BigDecimal newEffectiveAmt, Long contractId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdoCon);
        update.set(qdoCon.effectiveAmt,newEffectiveAmt);
        update.where(qdoCon.id.eq(contractId)).execute();
    }
}

