package com.elitesland.tw.tw5.server.prd.system.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.product.query.PrdProductCaseQuery;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemBusinessObjectQuery;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemBusinessObjectVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemNewFunctionVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemPermissionFieldObjRoleFunctionVO;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionTypeEnum;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.system.entity.*;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemBusinessObjectMenuRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 业务对象DAO
 *
 * @Author Bill
 * @Date 2023/9/15 11:25
 **/
@Repository
@RequiredArgsConstructor
public class PrdSystemBusinessObjectDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final QPrdSystemMenuDO qdoMenu = QPrdSystemMenuDO.prdSystemMenuDO;
    private final QPrdSystemBusinessObjectMenuDO qdoBusinessObejctMenu = QPrdSystemBusinessObjectMenuDO.prdSystemBusinessObjectMenuDO;
    private final QPrdSystemBusinessObjectDO qdo = QPrdSystemBusinessObjectDO.prdSystemBusinessObjectDO;
    private final PrdSystemBusinessObjectMenuRepo repoMenu;
    private final QPrdSystemNewFunctionDO qdoNewFunction = QPrdSystemNewFunctionDO.prdSystemNewFunctionDO;
    private final QPrdSystemPermissionFieldObjRoleFunctionDO qdofieldObjRoleFunctionDO = QPrdSystemPermissionFieldObjRoleFunctionDO.prdSystemPermissionFieldObjRoleFunctionDO;
    private final QPrdSystemRoleFunctionDO qdoRoleFunction = QPrdSystemRoleFunctionDO.prdSystemRoleFunctionDO;
    private final QPrdSystemPermissionFieldDO qdoField = QPrdSystemPermissionFieldDO.prdSystemPermissionFieldDO;
    private final QPrdSystemUserRoleDO qdoUserRole = QPrdSystemUserRoleDO.prdSystemUserRoleDO;
    private final QPrdSystemPermissionFunctionObjectDO qdoFunctionObject = QPrdSystemPermissionFunctionObjectDO.prdSystemPermissionFunctionObjectDO;

    /**
     * 查询
     *
     * @param id
     * @return
     */
    public List<Long> queryBusinessObjectMenuIds(Long id) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdoBusinessObejctMenu.menuId).from(qdoBusinessObejctMenu)
                .leftJoin(qdoMenu).on(qdoBusinessObejctMenu.menuId.longValue().eq(qdoMenu.id.longValue()));
        jpaQuery.where(qdoMenu.menuStatus.eq(0));
        jpaQuery.where(qdoMenu.deleteFlag.eq(0));
        jpaQuery.where(qdoBusinessObejctMenu.objectId.eq(id));
        return jpaQuery.fetch();
    }

    /**
     * 根据对象编号查询业务对象
     * @param objectCode
     * @return
     */
    public PrdSystemBusinessObjectVO queryByObjectCode(String objectCode) {
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.objectCode.eq(objectCode));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    public List<PrdSystemBusinessObjectVO> queryByObjectCodes(List<String> objectCodes){
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.objectCode.in(objectCodes));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    public void deleteByCodeNotIn(List<String> objectCodes){
        jpaQueryFactory.delete(qdo)
                .where(qdo.objectCode.notIn(objectCodes).and(qdo.deleteFlag.eq(0)))
                .execute();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemBusinessObjectVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemBusinessObjectVO.class,
                qdo.id,
                qdo.objectCode,
                qdo.objectName,
                qdo.objectStatus,
                qdo.objectVersion,
                qdo.remark,
                qdo.deleteFlag,
                qdo.createUserId,
                qdo.createTime
        )).from(qdo);
    }

    /**
     * 业务对象 删除菜单
     * @param id
     * @return 删除的行数
     */
    public long deleteBusinessObjectMenusByObjectId(Long id) {
        JPADeleteClause jpaDeleteClause = jpaQueryFactory.delete(qdoBusinessObejctMenu).where(qdoBusinessObejctMenu.objectId.eq(id));
        return jpaDeleteClause.execute();
    }

    /**
     * 业务对象 全插
     * @param businessObjectMenuDOS
     * @return 保存后的对象
     */
    public List<PrdSystemBusinessObjectMenuDO> saveBusinessObjectMenuAll(List<PrdSystemBusinessObjectMenuDO> businessObjectMenuDOS) {
        return repoMenu.saveAll(businessObjectMenuDOS);
    }

    /**
     * 对象列表 模糊查询
     * @param query
     * @return
     */
    public PagingVO<PrdSystemBusinessObjectVO> queryByKeyword(PrdSystemBusinessObjectQuery query) {
        long total = count(query);
        if (total == 0){
            return PagingVO.empty();
        }
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdSystemBusinessObjectVO> result = jpaQuery.offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdSystemBusinessObjectVO>builder().records(result).total(total).build();
    }

    /**
     * 统计
     * @param query
     * @return
     */
    private long count(PrdSystemBusinessObjectQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        //常用查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        Long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 拼装条件
     * @param query
     * @return
     */
    private JPAQuery<PrdSystemBusinessObjectVO> getJpaQueryWhere(PrdSystemBusinessObjectQuery query) {
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuerySelect = getJpaQuerySelect();
        //条件封装
        jpaQuerySelect.where(where(query));
        //常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuerySelect, qdo._super, query);
        //动态排序
        jpaQuerySelect.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuerySelect;
    }

    /**
     * 查询条件拼接
     * @param query
     * @return
     */
    private Predicate where(PrdSystemBusinessObjectQuery query) {
        List<Predicate> list = new ArrayList<>();
        if (StringUtils.hasText(query.getKeyword())){
            list.add(qdo.objectCode.like(SqlUtil.toSqlLikeString(query.getKeyword()))
                    .or(qdo.objectName.like(SqlUtil.toSqlLikeString(query.getKeyword()))));;
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemNewFunctionVO> getJpaQueryNewFunctionSelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemNewFunctionVO.class,
                qdoNewFunction.id,
                qdoNewFunction.functionName,
                qdoNewFunction.functionCode,
                qdoNewFunction.functionType,
                qdoNewFunction.createTime,
                qdoNewFunction.modifyTime
        )).from(qdoNewFunction);
    }

    /**
     * 查询业务对象下的功能信息
     * @param id
     * @return
     */
    public List<PrdSystemNewFunctionVO> queryFunctionByObjectId(Long id) {
        JPAQuery<PrdSystemNewFunctionVO> jpaQuery = getJpaQueryNewFunctionSelect();
        jpaQuery.where(qdoNewFunction.objectId.longValue().eq(id).and(qdoNewFunction.functionStatus.eq(0))
                .and(qdoNewFunction.deleteFlag.eq(0)));
        return jpaQuery.fetch();
    }

    /**
     * 获取角色集中不重复的功能
     * @param roleIds
     * @return
     */
    public List<PrdSystemNewFunctionVO> getDistinctFunctionByRoleIds(Set<Long> roleIds) {
        JPAQuery<PrdSystemNewFunctionVO> jpaQuery = getJpaQueryDistinctFunction();
        jpaQuery.where(qdoRoleFunction.roleId.longValue().in(roleIds));
        return jpaQuery.fetch();
    }

    /**
     * 获取角色集中不重复的功能的 查询参数拼接
     * @return
     */
    private JPAQuery<PrdSystemNewFunctionVO> getJpaQueryDistinctFunction() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemNewFunctionVO.class,
                qdoNewFunction.functionType,
                qdoNewFunction.functionCode,
                qdoNewFunction.functionName,
                qdoNewFunction.clientType,
                qdoNewFunction.id,
                qdoNewFunction.parentFunctionCode
                )).from(qdoRoleFunction).leftJoin(qdoNewFunction).on(qdoNewFunction.id.longValue().eq(qdoRoleFunction.functionId.longValue()))
                .groupBy(qdoNewFunction.id);
    }

    /**
     * 查询角色集 不重复的字段权限规则
     * @param roleIds
     * @return
     */
    public List<PrdSystemPermissionFieldObjRoleFunctionVO> getDistinctFieldRuleByRoleIds(Set<Long> roleIds) {
        JPAQuery<PrdSystemPermissionFieldObjRoleFunctionVO> jpaQuery = getJpaQueryDistinctFieldRule();
        jpaQuery.where(qdofieldObjRoleFunctionDO.roleId.longValue().in(roleIds));
        return jpaQuery.fetch();
    }

    /**
     * 查询角色集 不重复的字段权限 拼接插叙参数
     * @return
     */
    private JPAQuery<PrdSystemPermissionFieldObjRoleFunctionVO> getJpaQueryDistinctFieldRule() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemPermissionFieldObjRoleFunctionVO.class,
                qdofieldObjRoleFunctionDO.fieldId,
                qdofieldObjRoleFunctionDO.isEdit,
                qdofieldObjRoleFunctionDO.isVisible,
                qdoField.fieldName,
                qdoField.functionObjectId,
                qdoField.field.as("fieldKey"),
                qdoField.fieldType,
                qdoField.fieldTypeName
                )).from(qdofieldObjRoleFunctionDO).leftJoin(qdoField).on(qdofieldObjRoleFunctionDO.fieldId.longValue().eq(qdoField.id.longValue()))
                .groupBy(qdofieldObjRoleFunctionDO.fieldId);
    }

    /**
     * 通过用户主键查询 角色集
     *
     * @param id
     * @return
     */
    public List<Long> getRoleIdsByUserId(Long id) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdoUserRole.roleId).from(qdoUserRole).where(qdoUserRole.userId.longValue().in(id));
        return jpaQuery.fetch();
    }

    /**
     * 获取功能 是否关联功能对象
     *
     * @return
     */
    public List<PrdSystemNewFunctionVO> queryFunctionByHaveFunctionObject() {
        JPAQuery<PrdSystemNewFunctionVO> jpaQuery = getJpaQueryNewFunctionSelectTwo();
        jpaQuery.where(qdoFunctionObject.permissionType.eq(PermissionTypeEnum.DATA_PERMISSION.getName()));
        return jpaQuery.fetch();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemNewFunctionVO> getJpaQueryNewFunctionSelectTwo() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemNewFunctionVO.class,
                qdoNewFunction.id,
                qdoNewFunction.functionName,
                qdoNewFunction.functionCode,
                qdoNewFunction.functionType,
                qdoNewFunction.createTime,
                qdoNewFunction.modifyTime,
                qdoNewFunction.objectId
        )).from(qdoFunctionObject).leftJoin(qdoNewFunction).on(qdoFunctionObject.functionId.longValue().eq(qdoNewFunction.id.longValue()));
    }

    /**
     * 查询所有业务对象
     * @return
     */
    public List<PrdSystemBusinessObjectVO> findAll() {
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.deleteFlag.eq(0));
        return jpaQuerySelect.fetch();
    }

    public List<PrdSystemBusinessObjectVO> queryByNotObjectCodes(List<String> objectCodes) {
        JPAQuery<PrdSystemBusinessObjectVO> jpaQuerySelect = getJpaQuerySelect();
        jpaQuerySelect.where(qdo.objectCode.notIn(objectCodes));
        return jpaQuerySelect.fetch();
    }


    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}
