package com.elitesland.tw.tw5.server.prd.system.service;

import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemRoleGroupPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemRoleGroupQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemRoleGroupService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRoleGroupRoleVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRoleGroupVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRoleVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemRoleConvert;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemRoleGroupConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRoleGroupDAO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemRoleDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemRoleGroupDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemRoleGroupRoleDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemRoleGroupRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemRoleRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 角色组
 *
 * @Author Bill
 * @Date 2023/9/19 13:21
 **/
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemRoleGroupServiceImpl implements PrdSystemRoleGroupService {

    private final PrdSystemRoleGroupRepo repo;
    private final PrdSystemRoleGroupDAO dao;
    private final PrdSystemRoleRepo roleRepo;

    /**
     * 角色组添加
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemRoleGroupVO insert(PrdSystemRoleGroupPayload payload) {
        //校验
        check(payload);
        PrdSystemRoleGroupDO entityDo = PrdSystemRoleGroupConvert.INSTANCE.toDo(payload);
        repo.save(entityDo);
        return PrdSystemRoleGroupConvert.INSTANCE.toVo(entityDo);
    }

    /**
     * 角色组修改
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemRoleGroupVO update(PrdSystemRoleGroupPayload payload) {
        //校验数据
        checkUpdate(payload);
        //查询数据是否存在
        PrdSystemRoleGroupDO entity = repo.findById(payload.getId()).orElseGet(PrdSystemRoleGroupDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PrdSystemRoleGroupDO entityDo = PrdSystemRoleGroupConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final PrdSystemRoleGroupDO save = repo.save(entity);
        return PrdSystemRoleGroupConvert.INSTANCE.toVo(save);
    }

    /**
     * 角色组逻辑删除
     * @param keys
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!CollectionUtils.isEmpty(keys)){
            repo.deleteSoft(keys);
        }
    }

    /**
     * 角色组主键查询
     * @param id
     * @return
     */
    @Override
    public PrdSystemRoleGroupVO get(Long id) {
        PrdSystemRoleGroupDO entity = repo.findById(id).orElseGet(PrdSystemRoleGroupDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PrdSystemRoleGroupVO prdSystemRoleGroupVO = PrdSystemRoleGroupConvert.INSTANCE.toVo(entity);
        //查询分组下的角色
        List<PrdSystemRoleGroupRoleDO> roleGroupRoleDOS = dao.queryByRole(id);
        List<PrdSystemRoleVO> roles = new ArrayList<>();
        for (PrdSystemRoleGroupRoleDO roleGroupRoleDO : roleGroupRoleDOS) {
            PrdSystemRoleDO role = roleRepo.findById(roleGroupRoleDO.getRoleId()).orElseGet(PrdSystemRoleDO::new);
            PrdSystemRoleVO roleVO = PrdSystemRoleConvert.INSTANCE.toVo(role);
            roles.add(roleVO);
        }
        prdSystemRoleGroupVO.setRoleList(roles);
        return prdSystemRoleGroupVO;
    }

    /**
     * 角色组模糊分页
     * @param query
     * @return
     */
    @Override
    public List<PrdSystemRoleGroupVO> listGroup(PrdSystemRoleGroupQuery query) {
        // 拿到分组下所有分组
        List<PrdSystemRoleGroupVO> roleGroupVOS = dao.queryByRoleGroupType(query.getRoleGroupType());

        // 拿到角色组和角色的关系
        List<PrdSystemRoleGroupRoleDO> roleGroupRoleDOS = dao.queryByRoles(roleGroupVOS.stream().map(PrdSystemRoleGroupVO::getId).collect(Collectors.toList()));

        // 拿到角色信息
        List<PrdSystemRoleDO> roleList = roleRepo.findAllById(roleGroupRoleDOS.stream().map(PrdSystemRoleGroupRoleDO::getRoleId).collect(Collectors.toSet()));

        // 将角色信息 转为 map集合
        Map<Long, PrdSystemRoleDO> collect = roleList.stream().collect(Collectors.toMap(PrdSystemRoleDO::getId, prdSystemRoleDO -> prdSystemRoleDO));

        //遍历角色组
        for (PrdSystemRoleGroupVO next : roleGroupVOS) {
            List<PrdSystemRoleVO> roles = new ArrayList<>();
            for (PrdSystemRoleGroupRoleDO roleGroupRoleDO : roleGroupRoleDOS) {
                // 如果发现匹配到当前角色组
                if (next.getId().equals(roleGroupRoleDO.getRoleGroupId())) {
                    // 拿到当前角色信息
                    PrdSystemRoleDO prdSystemRoleDO = collect.get(roleGroupRoleDO.getRoleId());
                    if (!ObjectUtils.isEmpty(prdSystemRoleDO)) {
                        roles.add(PrdSystemRoleConvert.INSTANCE.toVo(prdSystemRoleDO));
                    }
                }
            }
            next.setRoleList(roles);
        }
        return roleGroupVOS;
    }

    /**
     * 角色组列表查询全部
     * @Params roleGroupType 角色组类型
     * @return
     */
    @Override
    public List<PrdSystemRoleGroupVO> list(String roleGroupType) {
        return dao.queryByRoleGroupType(roleGroupType);
    }

    /**
     * 校验数据
     * @param payload
     */
    private void checkUpdate(PrdSystemRoleGroupPayload payload) {
        if (payload.getId() == null){
            throw TwException.error("", "id不能为空");
        }
    }

    /**
     * 校验插入数据
     * @param payload
     */
    private void check(PrdSystemRoleGroupPayload payload) {
        if (!StringUtils.hasText(payload.getRoleGroupName())){
            throw TwException.error("", "roleGroupName不能为空");
        }
        if (!StringUtils.hasText(payload.getRoleGroupType())){
            throw TwException.error("", "roleGroupType不能为空");
        }
    }

}
