package com.elitesland.tw.tw5.server.prd.task.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.personplan.service.PmsProjectRoleAssignmentService;
import com.elitesland.tw.tw5.api.prd.pms.service.PmsProjectWbsService;
import com.elitesland.tw.tw5.api.prd.task.payload.PmsWbsTaskPayload;
import com.elitesland.tw.tw5.api.prd.task.query.PmsWbsTaskQuery;
import com.elitesland.tw.tw5.api.prd.task.service.PmsWbsTaskService;
import com.elitesland.tw.tw5.api.prd.task.vo.PmsWbsTaskVO;
import com.elitesland.tw.tw5.server.common.GenerateSeqNumConstants;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.constants.ProjectWbsStatusEnum;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;

import com.elitesland.tw.tw5.server.prd.task.common.TaskStatusEnum;
import com.elitesland.tw.tw5.server.prd.task.convert.PmsWbsTaskConvert;
import com.elitesland.tw.tw5.server.prd.task.dao.PmsWbsTaskDAO;
import com.elitesland.tw.tw5.server.prd.task.entity.PmsWbsTaskDO;
import com.elitesland.tw.tw5.server.prd.task.repo.PmsWbsTaskRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 节点任务
 *
 * @author xxb
 * @date 2023-07-11
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PmsWbsTaskServiceImpl extends BaseServiceImpl implements PmsWbsTaskService {

    private final PmsWbsTaskRepo pmsWbsTaskRepo;
    private final PmsWbsTaskDAO pmsWbsTaskDAO;

   // private final PmsBudgetDetailService budgetDetailService;

    private final CacheUtil cacheUtil;

    private final FileUtil fileUtil;

    private final PmsProjectWbsService pmsProjectWbsService;

    private final PmsProjectRoleAssignmentService pmsProjectRoleAssignmentService;

    @Override
    public PagingVO<PmsWbsTaskVO> queryPaging(PmsWbsTaskQuery query) {
        PagingVO<PmsWbsTaskVO> pageVo = pmsWbsTaskDAO.queryPaging(query);
        if (!ObjectUtils.isEmpty(pageVo.getRecords())) {
            pageVo.getRecords().forEach(this::transferData);
        }
        return pageVo;
    }

    @Override
    public List<PmsWbsTaskVO> queryListDynamic(PmsWbsTaskQuery query) {
        List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryListDynamic(query);
        if (!ObjectUtils.isEmpty(list)) {
            list.forEach(this::transferData);
        }
        return list;
    }

    @Override
    public PmsWbsTaskVO queryByKey(Long key) {
//        PmsWbsTaskDO entity = pmsWbsTaskRepo.findById(key).orElseGet(PmsWbsTaskDO::new);
//        Assert.notNull(entity.getId(), "不存在");
//        PmsWbsTaskVO vo = PmsWbsTaskConvert.INSTANCE.toVo(entity);
        PmsWbsTaskVO vo = pmsWbsTaskDAO.queryByKey(key);
        if (!ObjectUtils.isEmpty(vo)) {
            transferData(vo);
        }
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTaskVO insert(PmsWbsTaskPayload payload) {
        //验证非空
        checkData(payload);

        // 设置权重
        payload.setManualSettingWeight(0);
        List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryByWbsId(payload.getWbsId());
        if (ObjectUtils.isEmpty(list)) {
            payload.setWeight(BigDecimal.valueOf(100));
        } else {
            // 算出已经手动设置过值得节点 总权重
            BigDecimal weights = list.stream().filter(v -> 1 == v.getManualSettingWeight()).map(PmsWbsTaskVO::getWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
            // 计算出新的平均值
            long count = list.stream().filter(v -> 0 == v.getManualSettingWeight()).count();
            BigDecimal averageNew = BigDecimal.valueOf(100).subtract(weights).divide(BigDecimal.valueOf(count + 1), 2, RoundingMode.HALF_UP);
            payload.setWeight(averageNew);

            // 修改数据库里 权重不一致的值
            list.stream().filter(v -> 0 == v.getManualSettingWeight()).forEach(v -> v.setWeight(averageNew));
            pmsWbsTaskDAO.saveAll(PmsWbsTaskConvert.INSTANCE.toEntity(list));
        }

        String code = generateSeqNum(GenerateSeqNumConstants.PMS_TASK);
        payload.setTaskCode(code);
        payload.setProgress(BigDecimal.ZERO);
        payload.setTaskStatus(TaskStatusEnum.CREATE.getCode());
        PmsWbsTaskDO entityDo = PmsWbsTaskConvert.INSTANCE.toDo(payload);
        return PmsWbsTaskConvert.INSTANCE.toVo(pmsWbsTaskRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PmsWbsTaskVO update(PmsWbsTaskPayload payload) {
        //验证非空
        checkData(payload);
        //更新 任务
        PmsWbsTaskDO entity = pmsWbsTaskRepo.findById(payload.getId()).orElseGet(PmsWbsTaskDO::new);
        Assert.notNull(entity.getId(), "不存在");
        if (TaskStatusEnum.PENDING.getCode().equals(entity.getTaskStatus())) {
            throw TwException.error("", "暂挂状态的任务，只能查看详情，不能编辑，请核验！");
        }
        if (TaskStatusEnum.FINISH.getCode().equals(entity.getTaskStatus())) {
            throw TwException.error("", "已完成状态的任务，只能查看详情，不能编辑，请核验！");
        }
        // 判断 任务进度 是否有更改
        Boolean changeProgressFlag = payload.getProgress().compareTo(entity.getProgress()) != 0;

        PmsWbsTaskDO entityDo = PmsWbsTaskConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        PmsWbsTaskVO vo = PmsWbsTaskConvert.INSTANCE.toVo(pmsWbsTaskRepo.save(entity));

        // 更改活动进度、状态
        updateWBS(payload.getWbsId(), changeProgressFlag);

        return vo;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (ObjectUtils.isEmpty(keys)) {
            throw TwException.error("", "任务主键不能为空，请核验！");
        }
        List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryByKeys(keys);
        for (PmsWbsTaskVO vo : list) {
            if (!TaskStatusEnum.CREATE.getCode().equals(vo.getTaskStatus())) {
                throw TwException.error("", "只有新建任务才能删除，请核验！");
            }
        }
        pmsWbsTaskDAO.deleteSoft(keys);

        // 权重重新更新
        list = pmsWbsTaskDAO.queryByWbsId(list.get(0).getWbsId());
        // 算出已经手动设置过值得节点 总权重
        BigDecimal weights = list.stream().filter(v -> 1 == v.getManualSettingWeight() && keys.contains(v.getId())).map(PmsWbsTaskVO::getWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算出新的平均值
        list = list.stream().filter(v -> 0 == v.getManualSettingWeight()).collect(Collectors.toList());
        long count = list.size();
        if (count > 0) {
            BigDecimal averageNew = BigDecimal.valueOf(100).subtract(weights).divide(BigDecimal.valueOf(count), 2, RoundingMode.HALF_UP);
            // 修改数据库里 权重不一致的值
            list.forEach(v -> v.setWeight(averageNew));
            pmsWbsTaskDAO.saveAll(PmsWbsTaskConvert.INSTANCE.toEntity(list));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoftByWbsIds(List<Long> wbsIds) {
        pmsWbsTaskDAO.deleteSoftByWbsIds(wbsIds);
    }

    @Override
    public List<PmsWbsTaskVO> findByWbsIds(List<Long> wbsIds) {
        return pmsWbsTaskDAO.findByWbsIds(wbsIds);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void appointedTask(List<Long> keys, Long wbsId) {
        if (ObjectUtils.isEmpty(keys)) {
            throw TwException.error("", "任务主键不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(wbsId)) {
            throw TwException.error("", "节点ID不能为空，请核验！");
        }
        // 任务派发
        List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryByKeys(keys);
        for (PmsWbsTaskVO vo : list) {
            if (!TaskStatusEnum.CREATE.getCode().equals(vo.getTaskStatus())) {
                throw TwException.error("", "只有新建任务才能指派，请核验！");
            }
        }
        pmsWbsTaskDAO.updateTaskStatus(keys, TaskStatusEnum.APPROVED.getCode(), GlobalUtil.getLoginUserId());
        // 扣除相同角色 的 预算金额
     //   budgetDetailService.updateUsedMoney(wbsId, list);
        // 添加相关方角色关系
        Map<Long, List<PmsWbsTaskVO>> groupedMap = list.stream()
                .collect(Collectors.groupingBy(
                        PmsWbsTaskVO::getReceiverUserId,
                        Collectors.toList()
                ));
        Long projectId = list.get(0).getProjectId();
        for (Map.Entry<Long, List<PmsWbsTaskVO>> entry : groupedMap.entrySet()) {
            List<Long> projectRoleIds = entry.getValue().stream().map(v -> v.getProjectRoleId()).collect(Collectors.toList());
            pmsProjectRoleAssignmentService.insert(projectId, entry.getKey(), projectRoleIds);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateTaskStatus(List<Long> keys, String taskStatus) {
        if (ObjectUtils.isEmpty(keys)) {
            throw TwException.error("", "任务主键不能为空，请核验！");
        }
        // 当状态为已完成时，进度要改成100 而且会触发 更改节点进度
        if (TaskStatusEnum.FINISH.getCode().equals(taskStatus)) {
            pmsWbsTaskDAO.updateTaskStatus(keys, taskStatus, BigDecimal.valueOf(100L));
            List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryByKeys(keys);
            List<Long> wbsIds = list.stream().distinct().map(v -> v.getWbsId()).collect(Collectors.toList());
            wbsIds.forEach(wbsId -> updateWBS(wbsId, true));
        } else {
            pmsWbsTaskDAO.updateTaskStatus(keys, taskStatus);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateTaskWeight(Long wbsId, Long key, BigDecimal weight) {
        List<PmsWbsTaskVO> list = pmsWbsTaskDAO.queryByWbsId(wbsId);
        // 算出已经手动设置过值得节点 总权重  排除当前节点
        BigDecimal weights = list.stream().filter(v -> !key.equals(v.getId()) && 1 == v.getManualSettingWeight()).map(PmsWbsTaskVO::getWeight).reduce(BigDecimal.ZERO, BigDecimal::add);
        // 验证权重总和不可超过100
        weights = weights.add(weight);
        if (weights.doubleValue() > 100) {
            throw TwException.error("", "该节点下的手动设置任务权重总和超过100了，请核验！");
        }
        // 计算出新的平均值
        list = list.stream().filter(v -> !key.equals(v.getId()) && 0 == v.getManualSettingWeight()).collect(Collectors.toList());
        long count = list.size();
        if (count > 0) {
            BigDecimal averageNew = BigDecimal.valueOf(100).subtract(weights).divide(BigDecimal.valueOf(count), 2, RoundingMode.HALF_UP);
            // 修改数据库里 权重不一致的值
            list.forEach(v -> v.setWeight(averageNew));
            pmsWbsTaskDAO.saveAll(PmsWbsTaskConvert.INSTANCE.toEntity(list));
        }

        //活动变更进度
        PmsWbsTaskVO vo = pmsWbsTaskDAO.queryByKey(key);
        ;
        // 任务变更权限
        pmsWbsTaskDAO.updateTaskWeight(key, weight);

        // 更改活动进度、状态
        Boolean changeProgressFlag = !areBigDecimalsEqual(vo.getWeight(), weight);
        updateWBS(wbsId, changeProgressFlag);
    }

    /**
     * 活动进度 计算：当节点有任务，任务有进度和权重，节点的进度为其下任务进度的加权平均
     * 活动状态 计算：当任务状态全部已完成 ，活动状态已完成 ；
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateWBS(Long wbsId, Boolean changeProgressFlag) {
        List<PmsWbsTaskVO> taskList = pmsWbsTaskDAO.queryByWbsId(wbsId);
        //如果任务进度有变化 更新活动进度
        if (changeProgressFlag) {
            //获取活动进度 ：进度为其下任务进度的加权平均
            BigDecimal taskProgress = BigDecimal.ZERO;

            // 如果节点下的任务都已经完成   活动的进度也就是100了
            long size = taskList.stream().filter(v -> v.getProgress().doubleValue() == 100).count();
            if (taskList.size() == size) {
                taskProgress = BigDecimal.valueOf(100);
            } else {
                for (PmsWbsTaskVO task : taskList) {
                    taskProgress = taskProgress.add(task.getProgress().multiply(task.getWeight()));
                }
                taskProgress = taskProgress.divide(BigDecimal.valueOf(100));
            }
            if (taskProgress.doubleValue() > 0 && taskProgress.doubleValue() < 100) {
                pmsProjectWbsService.updateWbsProgress(wbsId, taskProgress, ProjectWbsStatusEnum.NOGIONG.getCode());
            } else if (taskProgress.doubleValue() == 100) {
                pmsProjectWbsService.updateWbsProgress(wbsId, taskProgress, ProjectWbsStatusEnum.FINISHED.getCode());
            } else if (taskProgress.doubleValue() == 0) {
                pmsProjectWbsService.updateWbsProgress(wbsId, taskProgress, ProjectWbsStatusEnum.NOSTART.getCode());
            }
        }
    }

    /**
     * 翻译数据
     *
     * @param taskVO
     */
    void transferData(PmsWbsTaskVO taskVO) {
        taskVO.setSourceTypeName(cacheUtil.transferSystemSelection("PMS:WBS:RESOURCE:TYPE", taskVO.getSourceType()));
        taskVO.setTaskStatusName(cacheUtil.transferSystemSelection("PMS:TASK:SATUS", taskVO.getTaskStatus()));
        taskVO.setReceiverUserName(cacheUtil.getUserName(taskVO.getReceiverUserId()));
        taskVO.setTaskFilesDatas(fileUtil.getFileDatas(taskVO.getDeliveryDocuments()));
        taskVO.setManagerUserName(cacheUtil.getUserName(taskVO.getManagerUserId()));
    }

    void checkData(PmsWbsTaskPayload payload) {
        // 判断 非空
        if (ObjectUtils.isEmpty(payload.getSourceType())) {
            throw TwException.error("", "资源类型不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getProjectRoleId())) {
            throw TwException.error("", "角色不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getTaskName())) {
            throw TwException.error("", "任务名称不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getReceiverUserId())) {
            throw TwException.error("", "资源名称不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getStartDate())) {
            throw TwException.error("", "任务起止时间不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getEndDate())) {
            throw TwException.error("", "任务起止时间不能为空，请核验！");
        }
        if (ObjectUtils.isEmpty(payload.getDays())) {
            throw TwException.error("", "人天数不能为空，请核验！");
        }
        BigDecimal progress = payload.getProgress();
        if (!ObjectUtils.isEmpty(progress)) {
            if (new BigDecimal(progress.intValue()).compareTo(progress) != 0) {
                throw TwException.error("", "进度只能是整数，请核验！");
            }
            if (progress.intValue() < 0 || progress.intValue() > 100) {
                throw TwException.error("", "进度只能是0-100的整数，请核验！");
            }
        }
        BigDecimal weight = payload.getWeight();
        if (!ObjectUtils.isEmpty(weight)) {
            if (new BigDecimal(weight.intValue()).compareTo(weight) != 0) {
                throw TwException.error("", "权重只能是整数，请核验！");
            }
            if (weight.intValue() < 0 || weight.intValue() > 100) {
                throw TwException.error("", "权重只能是0-100的整数，请核验！");
            }
        }
    }

    /**
     * 比较两个 BigDecimal
     *
     * @param bd1
     * @param bd2
     * @return
     */
    public static boolean areBigDecimalsEqual(BigDecimal bd1, BigDecimal bd2) {
        // Check if both BigDecimal objects are null
        if (bd1 == null && bd2 == null) {
            return true;
        }

        // Check if only one of the BigDecimal objects is null
        if ((bd1 == null && bd2 != null) || (bd1 != null && bd2 == null)) {
            return false;
        }

        // Use the equals() method to compare their values
        return bd1.compareTo(bd2) == 0;
    }

}
