package com.elitesland.tw.tw5.base.common;

import lombok.Data;

import java.util.ArrayList;
import java.util.List;

/**
 * 统一返回类
 *
 * @author duwh
 * @date 2023/02/14
 */
@Data
public class TwOutputUtil<T> {

    /**
     * 是否成功
     */
    private boolean ok;

    /**
     * 业务数据
     */
    private T data;

    /**
     * 成功信息
     */
    private List<TwMessage> infos = new ArrayList<>();

    /**
     * 错误信息
     */
    private List<TwMessage> errors = new ArrayList<>();

    /**
     * 警告信息
     */
    private List<TwMessage> warns = new ArrayList<>();

    public TwOutputUtil() {
    }

    private TwOutputUtil(boolean ok) {
        this.ok = ok;
    }

    /**
     * 返回
     *
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> output(boolean ok) {
        return new TwOutputUtil<>(ok);
    }

    /**
     * 返回
     *
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> output(boolean ok,T data,List<TwMessage> infos, List<TwMessage> errors, List<TwMessage> warns) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(ok);
        outputUtil.data = data;
        outputUtil.infos = infos;
        outputUtil.errors = errors;
        outputUtil.warns = warns;
        return outputUtil;
    }

    /**
     * 最简单形式成功返回
     *
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> ok() {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(true);
        outputUtil.infos.add(TwMessage.of("COM-SUCCESS", "操作成功", null));
        return outputUtil;
    }

    /**
     * 最简单形式成功返回
     *
     * @param data 返回数据
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> ok(T data) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(true);
        outputUtil.setData(data);
        outputUtil.infos.add(TwMessage.of("COM-SUCCESS", "操作成功", null));
        return outputUtil;
    }

    /**
     * 返回成功
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> ok(String localeCode, String defaultMsg, T data) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(true);
        outputUtil.data = data;
        outputUtil.infos.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }

    /**
     * 最简单形式失败返回
     *
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> error() {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(false);
        outputUtil.errors.add(TwMessage.of("COM-ERROR", "操作失败", null));
        return outputUtil;
    }

    /**
     * 返回失败
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> error(String localeCode, String defaultMsg) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(false);
        outputUtil.data = null;
        outputUtil.errors.add(TwMessage.of(localeCode, defaultMsg, null));
        return outputUtil;
    }

    /**
     * 返回失败
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> error(String localeCode, String defaultMsg, T data) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(false);
        outputUtil.data = data;
        outputUtil.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }


    /**
     * 添加失败
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public TwOutputUtil<T> addError(String localeCode, String defaultMsg, Object data) {
        this.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }

    /**
     * 最简单形式警告返回
     *
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> warn() {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(false);
        outputUtil.warns.add(TwMessage.of("COM-WARN", "是否确认？", null));
        return outputUtil;
    }

    /**
     * 返回警告
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static <T> TwOutputUtil<T> warn(String localeCode, String defaultMsg, T data) {
        TwOutputUtil<T> outputUtil = new TwOutputUtil<>(false);
        outputUtil.data = data;
        outputUtil.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }


    /**
     * 添加警告
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public TwOutputUtil<T> addWarn(String localeCode, String defaultMsg, T data) {
        this.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }

}
