package com.elitesland.tw.tw5.base.common.aspect;

import com.elitescloud.boot.redis.util.RedLockUtils;
import com.elitesland.tw.tw5.base.common.TwOutputUtil;
import com.elitesland.tw.tw5.base.common.annotation.DuplicateSubmissionPrevention;
import com.elitesland.tw.tw5.base.util.GlobalUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.redisson.RedissonRedLock;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.concurrent.TimeUnit;

@Slf4j
@Aspect
@Component
@Order(1)
@RequiredArgsConstructor
public class DuplicateSubmissionPreventionAspect {

    // redis中key的分隔符
    private static final String REDIS_SEPARATOR =":";

    // 默认提示语
    private static final String DEFAULT_TIP_MSG ="请勿重复操作";

    // 未获取到当前登录用户
    private static final String UN_AUTHORIZATION ="UN_AUTHORIZATION";

    @Value("${spring.application.name:MyDuplicateSubmissionPrevention}")
    private String applicationName;

    private final RedLockUtils redLockUtils;



    @Around(value = "@annotation(com.elitesland.tw.tw5.base.common.annotation.DuplicateSubmissionPrevention)")
    public Object doAround(ProceedingJoinPoint joinPoint){

        MethodSignature signature = (MethodSignature)joinPoint.getSignature();
        Method method = signature.getMethod();
        DuplicateSubmissionPrevention annotation = method.getAnnotation(DuplicateSubmissionPrevention.class);
        // 判断是否开启防重复
        boolean enabled = annotation.enabled();
        if (enabled){

            // 构建每个方法的唯一key值
            String redisLockKey = getRedisLockKey(annotation, joinPoint);

            // 指定时间内 redis同步锁 获取
            RedissonRedLock redisLock = redLockUtils.getRedLock("redisLockKey");
            boolean getLock = false;
            try {
                boolean tryLock = redisLock.tryLock(0L, annotation.timeIntervalMillis(), TimeUnit.MILLISECONDS);
                if (tryLock){
                    log.info("redisson获取分步式锁{},当前线程名称{}",redisLockKey,Thread.currentThread().getName());
                    getLock = true;
                }else {
                    String tipMsg = StringUtils.hasText(annotation.message()) ? annotation.message() : DEFAULT_TIP_MSG;
                    log.info("redisson未获取分步式锁{},当前线程名称{}",redisLockKey,Thread.currentThread().getName());
                    return TwOutputUtil.error("",tipMsg);
                }
            } catch (InterruptedException e) {
                log.info("redisson未获取分步式锁异常");
            }finally {
                if (getLock){
                    redisLock.unlock();
                    log.info("redisson分步式锁释放");
                }
            }

            try {
                return joinPoint.proceed();
            } catch (Throwable e) {
                log.error("系统发生异常");
                e.printStackTrace();
                return TwOutputUtil.error("","系统发生异常");
            }

        }else {
            try {
                return joinPoint.proceed();
            } catch (Throwable e) {
                log.error("系统发生异常");
                e.printStackTrace();
                return TwOutputUtil.error("","系统发生异常");
            }
        }


    }

    private String getRedisLockKey(DuplicateSubmissionPrevention annotation,ProceedingJoinPoint joinPoint ){
        String[] parametersArr = annotation.parametersForUniqueFlagGeneration();
        // 构建方法的唯一key 项目名 + 接口名 + 方法名 + 自定义参数
        String className = joinPoint.getSignature().getDeclaringType().getSimpleName();
        String methodName = joinPoint.getSignature().getName();
        // 匹配自定义的参数

        // 获取方法的入参对象
        Object[] args = joinPoint.getArgs();
        Object parameterEntity = args[0];

        StringBuilder annotateParameterSb = new StringBuilder();
        //把当前登录人拼接到自定义参数里面
        try {
            Long loginUserId = GlobalUtil.getLoginUserId();
            annotateParameterSb.append(loginUserId);
            annotateParameterSb.append("_");
        }catch (Exception e){
            annotateParameterSb.append(UN_AUTHORIZATION);
            annotateParameterSb.append("_");
        }


        for (String parameter:parametersArr){
            try {
                // 获取属性的值
                Field nameField = parameterEntity.getClass().getDeclaredField(parameter);
                // 设置可访问性
                nameField.setAccessible(true);
                String name = nameField.get(parameterEntity) + "";
                annotateParameterSb.append(name);
                annotateParameterSb.append("_");
            } catch (Exception e) {
                log.info("{}未能成功获取",parameter);
            }

        }

        return new StringBuilder()
                .append(applicationName)
                .append(REDIS_SEPARATOR)
                .append(className)
                .append(REDIS_SEPARATOR)
                .append(methodName)
                .append(REDIS_SEPARATOR)
                .append(annotateParameterSb)
                .toString();
    }
}
