package com.elitesland.yst.common.config;

import cn.hutool.core.util.StrUtil;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.config.TransportMode;
import org.redisson.spring.data.connection.RedissonConnectionFactory;
import org.redisson.spring.starter.RedissonAutoConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

@Configuration
@AutoConfigureBefore({RedisAutoConfiguration.class, RedissonAutoConfiguration.class})
@ConditionalOnClass({Redisson.class, RedisOperations.class})
public class RedissonConfig {
    private static final Logger log = LoggerFactory.getLogger(RedissonConfig.class);

    @Value("${redis.redisson-cluster-config.nodes:#{null}}")
    private String nodes;
    @Value("${redis.redisson-cluster-config.password:#{null}}")
    private String password;

    @Value("${redis.redlock-instance-config.database:#{0}}")
    private int database;

    @Value("${redis.redlock-instance-config.node0.host:#{null}}")
    private String node0Host;
    @Value("${redis.redlock-instance-config.node0.password:#{null}}")
    private String node0Password;

    @Value("${redis.redlock-instance-config.node1.host:#{null}}")
    private String node1Host;
    @Value("${redis.redlock-instance-config.node1.password:#{null}}")
    private String node1Password;

    @Value("${redis.redlock-instance-config.node2.host:#{null}}")
    private String node2Host;
    @Value("${redis.redlock-instance-config.node2.password:#{null}}")
    private String node2Password;

    @Value("${redis.prefix:#{''}}")
    private String prefix;

    @Primary
    @Bean("cluster_redis_client")
    @ConditionalOnProperty(prefix = "redis.redisson-cluster-config", name = "enabled", havingValue = "true")
    public RedissonClient initRedissonClient() {
        if (StrUtil.isBlank(nodes)) {
            log.warn("加载redission配置失败");
            return Redisson.create();
        }
        var config = new Config();
        config.setTransportMode(TransportMode.NIO);
        config.useClusterServers().addNodeAddress(Arrays.stream(nodes.split(",")).map(String::strip).toArray(String[]::new))
                .setPassword(password)
                .setScanInterval(3000)
                .setSubscriptionConnectionPoolSize(25)
                .setSlaveConnectionMinimumIdleSize(16)
                .setSlaveConnectionPoolSize(32)
                .setMasterConnectionMinimumIdleSize(16)
                .setMasterConnectionPoolSize(32)
                .setConnectTimeout(20000)
                .setTimeout(5000)
                .setRetryInterval(3000);
        return Redisson.create(config);
    }


    @Bean("redis_single_client0")
    @ConditionalOnProperty(prefix = "redis.redlock-instance-config", name = "enabled", havingValue = "true")
    public RedissonClient initRedissonClientSingle0() {
        if (StrUtil.isBlank(node0Host)) {
            log.warn("加载redission配置node0失败");
            return Redisson.create();
        }

        var config = new Config();
        config.useSingleServer()
                .setAddress(node0Host)
                .setPassword(node0Password)
                .setSubscriptionConnectionPoolSize(25)
                .setConnectionMinimumIdleSize(16)
                .setConnectionPoolSize(32)
                .setConnectTimeout(20000)
                .setTimeout(5000)
                .setRetryInterval(3000)
                .setDatabase(database);
        return Redisson.create(config);
    }

    @Bean("redis_single_client1")
    @ConditionalOnProperty(prefix = "redis.redlock-instance-config", name = "enabled", havingValue = "true")
    public RedissonClient initRedissonClientSingle1() {
        if (StrUtil.isBlank(node1Host)) {
            log.warn("加载redission配置node1失败");
            return Redisson.create();
        }

        Config config = new Config();
        config.useSingleServer()
                .setAddress(node1Host)
                .setPassword(node1Password)
                .setSubscriptionConnectionPoolSize(25)
                .setConnectionMinimumIdleSize(16)
                .setConnectionPoolSize(32)
                .setConnectTimeout(20000)
                .setTimeout(5000)
                .setRetryInterval(3000)
                .setDatabase(database);
        return Redisson.create(config);
    }

    @Bean("redis_single_client2")
    @ConditionalOnProperty(prefix = "redis.redlock-instance-config", name = "enabled", havingValue = "true")
    public RedissonClient initRedissonClientSingle2() {
        if (StrUtil.isBlank(node1Host)) {
            log.warn("加载redission配置node2失败");
            return Redisson.create();
        }
        var config = new Config();
        config.useSingleServer()
                .setAddress(node2Host)
                .setPassword(node2Password)
                .setSubscriptionConnectionPoolSize(25)
                .setConnectionMinimumIdleSize(16)
                .setConnectionPoolSize(32)
                .setConnectTimeout(20000)
                .setTimeout(5000)
                .setRetryInterval(3000)
                .setDatabase(database);
        return Redisson.create(config);
    }

    @Bean
    @ConditionalOnProperty(prefix = "redis.redisson-cluster-config", name = "enabled", havingValue = "true")
    public RedisConnectionFactory initRedissonConnectionFactory(@Qualifier("cluster_redis_client") RedissonClient redissonClient) {
        return new RedissonConnectionFactory(redissonClient);
    }

    @Bean
    public RedisTemplate redisTemplate(RedisConnectionFactory redisConnectionFactory) {
        RedisTemplate<String, Object> template = new RedisTemplate<>();

        RedisSerializer<String> redisSerializer = redisSerializer();

        template.setConnectionFactory(redisConnectionFactory);
        template.setKeySerializer(redisSerializer);
        template.setHashKeySerializer(redisSerializer);
        template.setValueSerializer(new JdkSerializationRedisSerializer());
        template.setHashValueSerializer(new JdkSerializationRedisSerializer());
        return template;
    }

    private RedisSerializer<String> redisSerializer() {
        return new RedisSerializer<>() {
            private final Charset charset = StandardCharsets.UTF_8;

            @Override
            public String deserialize(byte[] bytes) {
                if (bytes == null) return null;
                String saveKey = new String(bytes, charset);
                if (saveKey.startsWith(prefix)) {
                    saveKey = saveKey.substring(prefix.length());
                }
                return saveKey;
            }

            @Override
            public byte[] serialize(String string) {
                String key = prefix + string;
                return (string == null ? null : key.getBytes(charset));
            }
        };
    }
}