package com.elitesland.yst.common.exception.handler;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.common.exception.BadCaptchaException;
import com.elitesland.yst.common.exception.BadJwtTokenException;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.common.util.ThrowableUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.remoting.TimeoutException;
import org.apache.dubbo.rpc.RpcException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.ResourceAccessException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import javax.ws.rs.BadRequestException;

/**
 * <pre>
 * [功能说明]
 * </pre>
 *
 * @author Mir
 * @date 2020/6/22
 */
@Slf4j
@ControllerAdvice
public class GlobalExceptionHandler {

    @ExceptionHandler(BadRequestException.class)
    @ResponseBody
    public ApiResult<Object> handleBadRequestException(BadRequestException e) {
        log.error("请求错误", e);
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpMessageNotReadableException(HttpMessageNotReadableException e) {
        log.warn("请求数据格式有误", e);
        return ApiResult.fail(ApiCode.PARAMETER_PARSE_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpRequestMethodNotSupportedException(HttpRequestMethodNotSupportedException e) {
        log.warn("请求方式有误", e);
        return ApiResult.fail(ApiCode.METHOD_NOT_SUPPORT, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public ApiResult<Object> handleBusinessException(BusinessException e) {
        log.error("业务异常", e);
        return ApiResult.fail(e.getCode() == null ? ApiCode.BUSINESS_EXCEPTION : e.getCode(), ThrowableUtil.getStackTrace(e), e.getMessage());
    }

    @ExceptionHandler(BadCaptchaException.class)
    @ResponseBody
    public ApiResult<Object> handleBadCaptchaException(BadCaptchaException e) {
        log.error("验证码验证失败", e);
        return ApiResult.fail(ApiCode.VERIFICATION_CODE_EXCEPTION, e.getMessage());
    }

    @ExceptionHandler(BadJwtTokenException.class)
    @ResponseBody
    public ApiResult<Object> handleAuthenticationException(BadJwtTokenException e) {
        log.error("权限认证失败", e);
        return ApiResult.fail(ApiCode.UNAUTHENTICATED_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(ResourceAccessException.class)
    @ResponseBody
    public ApiResult<Object> handleAuthenticationException(ResourceAccessException e) {
        log.error("请求资源异常", e);
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpClientErrorException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpClientErrorException(HttpClientErrorException e) {
        log.error("认证授权异常：", e);
        return ApiResult.fail(ApiCode.AUTHENTICATION_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler({Exception.class})
    @ResponseBody
    public ApiResult<Object> handleException(Exception e) {
        log.error("后端处理异常：", e);
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, ThrowableUtil.getStackTrace(e), null);
    }

    /**
     * 参数校验异常
     *
     * @param exception 异常
     * @return 结果
     */
    @ExceptionHandler({IllegalArgumentException.class, MethodArgumentNotValidException.class,
            ConstraintViolationException.class, ValidationException.class})
    @ResponseBody
    public ApiResult<Object> handleArgumentInvalidException(Exception exception) {
        log.info("参数校验不通过：{}", exception.getMessage());

        if (exception instanceof IllegalArgumentException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, exception.getMessage());
        }

        if (exception instanceof MethodArgumentNotValidException) {
            String msg = ((MethodArgumentNotValidException) exception).getBindingResult().getAllErrors().get(0).getDefaultMessage();
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, msg);
        }

        if (exception instanceof ConstraintViolationException) {
            String msg = ((ConstraintViolationException) exception).getConstraintViolations().toArray(ConstraintViolation[]::new)[0].getMessage();
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, msg);
        }

        if (exception instanceof ValidationException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, exception.getMessage());
        }

        return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, ExceptionUtil.getRootCause(exception).getMessage());
    }

    private static final String EXP_NO_PROVIDER_AVAILABLE = "No provider available";
    private static final String EXP_NO_PROVIDER_AVAILABLE1 = "because channel is closed";
    private static final String EXP_THREAD_POOL_OVERFLOW = "thread pool is exhausted";

    @ExceptionHandler(RpcException.class)
    @ResponseBody
    public ApiResult<Object> handleRpcException(RpcException exception) {
        log.error("调用远程接口失败", exception);
        var rootCause = ExceptionUtil.getRootCause(exception);
        if (rootCause instanceof TimeoutException) {
            return ApiResult.fail(ApiCode.SERVER_RESPONSE_TIMEOUT, ThrowableUtil.getStackTrace(exception), "请求超时");
        }

        var rootCauseMsg = rootCause.getMessage();
        if (StrUtil.contains(rootCauseMsg, EXP_NO_PROVIDER_AVAILABLE) || StrUtil.contains(rootCauseMsg, EXP_NO_PROVIDER_AVAILABLE1)) {
            return ApiResult.fail(ApiCode.CLOUD_SERVICE_UNAVAILABLE, ThrowableUtil.getStackTrace(exception), "系统维护中，请稍后再试");
        }
        if (StrUtil.contains(rootCauseMsg, EXP_THREAD_POOL_OVERFLOW)) {
            return ApiResult.fail(ApiCode.THREAD_OVERFLOW, ThrowableUtil.getStackTrace(exception), "当前访问用户过多，请稍后再试");
        }
        return ApiResult.fail(ApiCode.CLOUD_SERVICE_EXCEPTION, ThrowableUtil.getStackTrace(exception), null);
    }

    @ExceptionHandler(DataAccessResourceFailureException.class)
    @ResponseBody
    public ApiResult<Object> handleDataAccessException(DataAccessResourceFailureException exception) {
        return ApiResult.fail(ApiCode.DAO_EXCEPTION, ThrowableUtil.getStackTrace(exception), "系统繁忙，请稍后再试");
    }
}
