package com.elitesland.yst.common.util;

import cn.hutool.core.net.NetUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.RuntimeUtil;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.common.micro.Snowflake;
import com.elitesland.yst.common.property.IdProperties;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SharedSessionContractImplementor;
import org.hibernate.id.IdentifierGenerator;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.time.Duration;
import java.util.List;
import java.util.Objects;

/**
 * 适用于JPA的基于Snowflake算法的ID生成器
 *
 * @author Moz
 * @date 3/17/2020
 */
@Slf4j
public class IdGenerator implements IdentifierGenerator {

    private static volatile Snowflake snowflake = null;
    private static IdProperties idProperties;
    private static RedisTemplate redisTemplate;
    private static final Object LOCK = new Object();
    private static String CACHE_KEY = "cloudt_snowflake";
    private static String dataCenterName = "unknown";
    private static Long dataCenterId;
    private static Long workerId;

    @Override
    public Serializable generate(SharedSessionContractImplementor sharedSessionContractImplementor, Object o) throws HibernateException {
        return getSnowflake().nextId();
    }

    public static void init(IdProperties idProperties, RedisTemplate redisTemplate, String defaultDataCenterName) {
        IdGenerator.idProperties = Objects.requireNonNull(idProperties);
        IdGenerator.redisTemplate = Objects.requireNonNull(redisTemplate);
        IdGenerator.dataCenterName = Objects.requireNonNull(defaultDataCenterName);
    }

    public static Snowflake getSnowflake() {
        if (snowflake == null) {
            synchronized (LOCK) {
                if (snowflake == null) {
                    snowflake = generateSnowflake();
                }
            }
        }
        return snowflake;
    }

    public static void destroy() {
        if (snowflake == null || redisTemplate == null) {
            return;
        }
        if (!Boolean.TRUE.equals(idProperties.getSnowflake().getGenAuto())) {
            return;
        }

        log.info("销毁Snowflake实例");
        try {
            var opsSet = redisTemplate.opsForSet();

            // 删除当前的workerId
            redisTemplate.delete(cacheKeyWorkerId(workerId));
            opsSet.remove(cacheKeyDataCenterWorker(), workerId.toString());

            // 如果当前dataCenter下已没有worker，则删除dataCenter
            Long remain = opsSet.size(cacheKeyDataCenterWorker());
            if (remain == null || remain == 0L) {
                log.info("{}下的worker实例均已不在", dataCenterName);
                redisTemplate.delete(List.of(cacheKeyDataCenterID(dataCenterName), cacheKeyDataCenterID(dataCenterId)));
            }
        } catch (Exception e) {
            log.error("销毁IdGenerator异常：", e);
        }
    }

    public static void refreshAlive() {
        if (snowflake == null || redisTemplate == null) {
            return;
        }

        if (dataCenterId == null || workerId == null) {
            return;
        }

        var opsValue = redisTemplate.opsForValue();
        var opsSet = redisTemplate.opsForSet();

        Duration alive = idProperties.getSnowflake().getAlive();
        opsValue.set(cacheKeyDataCenterID(dataCenterName), dataCenterId.toString(), alive);
        opsValue.set(cacheKeyDataCenterID(dataCenterId), dataCenterName, alive);
        opsValue.set(cacheKeyWorkerId(workerId), getIp(), alive);
        opsSet.add(cacheKeyDataCenterWorker(), workerId.toString());

    }

    private static Snowflake generateSnowflake() {
        Objects.requireNonNull(idProperties, "Snowflake尚未初始化");
        IdProperties.Snowflake snowflakeProperties = idProperties.getSnowflake();

        if (!Boolean.TRUE.equals(idProperties.getSnowflake().getGenAuto())) {
            // 不是自动生成
            return generateByManual(snowflakeProperties);
        }

        Objects.requireNonNull(redisTemplate);
        if (StringUtils.hasText(snowflakeProperties.getCachePrefix())) {
            CACHE_KEY = snowflakeProperties.getCachePrefix();
        }
        Assert.isTrue(snowflakeProperties.getAlive() != null && snowflakeProperties.getAlive().toSeconds() > 0,
                "alive设置有误");
        Assert.isTrue(snowflakeProperties.getHeartBeat() != null
                && snowflakeProperties.getHeartBeat().toSeconds() < snowflakeProperties.getAlive().toSeconds(), "heartBeat设置有误");

        // 先生成dataCenterId
        generateDataCenterId();

        // 再生成workerId
        generateWorkerId();

        RuntimeUtil.addShutdownHook(IdGenerator::destroy);

        log.info("ID生成器初始化完成：{}【{}-{}】", dataCenterName, dataCenterId, workerId);

        return buildSnowflake();
    }

    private static Snowflake generateByManual(IdProperties.Snowflake snowflakeProperties) {
        dataCenterId = snowflakeProperties.getDataCenterId();
        workerId = snowflakeProperties.getWorkerId();

        Assert.isTrue(dataCenterId != null && dataCenterId > 0, "dataCenterId设置有误");
        Assert.isTrue(workerId != null && workerId > 0, "dataCenterId设置有误");
        return buildSnowflake();
    }

    private static Snowflake buildSnowflake() {
        return new Snowflake(dataCenterId, workerId, true, 5L, true);
    }

    private static void generateDataCenterId() {
        // 先查询是否已存在对应应用的dataCenterId
        dataCenterId = existsDataCenterId(dataCenterName);
        if (dataCenterId != null) {
            // 已存在，则直接复用
            return;
        }

        // 不存在则生成
        long max = Snowflake.getMaxDataCenterId();
        long temp = 0;
        var opsValue = redisTemplate.opsForValue();
        while (temp < max) {
            Boolean result = opsValue.setIfAbsent(cacheKeyDataCenterID(temp), dataCenterName,
                    idProperties.getSnowflake().getAlive());
            if (Boolean.TRUE.equals(result)) {
                // temp可用，则再次检查是否可用
                checkDataCenterId(temp);
                break;
            }

            temp++;
        }
        if (temp >= max) {
            throw new BusinessException("dataCenterId超过限制" + max);
        }
    }

    private static void checkDataCenterId(Long tempDataCenterId) {
        // 如果已有其它实例设置，则取其它实例设置的
        Boolean result = redisTemplate.opsForValue().setIfAbsent(cacheKeyDataCenterID(dataCenterName), tempDataCenterId.toString(),
                idProperties.getSnowflake().getAlive());
        if (Boolean.TRUE.equals(result)) {
            dataCenterId = tempDataCenterId;
            return;
        }
        dataCenterId = existsDataCenterId(dataCenterName);

        // 其它应用已生成，则删除当前生成的
        redisTemplate.delete(cacheKeyDataCenterID(tempDataCenterId));
    }

    private static void generateWorkerId() {
        long max = Snowflake.getMaxWorkerId();
        long temp = 0;
        var opsValue = redisTemplate.opsForValue();
        var ip = getIp();
        while (temp < max) {
            Boolean result = opsValue.setIfAbsent(cacheKeyWorkerId(temp), ip, idProperties.getSnowflake().getAlive());
            if (Boolean.TRUE.equals(result)) {
                // workerId可用
                workerId = temp;
                // 添加dataCenter下的worker
                redisTemplate.opsForSet().add(cacheKeyDataCenterWorker(), workerId.toString());
                return;
            }

            temp++;
        }
        throw new BusinessException("workerId超过限制" + max);
    }

    private static Long existsDataCenterId(String dataCenterName) {
        Object val = redisTemplate.opsForValue().get(cacheKeyDataCenterID(dataCenterName));
        if (val == null || val instanceof Long) {
            return (Long) val;
        }

        return Long.parseLong(val.toString());
    }

    private static String cacheKeyDataCenterID(String dataCenterName) {
        return CACHE_KEY + ":" + dataCenterName;
    }

    private static String cacheKeyDataCenterID(Long theDataCenterId) {
        return CACHE_KEY + ":" + "dataCenter" + "_" + theDataCenterId;
    }

    private static String cacheKeyWorkerId(Long theWorkerId) {
        return CACHE_KEY + ":" + "worker" + "_" + dataCenterId + "_" + theWorkerId;
    }

    private static String cacheKeyDataCenterWorker() {
        return CACHE_KEY + ":" + dataCenterId;
    }

    private static String getIp() {
        return ObjectUtil.defaultIfNull(NetUtil.getLocalhost().getHostAddress(), "127.0.0.1");
    }
}
